# SPDX-License-Identifier: GPL-3.0-or-later
# Mask Filter Plugin — deletes features outside a selected polygon (mask polygon)
# Copyright (C) 2026  Majid Hamed Hobi
# https://github.com/majid1973fg/mask-filter-plugin
# mask_filter_plugin.py
from qgis.PyQt.QtWidgets import QAction, QInputDialog, QMessageBox
from qgis.core import QgsProject, QgsMapLayer


class MaskFilterPlugin:
    """
    Small plugin that provides one action:
    Delete features outside a polygon mask layer (specified by name)
    """

    def __init__(self, iface):
        self.iface = iface
        self.action_mask = None
        self.menu = "&Mask Filter"

    def initGui(self):
        # Action: Delete outside mask
        self.action_mask = QAction("Delete Outside Mask", self.iface.mainWindow())
        self.action_mask.triggered.connect(self.delete_outside_mask)
        self.iface.addToolBarIcon(self.action_mask)
        self.iface.addPluginToMenu(self.menu, self.action_mask)

    def unload(self):
        if self.action_mask:
            self.iface.removePluginMenu(self.menu, self.action_mask)
            self.iface.removeToolBarIcon(self.action_mask)

    def delete_outside_mask(self):
        # Ask user for mask layer name
        mask_name, ok = QInputDialog.getText(self.iface.mainWindow(), "Mask layer", "Enter mask layer name:")
        if not ok or not mask_name:
            return

        mask_layers = QgsProject.instance().mapLayersByName(mask_name)
        if not mask_layers:
            QMessageBox.critical(self.iface.mainWindow(), "Error", f"Mask layer '{mask_name}' not found.")
            return
        mask_layer = mask_layers[0]

        # Build union geometry of mask layer
        mask_geom = None
        for f in mask_layer.getFeatures():
            if mask_geom is None:
                mask_geom = f.geometry()
            else:
                mask_geom = mask_geom.combine(f.geometry())

        if mask_geom is None:
            QMessageBox.warning(self.iface.mainWindow(), "Warning", "Mask layer has no geometry.")
            return

        # Iterate through all vector layers (except the mask) and delete features outside mask
        total_deleted = 0
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.VectorLayer and layer != mask_layer:
                try:
                    layer.startEditing()
                except Exception:
                    continue

                ids_to_delete = [f.id() for f in layer.getFeatures() if not f.geometry().intersects(mask_geom)]
                if ids_to_delete:
                    layer.deleteFeatures(ids_to_delete)
                    total_deleted += len(ids_to_delete)

                try:
                    layer.commitChanges()
                except Exception:
                    try:
                        layer.rollBack()
                    except Exception:
                        pass

        QMessageBox.information(
            self.iface.mainWindow(),
            "Done",
            f"Deleted {total_deleted} features outside the mask."
        )