# SPDX-License-Identifier: GPL-3.0-or-later
# Mask Filter Plugin — deletes features outside a selected polygon (mask polygon)
# and removes features that do not contain 'MR' in the 'TYP' field.
# Copyright (C) 2026  Majid Hamed Hobi
# https://github.com/majid1973fg/mask-filter-plugin
# mask_filter_plugin.py
from qgis.PyQt.QtWidgets import QAction, QInputDialog, QMessageBox
from qgis.core import QgsProject, QgsMapLayer


class MaskFilterPlugin:
    """
    Small plugin that provides two actions:
    1) Delete features outside a polygon mask layer (specified by name)
    2) Delete features in the active layer that do NOT contain 'MR' in field 'TYP',
       and also remove duplicate geometries in the same layer.
    """

    def __init__(self, iface):
        self.iface = iface
        self.action_mask = None
        self.action_mr = None
        self.menu = "&Mask Filter"

    def initGui(self):
        # Action 1: Delete outside mask
        self.action_mask = QAction("Delete Outside Mask", self.iface.mainWindow())
        self.action_mask.triggered.connect(self.delete_outside_mask)
        self.iface.addToolBarIcon(self.action_mask)
        self.iface.addPluginToMenu(self.menu, self.action_mask)

        # Action 2: Delete features not containing 'MR' in TYP + remove duplicates
        self.action_mr = QAction("Delete Not Containing MR / Duplicates", self.iface.mainWindow())
        self.action_mr.triggered.connect(self.delete_not_containing_mr_and_duplicates)
        self.iface.addToolBarIcon(self.action_mr)
        self.iface.addPluginToMenu(self.menu, self.action_mr)

    def unload(self):
        if self.action_mask:
            self.iface.removePluginMenu(self.menu, self.action_mask)
            self.iface.removeToolBarIcon(self.action_mask)
        if self.action_mr:
            self.iface.removePluginMenu(self.menu, self.action_mr)
            self.iface.removeToolBarIcon(self.action_mr)

    def delete_outside_mask(self):
        # Ask user for mask layer name
        mask_name, ok = QInputDialog.getText(self.iface.mainWindow(), "Mask layer", "Enter mask layer name:")
        if not ok or not mask_name:
            return

        mask_layers = QgsProject.instance().mapLayersByName(mask_name)
        if not mask_layers:
            QMessageBox.critical(self.iface.mainWindow(), "Error", f"Mask layer '{mask_name}' not found.")
            return
        mask_layer = mask_layers[0]

        # Build union geometry of mask layer
        mask_geom = None
        for f in mask_layer.getFeatures():
            if mask_geom is None:
                mask_geom = f.geometry()
            else:
                mask_geom = mask_geom.combine(f.geometry())

        if mask_geom is None:
            QMessageBox.warning(self.iface.mainWindow(), "Warning", "Mask layer has no geometry.")
            return

        # Iterate through all vector layers (except the mask) and delete features outside mask
        total_deleted = 0
        for layer in QgsProject.instance().mapLayers().values():
            if layer.type() == QgsMapLayer.VectorLayer and layer != mask_layer:
                try:
                    layer.startEditing()
                except Exception:
                    continue

                ids_to_delete = [f.id() for f in layer.getFeatures() if not f.geometry().intersects(mask_geom)]
                if ids_to_delete:
                    layer.deleteFeatures(ids_to_delete)
                    total_deleted += len(ids_to_delete)

                try:
                    layer.commitChanges()
                except Exception:
                    try:
                        layer.rollBack()
                    except Exception:
                        pass

        QMessageBox.information(
            self.iface.mainWindow(),
            "Done",
            f"Deleted {total_deleted} features outside the mask."
        )

    def delete_not_containing_mr_and_duplicates(self):
        # Operates on the active layer in QGIS
        layer = self.iface.activeLayer()
        if not layer:
            QMessageBox.warning(self.iface.mainWindow(), "Warning", "No active layer selected.")
            return

        if layer.type() != QgsMapLayer.VectorLayer:
            QMessageBox.critical(self.iface.mainWindow(), "Error", "Active layer is not a vector layer.")
            return

        field_names = [f.name() for f in layer.fields()]
        if "TYP" not in field_names:
            QMessageBox.critical(self.iface.mainWindow(), "Error", "Field 'TYP' not found in selected layer.")
            return

        try:
            layer.startEditing()
        except Exception:
            QMessageBox.critical(self.iface.mainWindow(), "Error", "Could not start editing the active layer.")
            return

        # 1) Delete features not containing 'MR'
        ids_to_delete = [f.id() for f in layer.getFeatures() if "MR" not in str(f["TYP"])]
        deleted_count = 0
        if ids_to_delete:
            layer.deleteFeatures(ids_to_delete)
            deleted_count = len(ids_to_delete)

        # 2) Delete duplicate geometries (exact duplicates by WKT)
        seen_geoms = set()
        dup_ids = []
        for f in layer.getFeatures():
            geom = f.geometry().asWkt()
            if geom in seen_geoms:
                dup_ids.append(f.id())
            else:
                seen_geoms.add(geom)

        if dup_ids:
            layer.deleteFeatures(dup_ids)

        total_deleted = deleted_count + len(dup_ids)

        try:
            layer.commitChanges()
        except Exception:
            try:
                layer.rollBack()
            except Exception:
                pass

        QMessageBox.information(
            self.iface.mainWindow(),
            "Done",
            f"Deleted {deleted_count} features not containing 'MR' in field 'TYP' and "
            f"{len(dup_ids)} duplicate geometries. Total: {total_deleted}."
        )
