# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Mappy
                                 A QGIS plugin
 helper for consistent goelogical map generation
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-06-24
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Luca Penasa, PLANMAP and GMAP team
        email                : luca.penasa@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QUrl
from PyQt5.QtGui import QDesktopServices
from qgis.utils import showPluginHelp

from .resources import * # DO NOT DELETE
from qgis.PyQt.QtWidgets import QMessageBox
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
# Initialize Qt resources from file resources.py
from qgis.core import QgsVectorFileWriter, QgsProject, QgsVectorLayer,  \
    QgsFeature, QgsMessageLog


from .mappy_utils import load_mappy_info_text
from .qgismappy_dockwidget import MappyDockWidget
from qgis.core import QgsApplication

from .providers import MappyProvider
import os.path



class Mappy:
    """QGIS Plugin Implementation."""

    instance = None

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'Mappy_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Mappy')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'Mappy')
        self.toolbar.setObjectName(u'Mappy')

        # print "** INITIALIZING Mappy"

        self.pluginIsActive = False
        self.dockwidget = None

        self.provider = None
        self.info_text = load_mappy_info_text()

        self.config_dock = MappyDockWidget()
        self.config_dock.closingPlugin.connect(self.close_config)
        # print(f"setting infobox text to {self.info_text}")
        self.config_dock.infobox.setHtml(self.info_text)
        self.iface.addDockWidget(Qt.RightDockWidgetArea, self.config_dock)

        v = self.getVersion()

        self.log_message(f"Mappy version: {v}")

        Mappy.instance = self

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Mappy', message)



    def getVersion(self):
        import mappy
        from pathlib import Path
        f = Path(mappy.__file__).parent.joinpath("metadata.txt")
        self.log_message(f"Reading version from  {f}")

        try:
            with open(f) as file:
                for l in file.readlines():
                    if l.startswith("version"):
                        return l.split("=")[1]
        except Exception as e:
            self.log_message("cannot determine version of mappy")

    def log_message(self, message, level=0, notifyUser=True):
        QgsMessageLog.logMessage(message, "Mappy", level, notifyUser)





    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/qgismappy/icons/settings.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Toggle Mappy config'),
            callback=self.toggle_config_dock,
            parent=self.iface.mainWindow())

        icon_path = ':/plugins/qgismappy/icons/reload.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Recompute map. Will overwrite data, hence pay attention to the config settings.'),
            callback=self.recompute_map,
            parent=self.iface.mainWindow())


        icon_path = None
        self.add_action(
            icon_path,
            text=self.tr(u'Online Help'),
            callback=self.openHelp,
            parent=self.iface.mainWindow())

        self.initProcessing()


    def openHelp(self):
        QDesktopServices.openUrl(QUrl("https://mappy.readthedocs.io"))

    def toggle_config_dock(self):
        if self.config_dock.isVisible():
            self.config_dock.setVisible(False)
        else:
            self.config_dock.setVisible(True)

    def close_config(self):
        self.config_dock.closingPlugin.disconnect(self.close_config)
        pass  # nothing relevant for now

    def check_if_layer_not_none_or_invalid(self, layer: QgsVectorLayer):
        if layer is None:
            status = "None"
        elif not layer.isValid():
            status = "Invalid"
        else:
            status = "ok"

        if status != "ok":
            return False, status
        else:
            return True, status

    def alert_box(self, title, message):
        dlg = QMessageBox()
        dlg.setWindowTitle(title)
        dlg.setText(message)
        return dlg.exec()

    def check_input_pars(self, pars):
        print(pars)
        try:
            lines = pars["lines"]
        except:
            self.alert_box("Error", "Missing lines layer. Please select it in the settings.")
            return False


        try:
            points = pars["points"]
        except:
            self.alert_box("Error", "Missing points layer. Please select it in the settings.")
            return False


        # points = pars["points"]
        lines: QgsVectorLayer
        points: QgsVectorLayer
        print("---------->")
        print(lines.sourceCrs().mapUnits())
        print(lines.sourceCrs().mapUnits())

        b, status = self.check_if_layer_not_none_or_invalid(lines)
        if not b:
            self.alert_box(f"Line layer {status}", "The layer is missing or invalid")
            return False

        b, status = self.check_if_layer_not_none_or_invalid(points)
        if not b:
            self.alert_box(f"Line layer {status}", "The layer is missing or invalid")
            return False



        lines_is_mod =lines.isModified()
        points_is_mod = points.isModified()

        layers = ""
        if lines_is_mod:
            layers += "lines"

        if points_is_mod:
            if layers == "":
                layers += "points"
            else:
                layers += " and points"



        if lines.isModified() or points.isModified():
            dlg = QMessageBox()
            dlg.setWindowTitle("Unsaved changes")
            dlg.setText(f"Input layer(s) \"{layers}\" have unsaved changes. Click ok to save and proceed with map creation")
            dlg.setStandardButtons(QMessageBox.Save | QMessageBox.Cancel)
            button = dlg.exec()



            if button == QMessageBox.Save:
                lines.commitChanges(False)
                points.commitChanges(False)
                return True
            else:
                return False

        return True

    def recompute_map(self):
        from .mappy_utils import collect_parameters

        pars = collect_parameters(self.config_dock)

        from qgis import processing

        if not self.check_input_pars(pars):
            return

        ofile = pars["output"]
        olayername = pars["out_polygons_layer_name"]
        o_cont_name = pars["out_contacts_layer_name"]


        args = {"IN_LINES": pars["lines"],
                "IN_POINTS": pars["points"],
                "OUTPUT": "TEMPORARY_OUTPUT",
                "UNMATCHED": "TEMPORARY_OUTPUT"}
        o = processing.run("mappy:mapconstruction", args)

        layer = o["OUTPUT"]
        unmatched = o["UNMATCHED"]
        points_layer = pars["points"]

        if pars["add_indicators"]:
            newpoints = processing.run("mappy:labelspointsfrompolygons", {
                'IN_LAYER': unmatched,
                'TOLERANCE': 1, 'OUTPUT': 'TEMPORARY_OUTPUT'})["OUTPUT"]

            newfeats = []
            for feature in newpoints.getFeatures():
                feature: QgsFeature
                print(f"ADDING FEATURE {feature}")

                newf = QgsFeature()
                newf.setGeometry(feature.geometry())

                newfeats.append(newf)

            points_layer.dataProvider().addFeatures(newfeats)

            points_layer.dataProvider().reloadData()
            points_layer.triggerRepaint()

        self.write_layer_to_gpkg(layer, ofile, olayername)
        self.load_layer_if_not_loaded(ofile, olayername, field_style=pars["units_field"])




        if pars["generate_clean_contacts"]:
            opts = {'Extenddistance': 0, 'PrecisionjoinBuffer': 0.001,
                    'contacts': pars["lines"],
                    'polygonized': layer,
                    'OUTPUT': 'TEMPORARY_OUTPUT'}
            layer = processing.run("mappy:removedangles", opts)["OUTPUT"]
            self.write_layer_to_gpkg(layer, ofile, o_cont_name)
            layer = self.load_layer_if_not_loaded(ofile, o_cont_name, None)

            if pars["copyoverlinestyle"]:
                print("copying layer style for lines")
                layer: QgsVectorLayer
                linelayer: QgsVectorLayer = pars["lines"]
                # renderer: QgsFeatureRenderer = linelayer.renderer()

                # newrend = type(linelayer.renderer())()

                newrend = linelayer.renderer().clone() # we clone the renderer
                # renderer.copyRendererData(newrend)

                layer.setRenderer(newrend)

                # print(f"new renderer {newrend}")

                # iface.layerTreeView().refreshLayerSymbology(layer.id())

                # layer.rendererChanged.emit()
                # layer.dataSourceChanged.emit()
                #
                # layer.triggerRepaint()

                # print(f"on layer {layer.name()}")

                # layerTreeView().refreshLayerSymbology(vlayer.id())


    def load_layer_if_not_loaded(self, gpkgfile, layername, field_style=None) -> QgsVectorLayer:
        l: QgsVectorLayer = self.findLayer(gpkgfile, layername)
        if l is None:
            l = self.addLayerFromGeopackage(gpkgfile, layername)
        else:
            l.dataProvider().reloadData()
            l.triggerRepaint()

        l.setReadOnly()

        if field_style:
            from .mappy_utils import resetCategoriesIfNeeded
            resetCategoriesIfNeeded(l, field_style)

        return l

    def write_layer_to_gpkg(self, layer, gpkgfile, layername):
        self.log_message(f"Writing layer {layer} to file {gpkgfile} with layername {layername}")

        from .mappy_utils import write_layer_to_gpkg2
        write_layer_to_gpkg2(layer, gpkgfile, layername)

    def findLayer(self, gpkg, layer_name):
        gpkg = os.path.abspath(gpkg)

        gpkg += f"|layername={layer_name}"
        layers = QgsProject.instance().mapLayers()

        for name, layer in layers.items():
            luri = layer.dataProvider().dataSourceUri()

            r = os.path.realpath
            if r(luri) == r(gpkg):
                return layer

        return None

    def addLayerFromGeopackage(self, gpkgfile, layer_name, categories_field=None):
        from .mappy_utils import add_layer_from_geopackage
        l = add_layer_from_geopackage(gpkgfile, layer_name, categories_field=None)


        # if categories_field is not None:
        #     self.resetCategoriesIfNeeded(l, categories_field)

        # l.triggerRepaint()
        # l.dataChanged.emit()  # or dataSourceChanged?
        # l.dataSourceChanged.emit()
        return l

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Mappy'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar
        del self.config_dock

        QgsApplication.processingRegistry().removeProvider(self.provider)

    def initProcessing(self):
        self.provider = MappyProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)
