# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MappiaPublisher
                                 A QGIS plugin
 Publish your maps easily
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-12-09
        copyright            : (C) 2019 by Danilo/CSR UFMG
        email                : danilo@csr.ufmg.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Danilo da Silveira Figueira / CSR UFMG'
__date__ = '2020-01-21'
__copyright__ = '(C) 2020 by Danilo da Silveira Figueira / CSR UFMG'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import math
import os
import random
import traceback
import requests
import csv
import re
from datetime import date
import json
from pathlib import Path
import platform
import webbrowser
import tempfile
import subprocess
from mappia_publisher.WMSCapabilities import WMSCapabilities
from mappia_publisher.OptionsCfg import OptionsCfg
from mappia_publisher.GitHub import GitHub
from mappia_publisher.UTILS import UTILS
from mappia_publisher.UTILS import UserInterrupted
from enum import Enum
from qgis.utils import iface
from qgis.PyQt.QtCore import QCoreApplication, QSize, Qt, QVariant
from qgis.PyQt.QtWidgets import QMessageBox
from qgis.PyQt.QtGui import QImage, QColor, QPainter
from mappia_publisher.MetaTile import MetaTile
from qgis.core import (QgsProject,
                       QgsPointXY,
                       QgsLogger,
                       QgsField,
                       QgsProcessing,
                       QgsMessageLog,
                       QgsRectangle,
                       QgsMapSettings,
                       QgsRasterLayer,
                       QgsCoordinateTransform,
                       QgsRenderContext,
                       QgsMapRendererParallelJob,
                       QgsVectorFileWriter,
                       QgsVectorLayer,
                       QgsWkbTypes,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterExtent,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterString,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsLabelingEngineSettings,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterFolderDestination,
                       QgsMapRendererCustomPainterJob,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterMapLayer,
                       QgsProcessingParameterMultipleLayers,
                       QgsProcessingParameterEnum,
                       QgsVectorSimplifyMethod,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink)
from .OperationType import OperationType
from .DirectoryWriter import DirectoryWriter


class MappiaPublisherAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    LAYERS = 'LAYERS'
    OPERATION = 'OPERATION'
    OUTPUT_DIRECTORY = 'OUTPUT_DIRECTORY'
    ZOOM_MAX = 'ZOOM_MAX'
    EXTENT = 'EXTENT'
    LAYER_ATTRIBUTE = 'LAYER_ATTRIBUTE'
    INCLUDE_DOWNLOAD = 'INCLUDE_DOWNLOAD'
    ASK_USER = 'ASK_USER'

    GITHUB_REPOSITORY = 'GITHUB_REPOSITORY'
    GITHUB_USER = 'GITHUB_USER'
    GITHUB_PASS = 'GITHUB_PASS'
    GIT_EXECUTABLE = 'GIT_EXECUTABLE'
    SAVE_PASSWORD = 'SAVE_PASSWORD'

    # Default size of the WMS tiles.
    WIDTH = 256
    HEIGHT = 256

    OUTPUT_DIR_TMP = None

    version = '2.10.1'

    last_update_date = None

    found_git = ''

    ghPassword = ''

    def isPluginUpdated(self):
        todayStr = date.today().strftime("%Y %m %d")
        if self.last_update_date is None or self.last_update_date < todayStr:
            versions = self.version.split('.')
            nextSmalVersion = versions[0] + '.' + versions[1] + '.' + str(int(versions[2]) + 1)
            nextMidVersion = versions[0] + '.' + str(int(versions[1]) + 1) + '.0'
            nextBigVersion = str(int(versions[0]) + 1) + '.0.0'
            needUpdate = False

            seekMsg = 'Approved'

            for url in [nextSmalVersion, nextMidVersion, nextBigVersion]:
                if needUpdate == False:
                    res = requests.get('https://plugins.qgis.org/plugins/mappia_publisher/version/'+url+'/')
                    needUpdate = res.status_code == 200 and seekMsg in res.text and 'yes' in res.text[res.text.index(seekMsg):]
            return not needUpdate
        else:
            self.last_update_date = todayStr
            return True


    def initAlgorithm(self, config):
        print('initAlgorithm()')

        #super(MappiaPublisherAlgorithm, self).initAlgorithm()
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        options = OptionsCfg.read()

        self.last_update_date = options[OptionsCfg.UPDATE_CHECK]

        #self.addParameter(
        #    QgsProcessingParameterEnum(
        #        self.OPERATION,
        #        self.tr('Operation Type'),
        #        options=[self.tr(curOption) for curOption in OperationType.getOptions()],
        #        defaultValue=9
        #    )
        #)


        #TODO Implement layer attribute later.
        # self.addParameter(
        #     QgsProcessingParameterString(
        #         self.LAYER_ATTRIBUTE,
        #         self.tr('Layer style name (Change to different style names only if interested in publishing multiple (layers or attributes) of the same file.)'),
        #         optional=False,
        #         defaultValue=options['attrName']
        #     )
        # )

        self.addParameter(
            QgsProcessingParameterString(
                self.GITHUB_REPOSITORY,
                self.tr('* Repository name (or map group name)'),
                optional=False,
                defaultValue=options['gh_repository']
            )
        )

        # We add the input vector features source. It can have any kind of
        # geometry.
        layerParam = QgsProcessingParameterMultipleLayers(
            self.LAYERS,
            self.tr('* Maps to display online'),
            QgsProcessing.TypeMapLayer
            #, defaultValue=[layer.id() for layer in iface.mapCanvas().layers()]#[layer.dataProvider().dataSourceUri(False) for layer in iface.mapCanvas().layers()]
        )
        layerParam.setMinimumNumberInputs(1)
        self.addParameter(layerParam)
        self.m_layerParam = layerParam

        maxZoomParameter = QgsProcessingParameterNumber(
            self.ZOOM_MAX,
            self.tr('Map max Zoom level [1 ~ 13] (lower is faster)'),
            minValue=1,
            maxValue=13,
            defaultValue=options["zoom_max"]
        )
        maxZoomParameter.setFlags(maxZoomParameter.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(maxZoomParameter)

        includeSourceDownload = QgsProcessingParameterBoolean(
            self.INCLUDE_DOWNLOAD,
            self.tr('Upload maps for further download (2GB limit per unique map)'),
            defaultValue=options['include_dl']
        )
        includeSourceDownload.setFlags(includeSourceDownload.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(includeSourceDownload)

        gitExeParameter = QgsProcessingParameterString(
            self.GIT_EXECUTABLE,
            self.tr('Git client executable path.'),
            optional=True,
            defaultValue=UTILS.getGitDefault(options['git_exe'])
        )
        gitExeParameter.setFlags(gitExeParameter.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(gitExeParameter)

        ghUserParameter = QgsProcessingParameterString(
            self.GITHUB_USER,
            self.tr('Github USERNAME (Credentials for https://github.com)'),
            optional=True,
            defaultValue=options['gh_user']
        )
        ghUserParameter.setFlags(ghUserParameter.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(ghUserParameter)

        # ghPassParameter = QgsProcessingParameterString(
        #     self.GITHUB_PASS,
        #     self.tr('Github Access Token *'),
        #     optional=True,
        #     defaultValue=options['gh_pass']
        # )
        # ghPassParameter.setFlags(ghPassParameter.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        # self.addParameter(ghPassParameter)

        askUserConfirmation = QgsProcessingParameterBoolean(
            self.ASK_USER,
            self.tr('Ask user confirmation before each step.'),
            defaultValue=options['ask_user']
        )
        askUserConfirmation.setFlags(askUserConfirmation.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(askUserConfirmation)

        outputDirParameter = QgsProcessingParameterFolderDestination(
            self.OUTPUT_DIRECTORY,
            self.tr('Output directory'),
            optional=True,
            defaultValue=options["folder"] or None
        )
        outputDirParameter.setFlags(outputDirParameter.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(outputDirParameter)

    #Create the metatiles to the given layer in given zoom levels
    @staticmethod
    def createLayerMetatiles(projection, layer, minZoom, maxZoom):
        mapExtentReprojected = UTILS.getMapExtent(layer, projection)
        metatiles_by_zoom = {}
        metatilesize = 4
        for zoom in range(minZoom, maxZoom + 1):
            metatiles = UTILS.get_metatiles(mapExtentReprojected, zoom, metatilesize)
            metatiles_by_zoom[zoom] = metatiles
        return metatiles_by_zoom

    def getTotalTiles(self, wgs_crs, min_zoom, max_zoom, layers):
        total = 0
        allTiles = {}
        for layer in layers:
            metatiles_by_zoom = MappiaPublisherAlgorithm.createLayerMetatiles(wgs_crs, layer, min_zoom, max_zoom)
            allTiles[layer] = metatiles_by_zoom
            for zoom in range(min_zoom, max_zoom + 1):
                total = total + len(metatiles_by_zoom[zoom])
        return allTiles, total

    def preprocessParameters(self, parameters):
        print('preprocessParameters')
        MappiaPublisherAlgorithm.ghPassword = ''
        return parameters

    @staticmethod
    def isPointLayer(layer, feedback):
        try:
            return isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.GeometryType.PointGeometry
        except Exception as e:
            # try: #test in older qgis.
            #     return isinstance(layer, QgsVectorLayer) and (layer.geometryType() == QgsWkbTypes.Type.Point
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPoint
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPointZ
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPointM
            #                                                   or layer.geometryType() == gsWkbTypes.Type.MultiPointZM
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPoint25D
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPoint
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.PointZ
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPointZ
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.PointM
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPointM
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.PointZM
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPointZM
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.Point25D
            #                                                   or layer.geometryType() == QgsWkbTypes.Type.MultiPoint25D)
            # except:
            #     return False
            feedback.pushConsoleInfo("Warning: Error to identify the Geometry type " + str(e))
            return False


    def generate(self, writer, parameters, context, feedback):
        print('generate()')
        from concurrent import futures
        #print('Parameters:\n ' +json.dumps(parameters))
        # try:
        feedback.setProgress(1)
        feedback.setProgressText("This step can take a long time, and despite the job is going, sometime the interface can freezes.\nThe job is still going, please wait and do not close the application until it finishes.")
        min_zoom = 0
        max_zoom = self.parameterAsInt(parameters, self.ZOOM_MAX, context)
        outputFormat = QImage.Format_ARGB32 #Format_ARGB32_Premultiplied
        layerAttr = "1" #TODO self.parameterAsString(parameters, self.LAYER_ATTRIBUTE, context)
        cellType = 'int32'
        nullValue = -128
        includeDl = self.parameterAsBool(parameters, self.INCLUDE_DOWNLOAD, context)
        ghUser = self.parameterAsString(parameters, self.GITHUB_USER, context)
        ghPassword = MappiaPublisherAlgorithm.ghPassword
        ghRepository = self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context)
        mapOperation = OperationType.RGBA #OperationType(self.parameterAsEnum(parameters, self.OPERATION, context))
        wgs_crs = QgsCoordinateReferenceSystem('EPSG:4326')
        dest_crs = QgsCoordinateReferenceSystem('EPSG:3857')
        layers = self.parameterAsLayerList(parameters, self.LAYERS, context)
        #allLayerMetaTiles, metatilesCount = self.getTotalTiles(wgs_crs, min_zoom, max_zoom, layers)
        allLayerMetaTiles, metatilesCount = UTILS.runLongTask(self.getTotalTiles, feedback, 'Please wait, creating a zipfile of your map.', 180, wgs_crs, min_zoom, max_zoom, layers)
        renderedTiles = 0
        if includeDl:
            try:
                GitHub.createDownloadTag(ghUser, ghRepository, ghPassword, feedback)
            except Exception as ex:
                feedback.pushConsoleInfo('Warning: Ignoring uploading map files to repository. Reason: ' + str(ex))
                includeDl = False
            #Danilo precisa de commit?
        #FIXME Dar um update e um push
        progress = 0
        percentageByLayer = 90 / len(layers)
        toPercentage = 1
        for layer in layers:
            feedback.setProgress(toPercentage)
            layerTitle = layer.name()
            UTILS.checkForCanceled(feedback)
            feedback.setProgressText("Publishing layer: " + layerTitle)
            if MappiaPublisherAlgorithm.isPointLayer(layer, feedback):
                feedback.setProgressText("Publishing a shape of point geometry")
                if writer.processPointsLayer(feedback, layer, layerAttr, wgs_crs):
                    writer.write_custom_capabilities(layerTitle, layerAttr, "csv")
                else:
                    feedback.setProgressText("Error publishing map")
            else:
                feedback.pushConsoleInfo(f"Rendering tiles to publish online using {os.cpu_count()} threads.")
                metatiles_by_zoom = allLayerMetaTiles[layer] #MappiaPublisherAlgorithm.createLayerMetatiles(wgs_crs, layer, min_zoom, max_zoom)
                feedback.setProgressText(f"Rendering {renderedTiles}/{metatilesCount} MetaTile")
                with futures.ThreadPoolExecutor(max_workers=os.cpu_count()) as executor:
                    for zoom in range(min_zoom, max_zoom + 1):
                        UTILS.checkForCanceled(feedback)
                        feedback.pushConsoleInfo(f'Generating tiles for zoom level: {zoom}')

                        jobs = []
                        for metatile in metatiles_by_zoom[zoom]:
                            if feedback.isCanceled():
                                break
                            jobs.append(executor.submit(MappiaPublisherAlgorithm.drawMetatileToFile, mapOperation, layerAttr, layerTitle, metatile, layer, outputFormat, wgs_crs, dest_crs, feedback, context, writer))

                        for completed_job in futures.as_completed(jobs):
                            completed_job.result()  # Process job result if needed
                            renderedTiles += 1
                            progress += 1
                            feedback.setProgressText(f"Rendering MetaTile nº: {renderedTiles}/{metatilesCount}")
                            feedback.setProgress(toPercentage + (percentageByLayer * (progress / metatilesCount)))

                # writer.writeLegendPng(layer, layerTitle, layerAttr, mapOperation.getName())
                downloadLink = None #Zip file and upload it to download.
                if includeDl:
                    curFileDl = UTILS.runLongTask(UTILS.generateZipLayer, feedback, 'Please wait, creating a zipfile of your map.', 30, layer)
                    UTILS.checkForCanceled(feedback)
                    if curFileDl is not None:
                        downloadLink = GitHub.addReleaseFile(ghUser, ghPassword, ghRepository, 2, True, os.path.abspath(curFileDl.name), layer, feedback)
                        if downloadLink is not None:
                            feedback.pushConsoleInfo("Map download link: " + downloadLink)
                        # curFileDl.close()
                        # os.remove(curFileDl.name) #está dando erro algumas vezes ao remover o arquivo temporário.
                        curFileDl = None
                WMSCapabilities.write_description(writer.getPathForMap(layerTitle, layerAttr.lower()), layerTitle, layerAttr, cellType, nullValue, mapOperation.getName()) #Danilo acho q movi
                writer.write_capabilities(layer, layerTitle, layerAttr, max_zoom, downloadLink)
                writer.writeLegendJson(writer.getPathForMap(layerTitle, layerAttr.lower(), mapOperation.getName()), layer, layerTitle, layerAttr, mapOperation.getName())
                layerRenderSettings = MappiaPublisherAlgorithm.createLayerRenderSettings(layer, dest_crs, outputFormat, feedback)
                writer.writeThumbnail(UTILS.getMapExtent(layer, dest_crs), layerTitle, layerAttr, mapOperation.getName(), layerRenderSettings)
                toPercentage = toPercentage + percentageByLayer
        writer.setCapabilitiesDefaultMaxZoom()
        feedback.setProgressText('Finished map tile generation. Uploading changes to Github.')
        feedback.setProgress(94)
        GitHub.publishTilesToGitHub(writer.folder, ghUser, ghRepository, feedback, self.version, ghPassword)
        feedback.setProgress(96)
        storeUrl = GitHub.getGitUrl(ghUser, ghRepository)
        #FIXME Space in the map name will cause errors.
        feedback.pushConsoleInfo("All the maps in this repository:\n(Please only use maps with the same zoom level.)\n")
        allLayers = WMSCapabilities.getAllLayers(Path(os.path.join(writer.folder, "getCapabilities.xml")).read_text(), layerAttr)
        allPointLayers = WMSCapabilities.getAllCustomLayers(Path(os.path.join(writer.folder, "getCapabilities.xml")).read_text(), layerAttr)
        feedback.pushConsoleInfo("https://maps.csr.ufmg.br/calculator/?queryid=152&storeurl=" + storeUrl
            + "/" + "&remotemap=" + ",".join(allLayers) + "&points=" + ",".join(allPointLayers) + "\n")
        feedback.pushConsoleInfo("Current published Maps:\n")
        generatedMaps = ["GH:" + UTILS.normalizeName(layer.name()) + ";" + layerAttr for layer in layers if not self.isPointLayer(layer, feedback)]
        pointLayers = ["GH:" + UTILS.normalizeName(layer.name()) for layer in layers if self.isPointLayer(layer, feedback)]
        curMapsUrl = "https://maps.csr.ufmg.br/calculator/?queryid=152&storeurl=" + storeUrl + "/"
        if len(generatedMaps) > 0:
            curMapsUrl += "&remotemap=" + ",".join(generatedMaps)
        if len(pointLayers) > 0:
            curMapsUrl += "&points=" + ",".join(pointLayers)
        feedback.setProgressText("Copy the link above in any browser to see your maps online. ")
        feedback.pushConsoleInfo(curMapsUrl)
        feedback.setProgress(100)
        writer.close()
        webbrowser.open_new(curMapsUrl)
        #QMessageBox.warning(None, "Maps published online", "Congratulations your maps are now available online.\nThe shareable link was generated, and will be shown in your browser.")
        return {"MAPS_URL": curMapsUrl, 'PUBLISHED_MAPS': generatedMaps, 'REPOSITORY_MAPS': allLayers}
        # except UserInterrupted as ex:
        #     return {"CANCELED": ex}

    @staticmethod
    def drawMetatileToFile(mapOperation, layerAttr, layerTitle, metatile, layer, outputFormat, inputWgsCrs,
                           outputDestCrs, feedback, context, writer):
        layerRenderSettings = MappiaPublisherAlgorithm.createLayerRenderSettings(layer, outputDestCrs, outputFormat,
                                                                                 feedback)
        transformContext = context.transformContext()
        mapRendered = MappiaPublisherAlgorithm.renderMetatile(metatile, outputDestCrs, layerRenderSettings,
                                                              transformContext, inputWgsCrs, MappiaPublisherAlgorithm.WIDTH, MappiaPublisherAlgorithm.HEIGHT)
        for r, c, tile in metatile.tiles:
            tile_img = mapRendered.copy(MappiaPublisherAlgorithm.WIDTH * r, MappiaPublisherAlgorithm.HEIGHT * c, MappiaPublisherAlgorithm.WIDTH, MappiaPublisherAlgorithm.HEIGHT)
            writer.write_tile(tile, tile_img, mapOperation.getName(), layerTitle, layerAttr)

    #Return the rendered map (QImage) for the metatile zoom level.
    @staticmethod
    def renderMetatile(metatile, dest_crs, renderSettings, transformContext, sourceCrs, width, height):
        wgsTransform = QgsCoordinateTransform(sourceCrs, dest_crs, transformContext)
        renderSettings.setExtent(wgsTransform.transformBoundingBox(QgsRectangle(*metatile.extent())))
        size = QSize(width * metatile.rows(), height * metatile.columns())
        image = QImage(size, renderSettings.outputImageFormat())
        renderSettings.setOutputDpi(image.logicalDpiX())
        renderSettings.setOutputSize(image.size())
        image.fill(Qt.transparent)
        dpm = round(renderSettings.outputDpi() / 25.4 * 1000)
        image.setDotsPerMeterX(dpm)
        image.setDotsPerMeterY(dpm)
        painter = QPainter(image)
        #painter.setRenderHints(QPainter.RenderHint.NonCosmeticDefaultPen, on=True)
        #painter.setCompositionMode(QPainter.CompositionMode_Source)
        job = QgsMapRendererCustomPainterJob(renderSettings, painter)
        job.renderSynchronously()
        painter.end()
        return image

    #Configure the rendering settings for the WMS tiles.
    @staticmethod
    def createLayerRenderSettings(layer, dest_crs, outputFormat, feedback):
        settings = QgsMapSettings()
        try:
            settings.setFlag(QgsMapSettings.Flag.Antialiasing, False)
        except:
            pass

        try:
            settings.setFlag(QgsMapSettings.Flag.UseRenderingOptimization, True)
        except:
            feedback.pushConsoleInfo("Warning: Rendering optimizations not available in your QGIS version.")
            pass
        try:
            settings.setFlag(QgsMapSettings.Flag.UseAdvancedEffects, False)
        except:
            feedback.pushConsoleInfo("Warning: Opacity and blending effects not available in your QGIS version.")
            pass

        settings.setOutputImageFormat(outputFormat)
        settings.setDestinationCrs(dest_crs)
        settings.setLayers([layer])
        #settings.setOutputDpi(256)
        color = QColor(0, 0, 0, 0)
        settings.setBackgroundColor(color)
        try:
            labeling_engine_settings = settings.labelingEngineSettings()
            try:
                labeling_engine_settings.setFlag(QgsLabelingEngineSettings.UsePartialCandidates, False)
            except:
                feedback.pushConsoleInfo("Warning: Cannot draw label candidates that are partially outside of the map view. Your QGIS version has not 'UsePartialCandidates'.")
                pass
            settings.setLabelingEngineSettings(labeling_engine_settings)
        except:
            feedback.pushConsoleInfo("Warning: This QGIS version does not support label Engine.")
        try:
            layer.resampleFilter().setZoomedInResampler(None)
            layer.resampleFilter().setZoomedOutResampler(None)
            layer.resampleFilter().setOn(False)
        except:
            pass #Is not a raster
        return settings


    def checkParameterValues(self, parameters, context):
        def checkZoomLevels(self, parameters, context):
            min_zoom = 0
            max_zoom = self.parameterAsInt(parameters, self.ZOOM_MAX, context)

            if max_zoom < min_zoom:
                return self.tr("Invalid zoom levels range.")
            else:
                return None

        def checkLayerList(self, parameters, context):
            if len(self.parameterAsLayerList(parameters, self.LAYERS, context)) <= 0:
                return self.tr("Please select one map or more.")
            else:
                return None
            
        def checkGitRepositoryName(self, parameters, context):
            gitRepository = self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context)
            if ' ' in gitRepository:
                return self.tr("Space is not allowed in Repository name (remote folder).")
            else:
                return None
        
        def checkCredentials(self, parameters, context):
            self.findGithubUserName(parameters)
            if GitHub.testLogin(self.parameterAsString(parameters, self.GITHUB_USER, context), MappiaPublisherAlgorithm.ghPassword) == False:
                return self.tr("Invalid user or password. Please visit the link https://github.com/login and check your password.")
            else:
                return None
        
        def checkOutputDirectory(self, parameters, context):
            try:
                gitExe = GitHub.findGitExe(parameters[self.GIT_EXECUTABLE], self.found_git, None, False)

                if gitExe:
                    GitHub.prepareEnvironment(gitExe)
                    return self.checkOutputDirectoryMatchRepository(parameters, context)
            except Exception as e:
                pass # If failed to find the git executable just ignore it.
            
            return None

        print('checkParameterValues()')

        errors = []
        
        error = checkZoomLevels(self, parameters, context)
        if error != None:
            errors.append(error)
        
        error = checkLayerList(self, parameters, context)
        if error != None:
            errors.append(error)
        
        error = checkGitRepositoryName(self, parameters, context)
        if error != None:
            errors.append(error)
        
        error = checkOutputDirectory(self, parameters, context)
        if error != None:
            errors.append(error)

        if len(errors) > 0:
            errorMessage = "Erros:\n"
            errorsCount = 1
            for error in errors:
                errorMessage += str(errorsCount) + ") " + error + "\n\n"
                errorsCount += 1
            return False, errorMessage

        error = checkCredentials(self, parameters, context)
        if error != None:
            errorMessage = "Erros:\n\n1) " + error
            return False, errorMessage

        return super().checkParameterValues(parameters, context)    
    
    def findGithubUserName(self, parameters):
        result = GitHub.getGitCredentials(parameters[self.GITHUB_USER], MappiaPublisherAlgorithm.ghPassword, parameters[self.ASK_USER])
        parameters[self.GITHUB_USER], MappiaPublisherAlgorithm.ghPassword = result
    
    def checkOutputDirectoryMatchRepository(self, parameters, context):
        outputDirecotry = self.parameterAsString(parameters, self.OUTPUT_DIRECTORY, context)
        gitRepository = self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context)

        if GitHub.isGitRepository(outputDirecotry):
            outputDirectoryRepositoryName = GitHub.getRepoName(outputDirecotry)

            if outputDirectoryRepositoryName != gitRepository:
                return self.tr('The Repository name "' + gitRepository + '" does not match the output reposiory: ' + outputDirectoryRepositoryName + '".')
        
        return None

    def prepareAlgorithm(self, parameters, context, feedback):
        print("prepareAlgorithm() Started: Verify the input options.")
        feedback.pushConsoleInfo("Started: Verify the input options.")
        ghUser = self.parameterAsString(parameters, self.GITHUB_USER, context)
        mustAskUser = self.parameterAsBool(parameters, self.ASK_USER, context)
        ghRepository = self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context)
        ghPassword = MappiaPublisherAlgorithm.ghPassword
        if not self.isPluginUpdated():
            feedback.pushConsoleInfo("Warning: Please update your Mappia plugin, there is a new versison.")
        print("selfPass " + MappiaPublisherAlgorithm.ghPassword)

        gitExe = GitHub.findGitExe(parameters[self.GIT_EXECUTABLE], self.found_git, feedback, mustAskUser)
        feedback.pushConsoleInfo("Automatic Step: Checking git executable.")
        if gitExe:
            GitHub.prepareEnvironment(gitExe)
        feedback.pushConsoleInfo(gitExe)
        if not gitExe or not os.path.isfile(gitExe):
            feedback.pushConsoleInfo("Select your git executable program.\n" + str(
                gitExe) + "\nIt can be downloadable at: https://git-scm.com/downloads")
            return False
        
        self.OUTPUT_DIR_TMP = self.parameterAsString(parameters, self.OUTPUT_DIRECTORY, context)
        if len(self.OUTPUT_DIR_TMP) <= 0:
            self.OUTPUT_DIR_TMP = tempfile.mkdtemp()

        feedback.pushConsoleInfo("Automatic Step: Checking if the output directory has the right git repository.")
        error = self.checkOutputDirectoryMatchRepository(parameters, context)
        if error != None:
            feedback.pushConsoleInfo(error)
            return False
        
        feedback.pushConsoleInfo("Automatic Step: Checking remote repository.")
        if not UTILS.isQgisSupported():
            feedback.pushConsoleInfo("Warning: This plugin was developped for QGIS 3.x+ please consider update.\n" + (("The identified QGIS Verison is " + UTILS.getQGISversion()) if UTILS.getQGISversion() is not None else "Version could not be idenfied."))
        feedback.pushConsoleInfo("Automatic Step: Configuring git parameters.")
        if (not GitHub.existsRepository(ghUser, ghRepository, ghPassword)) and mustAskUser and (QMessageBox.Yes != QMessageBox.question(
                None,
                "Repository not found",
                "The repository was not found, want to create a new repository?",
                defaultButton=QMessageBox.Yes,
                buttons=(QMessageBox.Yes | QMessageBox.No | QMessageBox.Cancel))):
            feedback.pushConsoleInfo("Error: A valid repository is needed please enter a valid repository name or create a new one.")
            return False
        elif ((not GitHub.existsRepository(ghUser, ghRepository, ghPassword) and not GitHub.createRepo(ghRepository, ghUser, ghPassword, self.OUTPUT_DIR_TMP, feedback))
                or (GitHub.existsRepository(ghUser, ghRepository, ghPassword) and not GitHub.isRepositoryInitialized(ghUser, ghRepository)
                    and not GitHub.initializeRepository(self.OUTPUT_DIR_TMP, ghUser, ghRepository, ghPassword, feedback))):
            if not mustAskUser or QMessageBox.question(
                    None,
                    "The creation have failed. Want to open the link https://github.com/new to create a new repository?",
                    defaultButton=QMessageBox.Yes) == QMessageBox.Yes:
                webbrowser.open_new("https://github.com/new")
            feedback.pushConsoleInfo("Error: Failed to create the repository " + ghRepository + ".\nPlease create manually one at: https://github.com/new")
            return False
        if not self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context):
            feedback.pushConsoleInfo("Please specify your repository name.\nYou can create one at: https://github.com/new")
            return False
        ghIncludeDownload = self.parameterAsBool(parameters, self.INCLUDE_DOWNLOAD, context)
        #Danilo depois checar o tamanho dos mapas anteriormente e caso passar 2gb perguntar o usuário se poderia continuar.
        if not GitHub.isOptionsOk(self.OUTPUT_DIR_TMP, ghUser, ghRepository, feedback, ghPassword, mustAskUser):
            feedback.setProgressText("Error: Canceling the execution, please select another output folder.")
            return False
        feedback.pushConsoleInfo("Automatic Step: Saving option changes.")
        OptionsCfg.write(
            self.parameterAsInt(parameters, self.ZOOM_MAX, context),
            self.parameterAsString(parameters, self.GIT_EXECUTABLE, context),
            self.parameterAsString(parameters, self.LAYER_ATTRIBUTE, context),
            self.parameterAsString(parameters, self.GITHUB_USER, context),
            self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context),
            self.OUTPUT_DIR_TMP,
            self.parameterAsBool(parameters, self.INCLUDE_DOWNLOAD, context),
            self.parameterAsBool(parameters, self.ASK_USER, context),
            self.last_update_date
        )
        feedback.pushConsoleInfo("Automatic Step: Changes Saved.")
        return True

    def processAlgorithm(self, parameters, context, feedback):
        is_tms = False
        if not self.OUTPUT_DIR_TMP:
            raise QgsProcessingException(self.tr('You need to specify output directory.'))
        writer = DirectoryWriter(self.OUTPUT_DIR_TMP, is_tms)
        feedback.pushConsoleInfo("Ok, all inputs are valid.")
        try:
            return self.generate(writer, parameters, context, feedback)
        except Exception as e:
            reportContent = {'Status': 'Error, publication failed! Reporting error to Mappia development team.', "QGIS_version": UTILS.getQGISversion(), 'version': self.version, 'error': self.removeConfidential(traceback.format_exc()), 'exception': self.removeConfidential(str(e)), 'os': platform.platform()}
            UTILS.sendReport(reportContent)
            feedback.pushConsoleInfo('Error, publication failed! Reporting error to Mappia development team.')
            raise Exception(self.removeConfidential(str(e)))

    @staticmethod
    def removeConfidential(str):
        return re.sub(r'http[s]?://[^@\n]+@github\.com', 'https://USER:PASS@github.com', str)

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Share'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr('Share your maps')

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Online mapping'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return MappiaPublisherAlgorithm()