# -*- coding: utf-8 -*-

"""
/***************************************************************************
 MappiaPublisher
                                 A QGIS plugin
 Publish your maps easily
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-12-09
        copyright            : (C) 2019 by Danilo/CSR UFMG
        email                : danilo@csr.ufmg.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Danilo da Silveira Figueira / CSR UFMG'
__date__ = '2020-01-21'
__copyright__ = '(C) 2020 by Danilo da Silveira Figueira / CSR UFMG'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import math
import os
import re
import csv
import json
import collections
import io
from pathlib import Path
import pip
import requests
from datetime import datetime

from enum import Enum
from qgis.PyQt.QtCore import QCoreApplication, QSize, Qt
from qgis.PyQt.QtGui import QImage, QColor, QPainter
from qgis.core import (QgsProject,
                       QgsLogger,
                       QgsProcessing,
                       QgsMessageLog,
                       QgsRectangle,
                       QgsMapSettings,
                       QgsRasterLayer,
                       QgsCoordinateTransform,
                       QgsMapRendererParallelJob,
                       QgsProcessingParameterExtent,
                       QgsProcessingParameterString,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsLabelingEngineSettings,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterFolderDestination,
                       QgsMapRendererCustomPainterJob,
                       QgsCoordinateReferenceSystem,
                       QgsProcessingParameterMapLayer,
                       QgsProcessingParameterMultipleLayers,
                       QgsProcessingParameterEnum,
                       QgsVectorSimplifyMethod,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink)


# TMS functions taken from https://alastaira.wordpress.com/2011/07/06/converting-tms-tile-coordinates-to-googlebingosm-tile-coordinates/ #spellok
def tms(ytile, zoom):
    n = 2.0 ** zoom
    ytile = n - ytile - 1
    return int(ytile)


# Math functions taken from https://wiki.openstreetmap.org/wiki/Slippy_map_tilenames #spellok
def deg2num(lat_deg, lon_deg, zoom):
    QgsMessageLog.logMessage(" e ".join([str(lat_deg), str(lon_deg), str(zoom)]), tag="Processing")
    lat_rad = math.radians(lat_deg)
    n = 2.0 ** zoom
    xtile = int((lon_deg + 180.0) / 360.0 * n)
    ytile = int((1.0 - math.log(math.tan(lat_rad) + (1 / math.cos(lat_rad))) / math.pi) / 2.0 * n)
    return (xtile, ytile)


# Math functions taken from https://wiki.openstreetmap.org/wiki/Slippy_map_tilenames #spellok
def num2deg(xtile, ytile, zoom):
    n = 2.0 ** zoom
    lon_deg = xtile / n * 360.0 - 180.0
    lat_rad = math.atan(math.sinh(math.pi * (1 - 2 * ytile / n)))
    lat_deg = math.degrees(lat_rad)
    return (lat_deg, lon_deg)

# https://gis.stackexchange.com/questions/130027/getting-a-plugin-path-using-python-in-qgis
def resolve(name, basepath=None):
    if not basepath:
      basepath = os.path.dirname(os.path.realpath(__file__))
    return os.path.join(basepath, name)

# From Tiles XYZ algorithm
class Tile:
    def __init__(self, x, y, z):
        self.x = x
        self.y = y
        self.z = z

    def extent(self):
        lat1, lon1 = num2deg(self.x, self.y, self.z)
        lat2, lon2 = num2deg(self.x + 1, self.y + 1, self.z)
        return [lon1, lat2, lon2, lat1]

class OptionsCfg:

    @staticmethod
    def createCfg(zoom_max=None, gitExe=None, attrName=None, ghUser=None, ghRepository=None, folder=None):
        defaults = dict()
        defaults["zoom_max"] = zoom_max if zoom_max is not None else 9
        defaults["git_exe"] = gitExe if gitExe is not None else ''
        defaults["attrName"] = attrName if attrName is not None else '1'
        defaults["gh_user"] = ghUser if ghUser is not None else ''
        defaults["gh_repository"] = ghRepository if ghRepository is not None else ''
        defaults["folder"] = folder if folder is not None else ''
        return defaults

    @staticmethod
    def write(zoom_max, gitExe, attrName, ghUser, ghRepository, folder):
        with open(resolve("options.json"), 'w', encoding="utf-8") as f:
            json.dump(OptionsCfg.createCfg(zoom_max, gitExe, attrName, ghUser, ghRepository, folder), f)

    @staticmethod
    def read():
        with open(resolve("options.json"), 'r', encoding="utf-8") as f:
            cfg = json.load(f)
            defaults = OptionsCfg.createCfg()
            for key in cfg.keys():
                defaults[key] = cfg[key]
        return defaults


def get_metatiles(extent, zoom, size=4):
    #west_edge, south_edge, east_edge, north_edge = extent
    #[extent.xMinimum(), extent.yMinimum(), extent.xMaximum(), extent.yMaximum()]
    left_tile, top_tile = deg2num(extent.yMaximum(), extent.xMinimum(), zoom)
    right_tile, bottom_tile = deg2num(extent.yMinimum(), extent.xMaximum(), zoom)

    metatiles = {}
    for i, x in enumerate(range(left_tile, right_tile + 1)):
        for j, y in enumerate(range(top_tile, bottom_tile + 1)):
            meta_key = '{}:{}'.format(int(i / size), int(j / size))
            if meta_key not in metatiles:
                metatiles[meta_key] = MetaTile()
            metatile = metatiles[meta_key]
            metatile.add_tile(i % size, j % size, Tile(x, y, zoom))

    return list(metatiles.values())



# From Tiles XYZ algorithm
class MetaTile:
    def __init__(self):
        # list of tuple(row index, column index, Tile)
        self.tiles = []

    def add_tile(self, row, column, tile):
        self.tiles.append((row, column, tile))

    def rows(self):
        return max([r for r, _, _ in self.tiles]) + 1

    def columns(self):
        return max([c for _, c, _ in self.tiles]) + 1

    def extent(self):
        _, _, first = self.tiles[0]
        _, _, last = self.tiles[-1]
        lat1, lon1 = num2deg(first.x, first.y, first.z)
        lat2, lon2 = num2deg(last.x + 1, last.y + 1, first.z)
        return [lon1, lat2, lon2, lat1]


class DirectoryWriter:
    format = 'PNG'
    quality = 100

    def __init__(self, folder, is_tms):
        self.folder = folder
        self.is_tms = is_tms

    def write_tile(self, tile, image, operation, layerTitle, layerAttr):
        directory = os.path.join(self.folder, layerTitle.lower(), layerAttr.lower(), operation, str(tile.z))
        os.makedirs(directory, exist_ok=True)
        xtile = '{0:04d}'.format(tile.x)
        ytile = '{0:04d}'.format(tile.y)
        filename = xtile + "_" + ytile + "." + self.format.lower()
        path = os.path.join(directory, filename)
        image.save(path, self.format, self.quality)
        return path

    def write_capabilities(self, layerTitle, extents, projection):
        WMSCapabilities.updateXML(self.folder, layerTitle, extents, projection)

    def write_description(self, layerTitle, layerAttr, cellTypeName, nullValue, operation):
        directory = os.path.join(self.folder, layerTitle.lower(), layerAttr.lower())
        filecsv = "description.csv"
        csvPath = os.path.join(directory, filecsv)
        jsonPath = os.path.join(directory, "description.json")
        if os.path.isfile(csvPath):
            csvFile = open(csvPath, "r", encoding="utf-8")
        else:
            csvFile = io.StringIO("Key*, cell, null, operation, attr,\n-9999, \"\", -1, \"\", nenhuma, ")

        csv_reader = csv.DictReader(csvFile, delimiter=',')
        line_count = 0
        elements = [cellTypeName, str(nullValue), operation, layerAttr]
        for row in csv_reader:
            curEntry = {
                'cellType': row[' cell'].strip(),
                'nullValue': row[' null'].strip(),
                'operation': row[' operation'].strip(),
                'attribute': row[' attr'].strip()
            }
            if line_count > 0 and curEntry['operation'] != operation and curEntry['attribute'] != layerAttr:
                # Key *, cell, null, operation, attr,
                elements.append(curEntry['cellType'])
                elements.append(curEntry['nullValue'])
                elements.append(curEntry['operation'])
                elements.append(curEntry['attribute'])
            line_count += 1
        with open(jsonPath, "w", encoding="utf-8") as jsonFile:
            jsonFile.write(json.dumps(elements))
        csvFile.close()

    '''
    Desenha o thumbnail na projeção final do projeto.
    '''
    def writeThumbnail(self, mapDestExtent, mapTitle, mapAttr, operation, renderSettings):
        renderSettings.setExtent(mapDestExtent)
        size = QSize(180, 180)
        renderSettings.setOutputSize(size)
        image = QImage(size, QImage.Format_ARGB32_Premultiplied)
        image.fill(Qt.transparent)
        dpm = renderSettings.outputDpi() / 25.4 * 1000
        image.setDotsPerMeterX(dpm)
        image.setDotsPerMeterY(dpm)
        painter = QPainter(image)
        job = QgsMapRendererCustomPainterJob(renderSettings, painter)
        job.renderSynchronously()
        painter.end()
        legendFolder = os.path.join(self.folder, mapTitle, mapAttr, operation)
        os.makedirs(legendFolder, exist_ok=True)
        image.save(os.path.join(legendFolder, 'thumbnail.png'), self.format, self.quality)

    def writeLegendPng(self, layer, mapTitle, mapAttr, operation):
        legendFolder = os.path.join(self.folder, mapTitle, mapAttr, operation)

        # e.g. vlayer = iface.activeLayer()
        options = QgsMapSettings()
        options.setLayers([layer])
        options.setBackgroundColor(QColor(255, 128, 255))
        options.setOutputSize(QSize(60, 60))
        options.setExtent(layer.extent())
        render = QgsMapRendererParallelJob(options)

        def finished():
            img = render.renderedImage()
            # save the image; e.g. img.save("/Users/myuser/render.png","png")
            img.save(os.path.join(legendFolder, "legend.png"), "png")
            print("saved")
        render.finished.connect(finished)
        render.start()

    def writeLegendJson(self, layer, mapAttr, operation):
        mapTitle = layer.name()
        result = []
        if isinstance(layer, QgsRasterLayer):
            for simbology in layer.legendSymbologyItems():
                label, color = simbology
                result.append({"color": [color.red(), color.green(), color.blue()], "title": label})
        elif layer.renderer().type() == 'categorizedSymbol':
            for symbology in layer.renderer().categories():
                label = symbology.label()
                color = symbology.symbol().color()
                result.append({"color": [color.red(), color.green(), color.blue()], "title": label})
        elif layer.renderer().type() == 'singleSymbol':
            color = layer.renderer().symbol().color()
            result.append({"color": [color.red(), color.green(), color.blue()], "title": mapTitle})
        elif layer.renderer().type() == 'RuleRenderer':
            for symbology in layer.renderer().legendSymbolItems():
                label = symbology.label()
                color = symbology.symbol().color()
                result.append({"color": [color.red(), color.green(), color.blue()], "title": label})
        jsonFile = Path(os.path.join(self.folder, mapTitle, mapAttr, operation, "legend.json"))
        jsonFile.write_text(json.dumps(result), encoding="utf-8")

    def close(self):
        pass


class GitHub:
    @staticmethod
    def getGitUrl(githubUser, githubRepository):
        return "https://github.com/" + githubUser + "/" + githubRepository + "/"

    #No password to allow using configured SSHKey.
    @staticmethod
    def getGitPassUrl(user, repository, password):
        return "https://" + user + ":" + password + "@github.com/" + user + "/" + repository + ".git"

    @staticmethod
    def lsremote(url):
        import git
        remote_refs = {}
        g = git.cmd.Git()
        for ref in g.ls_remote(url).split('\n'):
            hash_ref_list = ref.split('\t')
            remote_refs[hash_ref_list[1]] = hash_ref_list[0]
        return remote_refs

    @staticmethod
    def existsRepository(user, repository, feedback):
        try:
            feedback.pushConsoleInfo("URL : " + GitHub.getGitUrl(user, repository))
            resp = requests.get(GitHub.getGitUrl(user, repository))
            if resp.status_code == 200:
                return True
            else:
                return False
            #result = GitHub.lsremote(GitHub.getGitUrl(user, repository))
        except:
            return False

    @staticmethod
    def isOptionsOk(folder, gitExecutable, user, repository, feedback):
        gitProgramFolder = os.path.dirname(gitExecutable)
        # if not os.environ.get('GIT_PYTHON_GIT_EXECUTABLE', ''):
        feedback.pushConsoleInfo(gitProgramFolder)
        os.environ['GIT_PYTHON_GIT_EXECUTABLE'] = gitExecutable
        initialPath = os.environ['PATH']
        try:
            os.environ['PATH'].split(os.pathsep).index(gitProgramFolder)
        except:
            os.environ["PATH"] = os.environ["PATH"] + os.pathsep + gitProgramFolder

        install("gitpython")
        from git import Repo
        from git import InvalidGitRepositoryError

        #feedback.pushConsoleInfo('Github found commiting to your account.')
        if not GitHub.existsRepository(user, repository, feedback):
            feedback.pushConsoleInfo("The repository " + repository + " doesn't exists.\nPlease create a new one at https://github.com/new .")
            return False

        #Cria ou pega o repositório atual.
        repo = None
        if not os.path.exists(folder) or os.path.exists(folder) and not os.listdir(folder):
            repo = Repo.clone_from(GitHub.getGitUrl(user, repository), folder)
            assert (os.path.exists(folder))
        else:
            try:
                repo = Repo(folder)
                repoUrl = repo.git.remote("-v")
                expectedUrl = GitHub.getGitUrl(user, repository)
                if repoUrl and not (expectedUrl in re.compile("[\\n\\t ]").split(repoUrl)):
                    feedback.pushConsoleInfo("Your remote URL " + repoUrl + " does not match the expected url " + expectedUrl)
                    return False
            except InvalidGitRepositoryError:
                feedback.pushConsoleInfo("The destination folder must be a repository or an empty folder.")
                #repo = Repo.init(folder, bare=False)
                return False

        if repo.git.status("--porcelain"):
            feedback.pushConsoleInfo("Error: Local repository is not clean.\nPlease commit the changes made to local repository before run.\nUse: git add * and git commit -m \"MSG\"")
            return False
        return True

    @staticmethod
    def publishTilesToGitHub(folder, gitExecutable, user, repository, feedback, password=None):  # ghRepository, ghUser, ghPassphrase
        gitProgramFolder = os.path.dirname(gitExecutable)
        #if not os.environ.get('GIT_PYTHON_GIT_EXECUTABLE', ''):
        feedback.pushConsoleInfo(gitProgramFolder)
        print(gitProgramFolder)
        os.environ['GIT_PYTHON_GIT_EXECUTABLE'] = gitExecutable
        initialPath = os.environ['PATH']
        try:
            os.environ['PATH'].split(os.pathsep).index(gitProgramFolder)
        except:
            os.environ["PATH"] = os.environ["PATH"] + os.pathsep + gitProgramFolder

        install("gitpython")
        from git import Repo
        from git import InvalidGitRepositoryError

        feedback.pushConsoleInfo('Github found commiting to your account.')

        #Não está funcionando a validação
        feedback.pushConsoleInfo(user + ' at ' + repository)
        if not GitHub.existsRepository(user, repository, feedback):
            feedback.pushConsoleInfo("The repository " + repository + " doesn't exists.\nPlease create a new one at https://github.com/new .")
            return None

        #Cria ou pega o repositório atual.
        repo = None
        if not os.path.exists(folder) or os.path.exists(folder) and not os.listdir(folder):
            repo = Repo.clone_from(GitHub.getGitUrl(user, repository), folder)
            assert (os.path.exists(folder))
        else:
            try:
                repo = Repo(folder)
            except InvalidGitRepositoryError:
                feedback.pushConsoleInfo("The destination folder must be a repository or an empty folder.")
                repo = Repo.init(folder, bare=False)

        now = datetime.now()
        # https://stackoverflow.com/questions/6565357/git-push-requires-username-and-password
        repo.git.config("credential.helper", "store")
        repo.git.config("--global", "credential.helper", "'cache --timeout 7200'")
        try:
            feedback.pushConsoleInfo("Git: Pulling your repository current state.")
            repo.git.pull("-s recursive", "-X ours", GitHub.getGitUrl(user, repository), "master")
            feedback.pushConsoleInfo("Git: Doing checkout.")
            repo.git.fetch(GitHub.getGitUrl(user, repository), "master")
            feedback.pushConsoleInfo("Git: Doing checkout.")
            repo.git.checkout("--ours")
        except:
            pass

        originName = "mappia"

        try:
            repo.git.remote("add", originName, GitHub.getGitUrl(user, repository))
        except:
            repo.git.remote("set-url", originName, GitHub.getGitUrl(user, repository))

        feedback.pushConsoleInfo('Git: Add all generated tiles to your repository.')
        repo.git.add(all=True)  # Adiciona todos arquivos
        #feedback.pushConsoleInfo("Git: Mergin.")
        #repo.git.merge("-s recursive", "-X ours")
        #feedback.pushConsoleInfo("Git: Pushing changes.")
        #repo.git.push(GitHub.getGitUrl(user, repository), "master:refs/heads/master")
        if repo.index.diff(None) or repo.untracked_files:
            feedback.pushConsoleInfo("No changes, nothing to commit.")
        feedback.pushConsoleInfo("Git: Committing changes.")
        repo.git.commit(m="QGIS - " + now.strftime("%d/%m/%Y %H:%M:%S"))
        # feedback.pushConsoleInfo("CREATING TAG")
        # tag = now.strftime("%Y%m%d-%H%M%S")
        # new_tag = repo.create_tag(tag, message='Automatic tag "{0}"'.format(tag))
        # repo.remotes[originName].push(new_tag)
        feedback.pushConsoleInfo("Git: Pushing modifications to remote repository.")
        repo.git.push(GitHub.getGitPassUrl(user, repository, password), "master:refs/heads/master")
        os.environ['PATH'] = initialPath
        return None


#Supported Operations
class OperationType(Enum):
    RAW = 0
    INTEGRAL = 1
    AREAINTEGRAL = 2
    MAX = 3
    AVERAGE = 4
    MIN = 5
    AREA = 6
    SUM = 7
    CELLS = 8
    RGBA = 9

    @staticmethod
    def getOptions():
        return [member for member in OperationType.__members__]

    def getName(self):
        return self.name.lower()

    def __str__(self):
        return self.name

class MappiaPublisherAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    LAYERS = 'LAYERS'
    OPERATION = 'OPERATION'
    OUTPUT_DIRECTORY = 'OUTPUT_DIRECTORY'
    ZOOM_MAX = 'ZOOM_MAX'
    EXTENT = 'EXTENT'
    LAYER_ATTRIBUTE = 'LAYER_ATTRIBUTE'

    GITHUB_KEY = 'GITHUB_KEY'
    GITHUB_REPOSITORY = 'GITHUB_REPOSITORY'
    GITHUB_USER = 'GITHUB_USER'
    GITHUB_PASS = 'GITHUB_PASS'
    GIT_EXECUTABLE = 'GIT_EXECUTABLE'

    # Default size of the WMS tiles.
    WIDTH = 256
    HEIGHT = 256

    def initAlgorithm(self, config):

        #super(MappiaPublisherAlgorithm, self).initAlgorithm()
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        options = OptionsCfg.read()

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.LAYERS,
                self.tr('Maps to publish'),
                QgsProcessing.TypeMapLayer
            )
        )


        self.addParameter(
            QgsProcessingParameterEnum(
                self.OPERATION,
                self.tr('Operation Type'),
                options=[self.tr(curOption) for curOption in OperationType.getOptions()],
                defaultValue=9
            )
        )

        self.addParameter(
            QgsProcessingParameterFolderDestination(
                self.OUTPUT_DIRECTORY,
                self.tr('Output directory'),
                optional=False,
                defaultValue=options["folder"]
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.ZOOM_MAX,
                self.tr('Maximum zoom [1 ~ 13]'),
                minValue=1,
                maxValue=13,
                defaultValue=options["zoom_max"]
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.LAYER_ATTRIBUTE,
                self.tr('Layer style name'),
                optional=False,
                defaultValue=options['attrName']
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.GIT_EXECUTABLE,
                self.tr('Executable git.exe'),
                optional=True,
                defaultValue=options['git_exe']
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.GITHUB_USER,
                self.tr('Github Username'),
                optional=True,
                defaultValue=options['gh_user']
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.GITHUB_PASS,
                self.tr('Github Password'),
                optional=True
            )
        )

        self.addParameter(
            QgsProcessingParameterString(
                self.GITHUB_REPOSITORY,
                self.tr('Github Repository'),
                optional=True,
                defaultValue=options['gh_repository']
            )
        )

    #Create the metatiles to the given layer in given zoom levels
    def createLayerMetatiles(self, projection, layer, minZoom, maxZoom):
        mapExtentReprojected = self.getMapExtent(layer, projection)
        metatiles_by_zoom = {}
        metatilesize = 4
        for zoom in range(minZoom, maxZoom + 1):
            metatiles = get_metatiles(mapExtentReprojected, zoom, metatilesize)
            metatiles_by_zoom[zoom] = metatiles
        return metatiles_by_zoom

    #Return the map extents in the given projection
    def getMapExtent(self, layer, projection):
        mapExtent = layer.extent()
        src_to_proj = QgsCoordinateTransform(layer.crs(), projection, layer.transformContext())
        return src_to_proj.transformBoundingBox(mapExtent)

    def generate(self, writer, parameters, context, feedback):
        feedback.setProgress(1)
        min_zoom = 0
        max_zoom = self.parameterAsInt(parameters, self.ZOOM_MAX, context)
        outputFormat = QImage.Format_ARGB32
        layerAttr = self.parameterAsString(parameters, self.LAYER_ATTRIBUTE, context)
        cellType = 'int32'
        nullValue = -128
        #ghKey = self.parameterAsString(parameters, self.GITHUB_KEY, context)
        ghUser = self.parameterAsString(parameters, self.GITHUB_USER, context)
        ghRepository = self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context)
        ghPassword = self.parameterAsString(parameters, self.GITHUB_PASS, context)
        gitExecutable = self.parameterAsString(parameters, self.GIT_EXECUTABLE, context)
        mapOperation = OperationType(self.parameterAsEnum(parameters, self.OPERATION, context))
        feedback.setProgressText("Operation: " + str(mapOperation))
        wgs_crs = QgsCoordinateReferenceSystem('EPSG:4326')
        dest_crs = QgsCoordinateReferenceSystem('EPSG:3857')
        layers = self.parameterAsLayerList(parameters, self.LAYERS, context)
        metatilesCount = sum([len(self.createLayerMetatiles(wgs_crs, layer, min_zoom, max_zoom)) for layer in layers])
        progress = 0
        if gitExecutable and not GitHub.isOptionsOk(writer.folder, gitExecutable, ghUser, ghRepository, feedback):
            feedback.setProgressText("Error : Invalid options.")
            return False
        for layer in layers:
            feedback.setProgressText("Publishing map: " + layer.name())
            layerRenderSettings = self.createLayerRenderSettings(layer, dest_crs, outputFormat)
            metatiles_by_zoom = self.createLayerMetatiles(wgs_crs, layer, min_zoom, max_zoom)
            for zoom in range(min_zoom, max_zoom + 1):
                feedback.pushConsoleInfo('Generating tiles for zoom level: %s' % zoom)
                for i, metatile in enumerate(metatiles_by_zoom[zoom]):
                    if feedback.isCanceled():
                        break
                    transformContext = context.transformContext()
                    mapRendered = self.renderMetatile(metatile, dest_crs, outputFormat, layerRenderSettings,
                                                      transformContext, wgs_crs)
                    for r, c, tile in metatile.tiles:
                        tile_img = mapRendered.copy(self.WIDTH * r, self.HEIGHT * c, self.WIDTH, self.HEIGHT)
                        writer.write_tile(tile, tile_img, mapOperation.getName(), layer.name(), layerAttr)
                    progress += 1
                    feedback.setProgress(100 * (progress / metatilesCount))
            writer.writeLegendPng(layer, layer.name(), layerAttr, mapOperation.getName())
            writer.write_description(layer.name(), layerAttr, cellType, nullValue, mapOperation.getName())
            writer.write_capabilities(layer.name(), self.getMapExtent(layer, wgs_crs), layer.crs().toWkt())
            writer.writeLegendJson(layer, layerAttr, mapOperation.getName())
            writer.writeThumbnail(self.getMapExtent(layer, dest_crs), layer.name(), layerAttr, mapOperation.getName(), layerRenderSettings)
        feedback.pushConsoleInfo('Finished map tile generation.')
        if gitExecutable:
            createdTag = GitHub.publishTilesToGitHub(writer.folder, gitExecutable, ghUser, ghRepository, feedback, ghPassword)
            giturl = GitHub.getGitUrl(ghUser, ghRepository)
            #FIXME Space in the map name will cause errors.
            storeUrl = giturl #+ "@" + createdTag + "/"
            feedback.pushConsoleInfo(
                "View the results online: \nhttps://maps.csr.ufmg.br/calculator/?queryid=152&storeurl=" + storeUrl
                + "/&zoomlevels="+str(max_zoom)+"&remotemap=" + ",".join(["GH:" + layer.name().lower() + ";" + layerAttr for layer in layers]))
        else:
            feedback.pushConsoleInfo("View the results online: \nhttps://maps.csr.ufmg.br/calculator/?queryid=152&remotemap="
                                     + ",".join(["GH:" + layer.name().lower() + ";" + layerAttr for layer in layers]))
        writer.close()

    #Return the rendered map (QImage) for the metatile zoom level.
    def renderMetatile(self, metatile, dest_crs, outputFormat, renderSettings, transformContext, sourceCrs):
        wgs_to_dest = QgsCoordinateTransform(sourceCrs, dest_crs, transformContext)
        renderSettings.setExtent(wgs_to_dest.transformBoundingBox(QgsRectangle(*metatile.extent())))
        size = QSize(self.WIDTH * metatile.rows(), self.HEIGHT * metatile.columns())
        renderSettings.setOutputSize(size)
        image = QImage(size, outputFormat)
        image.fill(Qt.transparent)
        dpm = renderSettings.outputDpi() / 25.4 * 1000
        image.setDotsPerMeterX(dpm)
        image.setDotsPerMeterY(dpm)
        painter = QPainter(image)
        job = QgsMapRendererCustomPainterJob(renderSettings, painter)
        job.renderSynchronously()
        painter.end()
        return image

    #Configure the rendering settings for the WMS tiles.
    def createLayerRenderSettings(self, layer, dest_crs, outputFormat):
        settings = QgsMapSettings()
        settings.setFlag(QgsMapSettings.Flag.Antialiasing, on=False)
        settings.setFlag(QgsMapSettings.Flag.UseRenderingOptimization, on=False)
        settings.setFlag(QgsMapSettings.Flag.UseAdvancedEffects, on=False)
        settings.setOutputImageFormat(outputFormat)
        settings.setDestinationCrs(dest_crs)
        simplifyMethod = QgsVectorSimplifyMethod()
        simplifyMethod.setSimplifyHints(QgsVectorSimplifyMethod.SimplifyHint.NoSimplification)
        settings.setSimplifyMethod(simplifyMethod)
        settings.setLayers([layer])
        dpi = 256
        settings.setOutputDpi(dpi)
        canvas_red = QgsProject.instance().readNumEntry('Gui', '/CanvasColorRedPart', 255)[0]
        canvas_green = QgsProject.instance().readNumEntry('Gui', '/CanvasColorGreenPart', 255)[0]
        canvas_blue = QgsProject.instance().readNumEntry('Gui', '/CanvasColorBluePart', 255)[0]
        color = QColor(canvas_red, canvas_green, canvas_blue, 0)
        settings.setBackgroundColor(color)
        labeling_engine_settings = settings.labelingEngineSettings()
        labeling_engine_settings.setFlag(QgsLabelingEngineSettings.UsePartialCandidates, False)
        settings.setLabelingEngineSettings(labeling_engine_settings)
        try:
            layer.resampleFilter().setZoomedInResampler(None)
            layer.resampleFilter().setZoomedOutResampler(None)
            layer.resampleFilter().setOn(False)
        except:
            pass #Is not a raster
        return settings

    def checkParameterValues(self, parameters, context):
        min_zoom = 0
        max_zoom = self.parameterAsInt(parameters, self.ZOOM_MAX, context)
        if max_zoom < min_zoom:
            return False, self.tr('Invalid zoom levels range.')
        if len(self.parameterAsLayerList(parameters, self.LAYERS, context)) <= 0:
            return False, self.tr("Please select one map at least.")
        return super().checkParameterValues(parameters, context)

    def prepareAlgorithm(self, parameters, context, feedback):
        project = context.project()
        visible_layers = [item.layer() for item in project.layerTreeRoot().findLayers() if item.isVisible()]
        self.layers = [l for l in project.layerTreeRoot().layerOrder() if l in visible_layers]
        OptionsCfg.write(
            self.parameterAsInt(parameters, self.ZOOM_MAX, context),
            self.parameterAsString(parameters, self.GIT_EXECUTABLE, context),
            self.parameterAsString(parameters, self.LAYER_ATTRIBUTE, context),
            self.parameterAsString(parameters, self.GITHUB_USER, context),
            self.parameterAsString(parameters, self.GITHUB_REPOSITORY, context),
            self.parameterAsString(parameters, self.OUTPUT_DIRECTORY, context)
        )
        return True

    #Danilo
    def processAlgorithm(self, parameters, context, feedback):
        is_tms = False
        output_dir = self.parameterAsString(parameters, self.OUTPUT_DIRECTORY, context)
        if not output_dir:
            raise QgsProcessingException(self.tr('You need to specify output directory.'))
        writer = DirectoryWriter(output_dir, is_tms)
        self.generate(writer, parameters, context, feedback)
        results = {'OUTPUT_DIRECTORY': output_dir}
        return results


    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Web Publisher'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Share maps'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return MappiaPublisherAlgorithm()

def install(package):
    if hasattr(pip, 'main'):
        pip.main(['install', '--user', package])
    else:
        from pip._internal import main as pip_main
        if hasattr(pip_main, 'main'):
            pip_main.main(['install', '--user', package])
        else:
            pip_main(['install', '--user', package])

try:
    import xmltodict
except:
    install("xmltodict")
    import xmltodict

class WMSCapabilities:

    @staticmethod
    def getDefaultCapabilities():
        return '''<?xml version="1.0" encoding="UTF-8"?><!DOCTYPE WMT_MS_Capabilities SYSTEM "http://maps.csr.ufmg.br:80/geoserver/schemas/wms/1.1.1/WMS_MS_Capabilities.dtd"[
    <!ELEMENT VendorSpecificCapabilities (TileSet*) >
    <!ELEMENT TileSet (SRS, BoundingBox?, Resolutions, Width, Height, Format, Layers*, Styles*) >
    <!ELEMENT Resolutions (#PCDATA) >
    <!ELEMENT Width (#PCDATA) >
    <!ELEMENT Height (#PCDATA) >
    <!ELEMENT Layers (#PCDATA) >
    <!ELEMENT Styles (#PCDATA) >
    ]>
    <WMT_MS_Capabilities version="1.1.1" updateSequence="63258">
      <Service>
        <Name>OGC:WMS</Name>
        <Title>CSR Web Map Service</Title>
        <Abstract>Compliant WMS Response</Abstract>
        <KeywordList>
          <Keyword>WFS</Keyword>
          <Keyword>WMS</Keyword>
          <Keyword>GEOSERVER</Keyword>
        </KeywordList>
        <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://geoserver.sourceforge.net/html/index.php"/>
        <ContactInformation>
          <ContactPersonPrimary>
            <ContactPerson>GITHUB owner</ContactPerson>
            <ContactOrganization>GITHUB owner</ContactOrganization>
          </ContactPersonPrimary>
          <ContactPosition>Chief geographer</ContactPosition>
          <ContactAddress>
            <AddressType>Work</AddressType>
            <Address/>
            <City>Alexandria</City>
            <StateOrProvince/>
            <PostCode/>
            <Country>Egypt</Country>
          </ContactAddress>
          <ContactVoiceTelephone/>
          <ContactFacsimileTelephone/>
          <ContactElectronicMailAddress>claudius.ptolomaeus@gmail.com</ContactElectronicMailAddress>
        </ContactInformation>
        <Fees>NONE</Fees>
        <AccessConstraints>NONE</AccessConstraints>
      </Service>
      <Capability>
        <Request>
          <GetCapabilities>
            <Format>application/vnd.ogc.wms_xml</Format>
            <DCPType>
              <HTTP>
                <Get>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Get>
                <Post>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Post>
              </HTTP>
            </DCPType>
          </GetCapabilities>
          <GetMap>
            <Format>image/png</Format>
            <DCPType>
              <HTTP>
                <Get>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Get>
              </HTTP>
            </DCPType>
          </GetMap>
          <GetFeatureInfo>
            <Format>text/plain</Format>
            <Format>application/vnd.ogc.gml</Format>
            <Format>application/vnd.ogc.gml/3.1.1</Format>
            <Format>text/html</Format>
            <Format>application/json</Format>
            <DCPType>
              <HTTP>
                <Get>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Get>
                <Post>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Post>
              </HTTP>
            </DCPType>
          </GetFeatureInfo>
          <DescribeLayer>
            <Format>application/vnd.ogc.wms_xml</Format>
            <DCPType>
              <HTTP>
                <Get>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Get>
              </HTTP>
            </DCPType>
          </DescribeLayer>
          <GetLegendGraphic>
            <Format>image/png</Format>
            <DCPType>
              <HTTP>
                <Get>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Get>
              </HTTP>
            </DCPType>
          </GetLegendGraphic>
          <GetStyles>
            <Format>application/vnd.ogc.sld+xml</Format>
            <DCPType>
              <HTTP>
                <Get>
                  <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="http://maps.csr.ufmg.br:80/geoserver/wms?SERVICE=WMS&amp;"/>
                </Get>
              </HTTP>
            </DCPType>
          </GetStyles>
        </Request>
        <Exception>
          <Format>application/vnd.ogc.se_xml</Format>
          <Format>application/vnd.ogc.se_inimage</Format>
        </Exception>
        <VendorSpecificCapabilities>
        </VendorSpecificCapabilities>
        <UserDefinedSymbolization SupportSLD="1" UserLayer="1" UserStyle="1" RemoteWFS="1"/>
        <Layer queryable="0" opaque="0" noSubsets="0">
          <Title>MapServer via GItHub</Title>
          <Abstract>GitHub WMS</Abstract>
          <!--All supported EPSG projections:-->
          <SRS>EPSG:3857</SRS>
          <SRS>EPSG:4326</SRS>
          <SRS>EPSG:900913</SRS>
          <SRS>EPSG:42303</SRS>
          <LatLonBoundingBox minx="-180.00004074199998" miny="-90.0" maxx="180.0000000000001" maxy="90.0"/>
        </Layer>
      </Capability>
    </WMT_MS_Capabilities>'''

    @staticmethod
    def convertCoordinateProj(crsProj, fromX, fromY, outputProjected):
        # dinamica.package("pyproj")
        try:
            from pyproj import Proj
        except:
            install("pyproj")
            from pyproj import Proj
        import re

        regex = r"^[ ]*PROJCS"
        isProjected = re.match(regex, crsProj)
        if (outputProjected and isProjected) or (not outputProjected and not isProjected):
            return (fromX, fromY)
        else:
            x, y = Proj(crsProj)(fromX, fromY, inverse= not outputProjected)
            return (x, y)

    @staticmethod
    def getMapDescription(fileNameId, latMinX, latMinY, latMaxX, latMaxY, projMinX, projMinY, projMaxX, projMaxY):
        # Inverti o minx/miny e maxX/maxY do epsg 4326 pq estava trocando no geoserver, mas n sei pq isso acontece.
        return """<CONTENT>
          <Layer queryable="1" opaque="0">
          <Name>GH:""" + fileNameId + """</Name>
          <Title>GH:""" + fileNameId + """</Title>
          <Abstract>GH:""" + fileNameId + """ ABSTRACT</Abstract>
          <KeywordList>
            <Keyword>GITHUB</Keyword>
          </KeywordList>
          <SRS>EPSG:4326</SRS>
          <LatLonBoundingBox minx=\"""" + str(latMinX) + """\" miny=\"""" + str(latMinY) + """\" maxx=\"""" + str(
            latMaxX) + """\" maxy=\"""" + str(latMaxY) + """\"/>
          <BoundingBox SRS="EPSG:4326" minx=\"""" + str(projMinX) + """\" miny=\"""" + str(
            projMinY) + """\" maxx=\"""" + str(projMaxX) + """\" maxy=\"""" + str(projMaxY) + """\"/>
          <Style>
           <Name>""" + fileNameId + """_1</Name>
           <Title>""" + fileNameId + """_1</Title>
           <Abstract/>
           <LegendURL width="20" height="20">
           <Format>image/png</Format>
           <OnlineResource xmlns:xlink="http://www.w3.org/1999/xlink" xlink:type="simple" xlink:href="https://maps.csr.ufmg.br:443/geoserver/wms?request=GetLegendGraphic&amp;format=image%2Fpng&amp;width=20&amp;height=20&amp;layer=""" + fileNameId + """\"/>
           </LegendURL>
          </Style>
        </Layer>
        </CONTENT>
        """

    @staticmethod
    def getTileSetDescription(fileNameId, latMinX, latMinY, latMaxX, latMaxY, projMinX, projMinY, projMaxX, projMaxY):
        # <BoundingBox SRS="EPSG:4326" minx=\"""" + str(latMinX) + """\" miny=\"""" + str(latMinY) + """\" maxx=\"""" + str(latMaxX) + """\" maxy=\"""" + str(latMaxY) + """\"/>
        return """<?xml version=\"1.0\" encoding=\"UTF-8\"?>
        <CONTENT>
          <TileSet>
            <SRS>EPSG:4326</SRS>
            <BoundingBox SRS="EPSG:4326" minx="-180.0" miny="-90.0" maxx="0.0" maxy="90.0"/>
        <Resolutions>0.703125 0.3515625 0.17578125 0.087890625 0.0439453125 0.02197265625 0.010986328125 0.0054931640625 0.00274658203125 0.001373291015625 6.866455078125E-4 3.4332275390625E-4 1.71661376953125E-4 8.58306884765625E-5 4.291534423828125E-5 2.1457672119140625E-5 1.0728836059570312E-5 5.364418029785156E-6 2.682209014892578E-6 1.341104507446289E-6 6.705522537231445E-7 3.3527612686157227E-7 </Resolutions>
            <Width>256</Width>
            <Height>256</Height>
            <Format>image/png</Format>
            <Layers>GH:""" + fileNameId + """</Layers>
            <Styles/>
          </TileSet>
          <TileSet>
            <SRS>EPSG:900913</SRS>
            <BoundingBox SRS="EPSG:900913" minx="-2.003750834E7" miny="-2.003750834E7" maxx="2.003750834E7" maxy="2.003750834E7"/>
        <Resolutions>156543.03390625 78271.516953125 39135.7584765625 19567.87923828125 9783.939619140625 4891.9698095703125 2445.9849047851562 1222.9924523925781 611.4962261962891 305.74811309814453 152.87405654907226 76.43702827453613 38.218514137268066 19.109257068634033 9.554628534317017 4.777314267158508 2.388657133579254 1.194328566789627 0.5971642833948135 0.29858214169740677 0.14929107084870338 0.07464553542435169 0.037322767712175846 0.018661383856087923 0.009330691928043961 0.004665345964021981 0.0023326729820109904 0.0011663364910054952 5.831682455027476E-4 2.915841227513738E-4 1.457920613756869E-4 </Resolutions>
            <Width>256</Width>
            <Height>256</Height>
            <Format>image/png</Format>
            <Layers>GH:""" + fileNameId + """</Layers>
            <Styles/>
          </TileSet>
          </CONTENT>
        """
        # <BoundingBox SRS="EPSG:900913" minx=\"""" + str(projMinX) + """\" miny=\"""" + str(projMinY) + """\" maxx=\"""" + str(projMaxX) + """\" maxy=\"""" + str(projMaxY) + """\"/>

    @staticmethod
    def updateXML(directory, layerTitle, extents, projection):

        xmlContent = ""
        capabilitiesPath = os.path.join(directory, "getCapabilities.xml")
        if os.path.isfile(capabilitiesPath):
            capabilitiesContent = Path(capabilitiesPath).read_text()
        else:
            capabilitiesContent = WMSCapabilities.getDefaultCapabilities()
        doc = xmltodict.parse(capabilitiesContent)

        filename = layerTitle #Danilo é layer Name no qgis

        #extents, projection
        latMaxX, latMaxY = WMSCapabilities.convertCoordinateProj(projection, extents.xMaximum(), extents.yMaximum(), outputProjected=True)
        latMinX, latMinY = WMSCapabilities.convertCoordinateProj(projection, extents.xMinimum(), extents.yMinimum(), outputProjected=True)
        projMaxX, projMaxY = WMSCapabilities.convertCoordinateProj(projection, extents.xMaximum(), extents.yMaximum(), outputProjected=False)
        projMinX, projMinY = WMSCapabilities.convertCoordinateProj(projection, extents.xMaximum(), extents.yMaximum(), outputProjected=False)

        if 'Layer' in doc['WMT_MS_Capabilities']['Capability']['Layer']:
            if type(doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer']) is collections.OrderedDict:
                curLayer = doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer']
                doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer'] = [curLayer]

        if 'Layer' in doc['WMT_MS_Capabilities']['Capability']['Layer']:
            for iLayer in range(len(doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer']) - 1, -1, -1):
                curLayer = doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer'][iLayer]
                if "Name" in curLayer and curLayer["Name"] == "GH:" + filename:
                    doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer'].pop(iLayer)

        newLayerDescription = xmltodict.parse(
            WMSCapabilities.getMapDescription(filename, latMinX, latMinY, latMaxX, latMaxY, projMinX, projMinY,
                                              projMaxX, projMaxY))['CONTENT']
        if 'Layer' in doc['WMT_MS_Capabilities']['Capability']['Layer']:
            doc['WMT_MS_Capabilities']['Capability']['Layer']['Layer'].append(newLayerDescription['Layer'])
        else:
            doc['WMT_MS_Capabilities']['Capability']['Layer'] = newLayerDescription

        newTileSetDescription = xmltodict.parse(
            WMSCapabilities.getTileSetDescription(filename, latMinX, latMinY, latMaxX, latMaxY, projMinX, projMinY,
                                                  projMaxX, projMaxY))['CONTENT']

        if doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities'] is not None and 'TileSet' in \
                doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']:
            if type(doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities'][
                        'TileSet']) is collections.OrderedDict:
                curTileSet = doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']['TileSet']
                doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']['TileSet'] = [
                    curTileSet]  # Transforms into a list

        if doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities'] is not None and 'TileSet' in \
                doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']:
            for iTileSet in range(
                    len(doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']['TileSet']) - 1,
                    -1, -1):
                curTileSet = doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']['TileSet'][iTileSet]
                if "Layers" in curTileSet and curTileSet["Layers"] == "GH:" + filename:
                    doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']['TileSet'].pop(iTileSet)
            doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities']['TileSet'] += newTileSetDescription[
                'TileSet']  # Dois elementos c mesma chave representa com lista
        else:
            doc['WMT_MS_Capabilities']['Capability']['VendorSpecificCapabilities'] = newTileSetDescription
        Path(capabilitiesPath).write_text(xmltodict.unparse(doc, pretty=True))
