"""
Test script for the stretcher position in the folder selection area.

This script tests that the stretcher is correctly positioned at the end of the layout,
ensuring that folder items are pushed to the top of the scroll area.

To run this test:
1. Open QGIS
2. Load some layers that are not connected to MapHub
3. Run this script from the QGIS Python console
"""

from qgis.utils import iface
from PyQt5.QtWidgets import QDialog, QVBoxLayout

# Import the dialog
from ui.dialogs.BatchConnectLayersDialog import BatchConnectLayersDialog

def test_stretcher_position():
    """Test that the stretcher is correctly positioned at the end of the layout."""
    # Get all layers from the current project
    layers = iface.mapCanvas().layers()
    
    # Filter for layers that are not connected to MapHub
    not_connected_layers = [layer for layer in layers if layer.customProperty("maphub/map_id") is None]
    
    if not not_connected_layers:
        print("No unconnected layers found. Please add some layers to the project that are not connected to MapHub.")
        return
    
    # Create and show the BatchConnectLayersDialog
    dialog = BatchConnectLayersDialog(iface, not_connected_layers)
    
    # Get the ProjectNavigationWidget from the dialog
    workspace_nav_widget = dialog.workspace_nav_widget
    project_nav_widget = workspace_nav_widget.project_nav_widget
    
    # Check the layout structure
    list_layout = project_nav_widget.list_layout
    item_count = list_layout.count()
    
    print(f"Total items in layout: {item_count}")
    
    # Check if the last item is a stretch
    if item_count > 0:
        last_item = list_layout.itemAt(item_count - 1)
        is_stretch = last_item.spacerItem() is not None
        print(f"Is the last item a stretch: {is_stretch}")
    else:
        print("No items in layout")
    
    # Show the dialog for manual verification
    dialog.exec_()
    
    print("Test completed. Please verify that:")
    print("1. Folder items are aligned at the top of the scroll area")
    print("2. There is empty space at the bottom of the scroll area")
    print("3. The items don't stretch to fill the entire height")

if __name__ == "__main__":
    test_stretcher_position()