"""
Test script for the Create Folder functionality in the ProjectNavigationWidget.

This script tests the new Create Folder button in the ProjectNavigationWidget by:
1. Opening the BatchConnectLayersDialog which contains the ProjectNavigationWidget
2. Verifying the Create Folder button is present and functional

IMPORTANT: This script must be run from the QGIS Python console, not directly from the command line.

To run this test:
1. Open QGIS
2. Load some layers that are not connected to MapHub
3. Open the Python Console in QGIS (Plugins > Python Console)
4. In the Python Console, run:
   exec(open('/path/to/test_create_folder.py').read())
"""

from qgis.utils import iface
from PyQt5.QtWidgets import QDialog, QPushButton

# Import the dialog
from ui.dialogs.BatchConnectLayersDialog import BatchConnectLayersDialog

def test_create_folder():
    """Test the Create Folder functionality in the ProjectNavigationWidget."""
    # Get all layers from the current project
    layers = iface.mapCanvas().layers()
    
    # Filter for layers that are not connected to MapHub
    not_connected_layers = [layer for layer in layers if layer.customProperty("maphub/map_id") is None]
    
    if not not_connected_layers:
        print("No unconnected layers found. Please add some layers to the project that are not connected to MapHub.")
        return
    
    # Create the BatchConnectLayersDialog
    dialog = BatchConnectLayersDialog(iface, not_connected_layers)
    
    # Get the ProjectNavigationWidget from the dialog
    workspace_nav_widget = dialog.workspace_nav_widget
    project_nav_widget = workspace_nav_widget.project_nav_widget
    
    # Find the Create Folder button in the navigation controls
    create_folder_button = None
    for i in range(project_nav_widget.list_layout.count()):
        item = project_nav_widget.list_layout.itemAt(i)
        widget = item.widget()
        if widget and widget.objectName() == "navigationFrame":
            # Search for the Create Folder button in the navigation frame
            for child in widget.children():
                if isinstance(child, QPushButton) and child.text() == "Create Folder":
                    create_folder_button = child
                    break
    
    if create_folder_button:
        print("Create Folder button found in the navigation controls.")
    else:
        print("ERROR: Create Folder button not found in the navigation controls.")
        return
    
    # Show the dialog for manual testing
    print("\nTest instructions:")
    print("1. The dialog will open showing the ProjectNavigationWidget")
    print("2. Verify that the Create Folder button is visible in the navigation controls")
    print("3. Click the Create Folder button")
    print("4. Enter a name for the new folder in the dialog that appears")
    print("5. Click OK to create the folder")
    print("6. Verify that the new folder appears in the folder list")
    
    dialog.exec_()
    
    print("\nTest completed.")

if __name__ == "__main__":
    test_create_folder()