# Shapefile Validation in MapHub QGIS Plugin

## Overview

This document describes the implementation of shapefile validation in the MapHub QGIS Plugin. The validation ensures that all required components of a shapefile are present before attempting to upload or synchronize the shapefile with MapHub.

## Background

A shapefile consists of multiple files with the same base name but different extensions. At minimum, a valid shapefile requires:
- `.shp` - The main file containing the geometry
- `.dbf` - The attribute data
- `.shx` - The shape index file

Other files like `.prj` (projection information), `.qpj` (QGIS projection), and `.cpg` (code page) are optional but useful.

## Implementation

The validation has been implemented in three key locations:

### 1. MapsEndpoint.upload_map() in maphub/endpoints/maps.py

This method handles the direct upload of maps to MapHub. The validation:
- Checks if the file being uploaded is a shapefile-related file
- Verifies that all required components (.shp, .dbf, .shx) exist
- Raises a ValueError with a descriptive message if any required components are missing
- Ensures at least one file with the base name exists before attempting to zip

```python
# Check if all required shapefile components exist
required_extensions = ['.shp', '.dbf', '.shx']
missing_extensions = []

for ext in required_extensions:
    if not (file_dir / f"{file_base}{ext}").exists():
        missing_extensions.append(ext)

if missing_extensions:
    raise ValueError(f"Missing required shapefile components: {', '.join(missing_extensions)}. "
                     f"A valid shapefile must include .shp, .dbf, and .shx files.")
```

### 2. MapHubSyncManager.add_layer() in utils/sync_manager.py

This method handles adding a layer to MapHub through the synchronization interface. The validation:
- Checks if the layer being added is a shapefile
- Verifies that all required components (.dbf, .shx) exist
- Displays an error message in the QGIS interface if any required components are missing
- Raises an Exception with a descriptive message

```python
# Check if all required shapefile components exist
required_extensions = ['.dbf', '.shx']
missing_extensions = []

for ext in required_extensions:
    related_file = f"{base_name}{ext}"
    if not os.path.exists(related_file):
        missing_extensions.append(ext)

if missing_extensions:
    error_msg = f"Missing required shapefile components: {', '.join(missing_extensions)}. A valid shapefile must include .shp, .dbf, and .shx files."
    if self.iface:
        self.iface.messageBar().pushCritical("MapHub", error_msg)
    raise Exception(error_msg)
```

### 3. MapHubSyncManager.synchronize_layer() in utils/sync_manager.py

This method handles synchronizing a layer with MapHub. The validation:
- Checks if the layer being synchronized is a shapefile
- Verifies that all required components (.dbf, .shx) exist
- Displays an error message in the QGIS interface if any required components are missing
- Raises an Exception with a descriptive message

## User Experience

When a user attempts to upload or synchronize an incomplete shapefile:

1. The operation is halted before any data is sent to the server
2. A clear error message is displayed indicating which components are missing
3. The user is informed that a valid shapefile must include .shp, .dbf, and .shx files

This prevents errors on the server side and provides immediate feedback to the user about the issue.

## Testing

The implementation should be tested with various scenarios:
- Complete shapefiles with all required components
- Incomplete shapefiles missing one or more required components
- Shapefiles with optional components missing
- Non-shapefile formats to ensure they're not affected

## Future Enhancements

Potential future enhancements could include:
- Automatic repair of shapefiles (where possible)
- More detailed validation of shapefile content
- Support for other multi-file formats