import os
import logging
import asyncio

from PyQt5.QtCore import Qt, pyqtSignal, QTimer, QMimeData, QByteArray, QDataStream, QIODevice
from PyQt5.QtWidgets import (QDockWidget, QWidget, QVBoxLayout, QHBoxLayout, QTreeWidget, 
                            QMenu, QAction, QMessageBox, QToolButton)
from PyQt5.QtGui import QIcon, QDrag

from ...utils.utils import get_maphub_client
from ...utils.map_operations import download_map, add_map_as_tiling_service, add_folder_maps_as_tiling_services, download_folder_maps, load_and_sync_folder
from ...utils.sync_manager import MapHubSyncManager
from ...utils.project_utils import get_project_folder_id
from .MapItemDelegate import MapItemDelegate, STATUS_INDICATOR_ROLE, PROJECT_FOLDER_ROLE
from ...utils.error_manager import handled_exceptions, ensure_api_key
from ...ui.dialogs.SynchronizeLayersDialog import SynchronizeLayersDialog
from .TreeNode import TreeNode, WorkspaceNode, FolderNode, MapNode, SortableTreeWidgetItem


class MapBrowserTreeWidget(QTreeWidget):
    """
    Custom tree widget for the MapHub browser that supports drag and drop.
    """
    MIME_TYPE = "application/x-maphub-item"
    
    def __init__(self, parent=None, icon_dir=None):
        super(MapBrowserTreeWidget, self).__init__(parent)
        self.icon_dir = icon_dir
        self.setDragEnabled(True)
        self.setHeaderHidden(True)
        self.setSortingEnabled(True)
        self.sortByColumn(0, Qt.AscendingOrder)
        
    def startDrag(self, supportedActions):
        """
        Start drag operation for the selected item.
        """
        item = self.currentItem()
        if not item:
            return
            
        # Get item data
        item_data = item.data(0, Qt.UserRole)
        if not item_data:
            return
            
        item_type = item_data.get('type')
        item_id = item_data.get('id')
        
        # Only allow dragging maps and folders
        if item_type not in ['map', 'folder']:
            return
            
        # Create mime data
        mime_data = QMimeData()
        encoded_data = QByteArray()
        stream = QDataStream(encoded_data, QIODevice.WriteOnly)
        
        # Write item type and ID
        stream.writeQString(item_type)
        stream.writeQString(item_id)
        
        # For maps, also write additional data
        if item_type == 'map':
            stream.writeQString(item_id)  # map_id
            stream.writeQString(item.text(0))  # map_name
            
            # Get map type (vector or raster)
            map_type = 'vector'
            if item_data.get('file_format') in ['tif', 'tiff', 'png', 'jpg', 'jpeg']:
                map_type = 'raster'
            stream.writeQString(map_type)
            
            # Get folder ID
            folder_id = item_data.get('folder_id', '')
            stream.writeQString(folder_id)
            
        # Set mime data
        mime_data.setData(self.MIME_TYPE, encoded_data)
        
        # Create drag object
        drag = QDrag(self)
        drag.setMimeData(mime_data)
        
        # Set drag icon
        if item_type == 'map':
            icon_name = 'vector_map.svg'
            if item_data.get('file_format') in ['tif', 'tiff', 'png', 'jpg', 'jpeg']:
                icon_name = 'raster_map.svg'
        else:
            icon_name = 'folder.svg'
            
        icon_path = os.path.join(self.icon_dir, icon_name)
        drag.setPixmap(QIcon(icon_path).pixmap(32, 32))
        
        # Execute drag
        drag.exec_(supportedActions)


class MapBrowserDockWidget(QDockWidget):
    """
    A dock widget that displays MapHub workspaces, folders, and maps in a tree structure.
    Uses async functions for API calls to prevent UI blocking.
    """
    
    def __init__(self, iface, parent=None, refresh_callback=None):
        super(MapBrowserDockWidget, self).__init__(parent)
        self.setWindowTitle("MapHub Browser")
        self.iface = iface
        self.refresh_callback = refresh_callback
        self.logger = logging.getLogger(__name__)
        
        # Setup UI components
        self.content_widget = QWidget()
        self.setWidget(self.content_widget)
        
        # Create layout
        self.layout = QVBoxLayout(self.content_widget)
        self.layout.setContentsMargins(0, 0, 0, 0)
        
        # Create toolbar
        self.toolbar_layout = QHBoxLayout()
        self.toolbar_layout.setContentsMargins(0, 0, 0, 0)
        
        # Add refresh button
        self.icon_dir = os.path.join(os.path.dirname(__file__), '..', '..', 'icons')
        self.refresh_button = QToolButton()
        self.refresh_button.setIcon(QIcon(os.path.join(self.icon_dir, 'refresh.svg')))
        self.refresh_button.setToolTip("Refresh")
        self.refresh_button.clicked.connect(self.on_refresh_clicked)
        self.toolbar_layout.addWidget(self.refresh_button)
        
        # Add synchronize layers button
        self.sync_button = QToolButton()
        self.sync_button.setIcon(QIcon(os.path.join(self.icon_dir, 'sync.svg')))
        self.sync_button.setToolTip("Synchronize Layers with MapHub")
        self.sync_button.clicked.connect(self.on_synchronize_layers_clicked)
        self.toolbar_layout.addWidget(self.sync_button)
        
        # Add spacer to push buttons to the left
        self.toolbar_layout.addStretch()
        
        # Add toolbar to main layout
        self.layout.addLayout(self.toolbar_layout)
        
        # Create tree widget
        self.tree_widget = MapBrowserTreeWidget(parent=self.content_widget, icon_dir=self.icon_dir)
        self.tree_widget.itemExpanded.connect(self.on_item_expanded)
        self.tree_widget.setContextMenuPolicy(Qt.CustomContextMenu)
        self.tree_widget.customContextMenuRequested.connect(self.show_context_menu)
        
        # Set custom delegate for drawing status indicators
        self.tree_widget.setItemDelegate(MapItemDelegate(self.tree_widget))
        
        # Add tree widget to main layout
        self.layout.addWidget(self.tree_widget)
        
        # Initialize state variables
        self.refresh_in_progress = False
        self.expanding_items = set()
        self.custom_context_menu_actions = {}
        
        # Initialize sync manager
        self.sync_manager = MapHubSyncManager(self.iface)
        
        # Load initial data
        asyncio.create_task(self.refresh_browser_async())
    
    def closeEvent(self, event):
        """
        Handle close event for the dock widget.
        """
        # Accept the event
        event.accept()
    
    async def refresh_browser_async(self):
        """
        Async version of refresh_browser that uses asyncio for concurrency.
        Refreshes the browser dock from bottom up, including:
        - Update status indicators for connected maps
        - Reload contents of expanded folders to check for changes
        - Refresh workspaces and their root folders
        - Highlight the folder associated with the current project
        """
        # Check if a refresh is already in progress
        if self.refresh_in_progress:
            self.logger.debug("Refresh already in progress, skipping this request")
            return
            
        # Set the flag to indicate a refresh is in progress
        self.refresh_in_progress = True
        self.logger.debug("Starting async browser refresh")
        
        try:
            # Clear the expanding_items set to ensure a clean state
            if self.expanding_items:
                self.logger.debug(f"Clearing {len(self.expanding_items)} items from expanding_items set")
                self.expanding_items.clear()
            
            # First, update all connected maps concurrently
            connected_maps = 0
            map_refresh_tasks = []
            
            for layer in self.iface.mapCanvas().layers():
                map_id = layer.customProperty("maphub/map_id")
                if map_id:
                    self.logger.debug(f"Scheduling refresh for map item for layer '{layer.name()}' (Map ID: {map_id})")
                    map_refresh_tasks.append(self._refresh_map_item_async(map_id))
                    connected_maps += 1
            
            # Wait for all map refresh tasks to complete
            if map_refresh_tasks:
                await asyncio.gather(*map_refresh_tasks)
                
            self.logger.debug(f"Updated {connected_maps} connected map items")
            
            # Then, reload contents of expanded folders from bottom up
            root = self.tree_widget.invisibleRootItem()
            await self._refresh_expanded_folders_async(root)
            
            # Finally, refresh workspaces and their root folders
            self.logger.debug("Refreshing workspaces and root folders")
            await self._refresh_workspaces_async()
            
            # Highlight the folder associated with the current project
            self.logger.debug("Highlighting project folder")
            self.highlight_project_folder()
            
            self.logger.debug("Async browser refresh completed")
        except Exception as e:
            self.logger.error(f"Error refreshing browser: {str(e)}")
        finally:
            # Clear the flag
            self.refresh_in_progress = False
    
    def refresh_browser(self):
        """
        Refresh the browser dock (synchronous wrapper for backward compatibility).
        This function creates an asyncio task to run the async version.
        """
        # Create an asyncio task to run the async version
        asyncio.create_task(self.refresh_browser_async())
    
    async def _refresh_map_item_async(self, map_id):
        """
        Async version of refresh_map_item.
        
        Args:
            map_id: The ID of the map to refresh
        """
        try:
            # Find the map item in the tree
            map_item = self._find_map_item(self.tree_widget.invisibleRootItem(), map_id)
            if not map_item:
                return
                
            # Get the layer connected to this map
            layer = self.sync_manager.find_layer_by_map_id(map_id)
            if not layer:
                return
                
            # Get the sync status
            status = await asyncio.to_thread(self.sync_manager.get_layer_sync_status, layer)
            
            # Update the status indicator
            self._add_status_indicator(map_item, status)
        except Exception as e:
            self.logger.error(f"Error refreshing map item {map_id}: {str(e)}")
    
    def on_item_expanded(self, item):
        """
        Handle item expansion by loading children if needed.
        """
        # Skip if refresh is in progress
        if self.refresh_in_progress:
            return
            
        # Skip if this item is already being expanded
        if item in self.expanding_items:
            return
            
        item_data = item.data(0, Qt.UserRole)
        if not item_data:
            return
            
        # Skip if this is a leaf node
        if item_data.get('type') == 'map' or item_data.get('type') == 'error':
            return
            
        # Check if this item has only a placeholder child
        has_only_placeholder = (item.childCount() == 1 and 
                               item.child(0).data(0, Qt.UserRole) and 
                               item.child(0).data(0, Qt.UserRole).get('type') == 'placeholder')
                               
        if has_only_placeholder:
            # Mark this item as being expanded
            self.expanding_items.add(item)
            
            # Create appropriate node
            node = self._create_node_from_item(item)
            if node:
                # Load children asynchronously
                asyncio.create_task(self._load_item_children(node))
    
    async def _load_item_children(self, node):
        """
        Load children for a node when it's expanded.
        """
        try:
            # Load children
            children = await node.load_children()
            
            # Remove placeholder
            self._remove_placeholders(node.tree_item)
            
            # Add children to tree
            for child_node in children:
                child_node.create_tree_item(node.tree_item)
                
        except Exception as e:
            self.logger.error(f"Error loading children for {node.get_type()} '{node.name}': {str(e)}")
            
            # Replace placeholder with error message
            self._remove_placeholders(node.tree_item)
            error_item = SortableTreeWidgetItem(node.tree_item)
            error_item.setText(0, f"Error: {str(e)}")
            error_item.setData(0, Qt.UserRole, {'type': 'error'})
            
        finally:
            # Remove this item from expanding items
            self.expanding_items.discard(node.tree_item)
    
    def _create_node_from_item(self, item):
        """
        Create a node object from a tree item.
        """
        item_data = item.data(0, Qt.UserRole)
        if not item_data:
            return None
            
        item_type = item_data.get('type')
        item_id = item_data.get('id')
        item_name = item.text(0)
        
        if item_type == 'workspace':
            return WorkspaceNode(item_id, item_name, item.parent())
        elif item_type == 'folder':
            return FolderNode(item_id, item_name, item.parent())
        elif item_type == 'map':
            return MapNode(item_id, item_name, item.parent(), item_data)
            
        return None
    
    def _add_placeholder(self, parent_item):
        """
        Add a placeholder item to indicate loading.
        """
        placeholder = SortableTreeWidgetItem(parent_item)
        placeholder.setText(0, "Loading...")
        placeholder.setData(0, Qt.UserRole, {'type': 'placeholder'})
        placeholder.setIcon(0, QIcon(os.path.join(self.icon_dir, 'loading.svg')))
        return placeholder
    
    def _remove_placeholders(self, parent_item):
        """
        Remove all placeholder items from a parent item.
        """
        for i in range(parent_item.childCount() - 1, -1, -1):
            child = parent_item.child(i)
            item_data = child.data(0, Qt.UserRole)
            if item_data and item_data.get('type') == 'placeholder':
                parent_item.removeChild(child)
                
    async def _refresh_expanded_folders_async(self, parent_item):
        """
        Recursively refresh the contents of expanded folders from bottom up.
        
        Args:
            parent_item: The parent item to check
        """
        # Get parent item info for logging
        parent_data = parent_item.data(0, Qt.UserRole)
        parent_type = parent_data.get('type') if parent_data else 'root'
        parent_text = parent_item.text(0) if parent_type != 'root' else 'Root'
        
        self.logger.debug(f"Refreshing children of {parent_type} '{parent_text}'")
        
        # First, recursively process all children (bottom-up approach)
        child_refresh_tasks = []
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            if child.childCount() > 0:
                child_refresh_tasks.append(self._refresh_expanded_folders_async(child))
        
        # Wait for all child refresh tasks to complete
        if child_refresh_tasks:
            await asyncio.gather(*child_refresh_tasks)
        
        # Now process this parent's expanded folders
        folder_refresh_tasks = []
        expanded_folders = 0
        
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            item_data = child.data(0, Qt.UserRole)
            
            # Skip placeholder items
            if not item_data or item_data.get('type') == 'placeholder':
                continue
                
            # If this is an expanded folder, reload its contents
            if item_data.get('type') == 'folder' and child.isExpanded():
                folder_id = item_data.get('id')
                folder_name = child.text(0)
                
                self.logger.debug(f"  - Refreshing expanded folder '{folder_name}' (ID: {folder_id})")
                expanded_folders += 1
                
                # Capture expanded state of all nested folders
                expanded_child_folders = self._capture_expanded_state_recursive(child)
                self.logger.debug(f"    - Captured expanded state for {len(expanded_child_folders)} child folders")
                
                # Store expanded state in the folder item for later use
                child.setData(0, Qt.UserRole + 2, expanded_child_folders)
                
                # Store the expanded state of the folder itself for delayed restoration
                was_expanded = child.isExpanded()
                child.setData(0, Qt.UserRole + 3, was_expanded)
                self.logger.debug(f"    - Stored expanded state: {was_expanded}")
                
                # Remove all children except the first one if it's a placeholder
                child_count_before = child.childCount()
                while child.childCount() > 0:
                    # Keep the placeholder if it exists and is the only child
                    if (child.childCount() == 1 and 
                        child.child(0).data(0, Qt.UserRole) and
                        child.child(0).data(0, Qt.UserRole).get('type') == 'placeholder'):
                        break
                    child.removeChild(child.child(0))
                
                # Add a placeholder if there isn't one
                if child.childCount() == 0:
                    placeholder = SortableTreeWidgetItem(child)
                    placeholder.setText(0, "Loading...")
                    placeholder.setData(0, Qt.UserRole, {'type': 'placeholder'})
                    # Add loading icon to placeholder
                    placeholder.setIcon(0, QIcon(os.path.join(self.icon_dir, 'loading.svg')))
                
                self.logger.debug(f"    - Removed {child_count_before - child.childCount()} children")
                
                # Schedule folder content loading
                folder_refresh_tasks.append(self._load_folder_content_async(child, folder_id))
        
        # Wait for all folder refresh tasks to complete
        if folder_refresh_tasks:
            await asyncio.gather(*folder_refresh_tasks)
        
        self.logger.debug(f"Refreshed {expanded_folders} expanded folders under {parent_type} '{parent_text}'")
        return expanded_folders
        
    async def _load_folder_content_async(self, folder_item, folder_id):
        """
        Load folder contents asynchronously.
        
        Args:
            folder_item: The folder item to load contents for
            folder_id: The ID of the folder
        """
        try:
            # Use asyncio.to_thread to wrap the API call
            client = get_maphub_client()
            folder_details = await asyncio.to_thread(client.folder.get_folder, folder_id)
            
            # Process the folder details
            self._process_folder_content(folder_item, folder_details)
            
            # If the folder was expanded, restore its expanded state
            was_expanded = folder_item.data(0, Qt.UserRole + 3)
            if was_expanded:
                folder_text = folder_item.text(0)
                self.logger.debug(f"  - Scheduling delayed expansion for folder '{folder_text}'")
                QTimer.singleShot(250, lambda p=folder_item, t=folder_text: self._delayed_expand(p, t))
                
        except Exception as e:
            self.logger.error(f"Error loading folder content: {str(e)}")
            
            # Replace placeholder with error message
            self._remove_placeholders(folder_item)
            error_item = SortableTreeWidgetItem(folder_item)
            error_item.setText(0, f"Error: {str(e)}")
            error_item.setData(0, Qt.UserRole, {'type': 'error'})
            
    def _process_folder_content(self, folder_item, folder_details):
        """
        Process folder content data and update the tree.
        
        Args:
            folder_item: The folder item to update
            folder_details: The folder details from the API
        """
        # Remove placeholder
        self._remove_placeholders(folder_item)
        
        # Add subfolders
        for subfolder in folder_details.get("subfolders", []):
            subfolder_item = SortableTreeWidgetItem(folder_item)
            subfolder_item.setText(0, subfolder.get("name", "Unnamed Folder"))
            subfolder_item.setData(0, Qt.UserRole, {
                'type': 'folder',
                'id': subfolder.get("id")
            })
            subfolder_item.setIcon(0, QIcon(os.path.join(self.icon_dir, 'folder.svg')))
            
            # Add placeholder for the subfolder
            placeholder = SortableTreeWidgetItem(subfolder_item)
            placeholder.setText(0, "Loading...")
            placeholder.setData(0, Qt.UserRole, {'type': 'placeholder'})
            placeholder.setIcon(0, QIcon(os.path.join(self.icon_dir, 'loading.svg')))
        
        # Add maps
        for map_item in folder_details.get("maps", []):
            map_id = map_item.get("id")
            map_name = map_item.get("name", "Unnamed Map")
            
            # Create map item
            map_tree_item = SortableTreeWidgetItem(folder_item)
            map_tree_item.setText(0, map_name)
            
            # Determine icon based on file format
            icon_name = 'vector_map.svg'
            if map_item.get('file_format') in ['tif', 'tiff', 'png', 'jpg', 'jpeg']:
                icon_name = 'raster_map.svg'
            map_tree_item.setIcon(0, QIcon(os.path.join(self.icon_dir, icon_name)))
            
            # Store map data
            map_tree_item.setData(0, Qt.UserRole, {
                'type': 'map',
                'id': map_id,
                'name': map_name,
                'description': map_item.get('description', ''),
                'created_at': map_item.get('created_at', ''),
                'updated_at': map_item.get('updated_at', ''),
                'file_format': map_item.get('file_format', ''),
                'version_id': map_item.get('version_id', ''),
                'folder_id': folder_details.get('id', '')
            })
            
            # Check if this map is connected to a layer
            layer = self.sync_manager.find_layer_by_map_id(map_id)
            if layer:
                # Get sync status
                status = self.sync_manager.get_layer_sync_status(layer)
                # Add status indicator
                self._add_status_indicator(map_tree_item, status)
                
    async def _refresh_workspaces_async(self):
        """
        Refresh the list of workspaces and their root folders asynchronously.
        """
        self.logger.debug("Refreshing workspaces")
        
        # Get all workspace items
        root = self.tree_widget.invisibleRootItem()
        workspace_items = []
        
        # Collect all workspace items
        for i in range(root.childCount()):
            child = root.child(i)
            item_data = child.data(0, Qt.UserRole)
            if item_data and item_data.get('type') == 'workspace':
                workspace_items.append(child)
        
        self.logger.debug(f"Found {len(workspace_items)} workspace items")
        
        # If there are no workspace items, reload all workspaces
        if not workspace_items:
            self.logger.debug("No workspace items found, reloading all workspaces")
            await self._load_workspaces_async()
            return
            
        # For each workspace item that is expanded, refresh its root folder
        workspace_refresh_tasks = []
        expanded_workspaces = 0
        
        for workspace_item in workspace_items:
            workspace_name = workspace_item.text(0)
            
            if workspace_item.isExpanded():
                expanded_workspaces += 1
                workspace_id = workspace_item.data(0, Qt.UserRole).get('id')
                self.logger.debug(f"Refreshing expanded workspace '{workspace_name}' (ID: {workspace_id})")
                
                # Store expanded state of child folders and their nested folders
                expanded_child_folders = self._capture_expanded_state_recursive(workspace_item)
                self.logger.debug(f"  - Captured expanded state for {len(expanded_child_folders)} child folders")
                
                # Store the expanded state of the workspace item itself
                was_expanded = workspace_item.isExpanded()
                self.logger.debug(f"  - Current expanded state: {was_expanded}")
                
                # Remove all children except the first one if it's a placeholder
                child_count_before = workspace_item.childCount()
                while workspace_item.childCount() > 0:
                    # Keep the placeholder if it exists and is the only child
                    if (workspace_item.childCount() == 1 and 
                        workspace_item.child(0).data(0, Qt.UserRole) and
                        workspace_item.child(0).data(0, Qt.UserRole).get('type') == 'placeholder'):
                        break
                    workspace_item.removeChild(workspace_item.child(0))
                
                # Add a placeholder if there isn't one
                if workspace_item.childCount() == 0:
                    placeholder = SortableTreeWidgetItem(workspace_item)
                    placeholder.setText(0, "Loading...")
                    placeholder.setData(0, Qt.UserRole, {'type': 'placeholder'})
                    # Add loading icon to placeholder
                    placeholder.setIcon(0, QIcon(os.path.join(self.icon_dir, 'loading.svg')))
                
                self.logger.debug(f"  - Removed {child_count_before - workspace_item.childCount()} children")
                
                # Store expanded states for later use
                workspace_item.setData(0, Qt.UserRole + 1, expanded_child_folders)
                
                # Store the expanded state in a special role for delayed restoration
                workspace_item.setData(0, Qt.UserRole + 3, was_expanded)
                
                # Schedule workspace content loading
                workspace_refresh_tasks.append(self._load_workspace_content_async(workspace_item, workspace_id))
            else:
                self.logger.debug(f"Skipping collapsed workspace '{workspace_name}'")
        
        # Wait for all workspace refresh tasks to complete
        if workspace_refresh_tasks:
            await asyncio.gather(*workspace_refresh_tasks)
        
        self.logger.debug(f"Refreshed {expanded_workspaces} expanded workspaces")
        
    async def _load_workspace_content_async(self, workspace_item, workspace_id):
        """
        Load workspace contents asynchronously.
        
        Args:
            workspace_item: The workspace item to load contents for
            workspace_id: The ID of the workspace
        """
        try:
            # Use asyncio.to_thread to wrap the API call
            client = get_maphub_client()
            root_folder = await asyncio.to_thread(client.folder.get_root_folder, workspace_id)
            folder_id = root_folder["folder"]["id"]
            
            # Process the workspace contents
            self._process_workspace_content(workspace_item, folder_id)
            
            # If the workspace was expanded, restore its expanded state
            was_expanded = workspace_item.data(0, Qt.UserRole + 3)
            if was_expanded:
                workspace_text = workspace_item.text(0)
                self.logger.debug(f"  - Scheduling delayed expansion for workspace '{workspace_text}'")
                QTimer.singleShot(250, lambda p=workspace_item, t=workspace_text: self._delayed_expand(p, t))
                
        except Exception as e:
            self.logger.error(f"Error loading workspace content: {str(e)}")
            
            # Replace placeholder with error message
            self._remove_placeholders(workspace_item)
            error_item = SortableTreeWidgetItem(workspace_item)
            error_item.setText(0, f"Error: {str(e)}")
            error_item.setData(0, Qt.UserRole, {'type': 'error'})
            
    def _process_workspace_content(self, workspace_item, folder_id):
        """
        Process workspace content data and update the tree.
        
        Args:
            workspace_item: The workspace item to update
            folder_id: The ID of the root folder
        """
        # Remove placeholder
        self._remove_placeholders(workspace_item)
        
        # Add root folder
        client = get_maphub_client()
        try:
            folder_data = client.folder.get_folder(folder_id)
            
            # Create folder item
            folder_item = SortableTreeWidgetItem(workspace_item)
            folder_item.setText(0, folder_data.get("name", "Root Folder"))
            folder_item.setData(0, Qt.UserRole, {
                'type': 'folder',
                'id': folder_id
            })
            folder_item.setIcon(0, QIcon(os.path.join(self.icon_dir, 'folder.svg')))
            
            # Add placeholder for the folder
            placeholder = SortableTreeWidgetItem(folder_item)
            placeholder.setText(0, "Loading...")
            placeholder.setData(0, Qt.UserRole, {'type': 'placeholder'})
            placeholder.setIcon(0, QIcon(os.path.join(self.icon_dir, 'loading.svg')))
            
        except Exception as e:
            self.logger.error(f"Error loading root folder: {str(e)}")
            
            # Add error item
            error_item = SortableTreeWidgetItem(workspace_item)
            error_item.setText(0, f"Error: {str(e)}")
            error_item.setData(0, Qt.UserRole, {'type': 'error'})
            
    async def _load_workspaces_async(self):
        """
        Load all workspaces asynchronously.
        """
        try:
            # Use asyncio.to_thread to wrap the API call
            client = get_maphub_client()
            workspaces = await asyncio.to_thread(client.workspace.get_workspaces)
            
            # Process the workspaces
            self._process_workspaces(workspaces)
        except Exception as e:
            self.logger.error(f"Error loading workspaces: {str(e)}")
            
            # Clear the tree and add error item
            self.tree_widget.clear()
            error_item = SortableTreeWidgetItem(self.tree_widget)
            error_item.setText(0, f"Error loading workspaces: {str(e)}")
            error_item.setData(0, Qt.UserRole, {'type': 'error'})
            
    def _process_workspaces(self, workspaces):
        """
        Process workspaces data and update the tree.
        
        Args:
            workspaces: The workspaces data from the API
        """
        # Clear the tree
        self.tree_widget.clear()
        
        # Add workspaces
        for workspace in workspaces:
            workspace_id = workspace.get("id")
            workspace_name = workspace.get("name", "Unnamed Workspace")
            
            # Create workspace item
            workspace_item = SortableTreeWidgetItem(self.tree_widget)
            workspace_item.setText(0, workspace_name)
            workspace_item.setData(0, Qt.UserRole, {
                'type': 'workspace',
                'id': workspace_id
            })
            workspace_item.setIcon(0, QIcon(os.path.join(self.icon_dir, 'workspace.svg')))
            
            # Add placeholder for the workspace
            placeholder = SortableTreeWidgetItem(workspace_item)
            placeholder.setText(0, "Loading...")
            placeholder.setData(0, Qt.UserRole, {'type': 'placeholder'})
            placeholder.setIcon(0, QIcon(os.path.join(self.icon_dir, 'loading.svg')))
            
    def _capture_expanded_state_recursive(self, parent_item):
        """
        Capture the expanded state of all children recursively.
        
        Args:
            parent_item: The parent item to check
            
        Returns:
            A dictionary mapping item IDs to their expanded state
        """
        expanded_state = {}
        
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            item_data = child.data(0, Qt.UserRole)
            
            if not item_data:
                continue
                
            item_id = item_data.get('id')
            if not item_id:
                continue
                
            # Store expanded state
            expanded_state[item_id] = {
                'expanded': child.isExpanded(),
                'children': self._capture_expanded_state_recursive(child) if child.childCount() > 0 else {}
            }
            
        return expanded_state
        
    def _delayed_expand(self, item, item_text):
        """
        Expand an item after a delay to allow the UI to update.
        
        Args:
            item: The item to expand
            item_text: The text of the item (for logging)
        """
        if not item:
            return
            
        self.logger.debug(f"Expanding item '{item_text}'")
        
        # Expand the item
        item.setExpanded(True)
        
        # Get expanded child folders
        expanded_child_folders = None
        
        if item.data(0, Qt.UserRole).get('type') == 'workspace':
            expanded_child_folders = item.data(0, Qt.UserRole + 1)
        else:
            expanded_child_folders = item.data(0, Qt.UserRole + 2)
            
        # Expand child folders
        if expanded_child_folders:
            self._expand_child_folders(item, expanded_child_folders)
            
    def _expand_child_folders(self, parent_item, child_expanded_states):
        """
        Expand child folders based on their expanded state.
        
        Args:
            parent_item: The parent item
            child_expanded_states: A dictionary mapping item IDs to their expanded state
        """
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            item_data = child.data(0, Qt.UserRole)
            
            if not item_data:
                continue
                
            item_id = item_data.get('id')
            if not item_id or item_id not in child_expanded_states:
                continue
                
            # Get expanded state
            expanded_state = child_expanded_states[item_id]
            
            # If this child was expanded, expand it
            if expanded_state['expanded']:
                # Schedule expansion after a delay
                child_text = child.text(0)
                QTimer.singleShot(250, lambda p=child, t=child_text: self._delayed_expand(p, t))
                
    def _find_map_item(self, parent_item, map_id):
        """
        Find a map item by its ID.
        
        Args:
            parent_item: The parent item to search in
            map_id: The ID of the map to find
            
        Returns:
            The map item if found, None otherwise
        """
        # Check all children
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            item_data = child.data(0, Qt.UserRole)
            
            if item_data and item_data.get('type') == 'map' and item_data.get('id') == map_id:
                return child
                
            # Recursively search in child folders
            if item_data and (item_data.get('type') == 'folder' or item_data.get('type') == 'workspace'):
                map_item = self._find_map_item(child, map_id)
                if map_item:
                    return map_item
                    
        return None
        
    def _add_status_indicator(self, map_item, status):
        """
        Add a status indicator to a map item.
        
        Args:
            map_item: The map item to add the indicator to
            status: The sync status
        """
        # Set the status indicator
        map_item.setData(0, STATUS_INDICATOR_ROLE, status)
        
        # Force the item to update
        map_item.setData(0, Qt.DisplayRole, map_item.text(0))
        
    def highlight_project_folder(self):
        """
        Highlight the folder associated with the current project.
        """
        # Get the project folder ID
        folder_id = get_project_folder_id()
        if not folder_id:
            return
            
        # Clear any existing highlighting
        root = self.tree_widget.invisibleRootItem()
        self._clear_project_folder_highlighting(root)
        
        # Find and highlight the folder
        self._find_and_highlight_folder(root, folder_id)
        
    def _clear_project_folder_highlighting(self, parent_item):
        """
        Clear project folder highlighting from all items.
        
        Args:
            parent_item: The parent item to clear highlighting from
        """
        # Clear highlighting from this item
        parent_item.setData(0, PROJECT_FOLDER_ROLE, False)
        
        # Clear highlighting from all children
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            self._clear_project_folder_highlighting(child)
            
    def _find_and_highlight_folder(self, parent_item, folder_id):
        """
        Find a folder by its ID and highlight it.
        
        Args:
            parent_item: The parent item to search in
            folder_id: The ID of the folder to find
            
        Returns:
            True if the folder was found and highlighted, False otherwise
        """
        # Check all children
        for i in range(parent_item.childCount()):
            child = parent_item.child(i)
            item_data = child.data(0, Qt.UserRole)
            
            # Check if this is the folder we're looking for
            if item_data and item_data.get('type') == 'folder' and item_data.get('id') == folder_id:
                # Highlight this folder
                child.setData(0, PROJECT_FOLDER_ROLE, True)
                self.logger.debug(f"Found and highlighted project folder: {child.text(0)}")
                return True
                
            # Recursively search in child folders
            if item_data and (item_data.get('type') == 'folder' or item_data.get('type') == 'workspace'):
                if self._find_and_highlight_folder(child, folder_id):
                    return True
                    
        return False
        
    def show_context_menu(self, position):
        """
        Show context menu for the selected item.
        """
        item = self.tree_widget.itemAt(position)
        if not item:
            return
            
        item_data = item.data(0, Qt.UserRole)
        if not item_data:
            return
            
        item_type = item_data.get('type')
        
        # Create menu
        menu = QMenu()
        
        # Add type-specific actions
        if item_type == 'workspace':
            self._add_workspace_actions(menu, item, item_data)
        elif item_type == 'folder':
            self._add_folder_actions(menu, item, item_data)
        elif item_type == 'map':
            self._add_map_actions(menu, item, item_data)
            
        # Add custom actions
        if item_type in self.custom_context_menu_actions:
            for name, callback in self.custom_context_menu_actions[item_type].items():
                action = QAction(name, menu)
                action.triggered.connect(lambda checked, cb=callback, i=item: cb(i))
                menu.addAction(action)
                
        # Show menu if it has actions
        if not menu.isEmpty():
            menu.exec_(self.tree_widget.viewport().mapToGlobal(position))
            
    def _add_workspace_actions(self, menu, item, item_data):
        """
        Add workspace-specific actions to the context menu.
        
        Args:
            menu: The menu to add actions to
            item: The workspace item
            item_data: The workspace item data
        """
        # Add refresh action
        refresh_action = QAction("Refresh", menu)
        refresh_action.triggered.connect(lambda: self.refresh_browser())
        menu.addAction(refresh_action)
        
    def _add_folder_actions(self, menu, item, item_data):
        """
        Add folder-specific actions to the context menu.
        
        Args:
            menu: The menu to add actions to
            item: The folder item
            item_data: The folder item data
        """
        folder_id = item_data.get('id')
        
        # Add download all action
        download_all_action = QAction("Download All Maps", menu)
        download_all_action.triggered.connect(lambda: self.on_download_all_clicked(folder_id))
        menu.addAction(download_all_action)
        
        # Add tiling service action
        tiling_all_action = QAction("Add All Maps as Tiling Services", menu)
        tiling_all_action.triggered.connect(lambda: self.on_tiling_all_clicked(folder_id))
        menu.addAction(tiling_all_action)
        
        # Add load and sync action
        load_and_sync_action = QAction("Load and Synchronize Folder", menu)
        load_and_sync_action.triggered.connect(lambda: self.on_load_and_sync_clicked(folder_id))
        menu.addAction(load_and_sync_action)
        
    def _add_map_actions(self, menu, item, item_data):
        """
        Add map-specific actions to the context menu.
        
        Args:
            menu: The menu to add actions to
            item: The map item
            item_data: The map item data
        """
        map_id = item_data.get('id')
        
        # Check if this map is connected to a layer
        layer = self.sync_manager.find_layer_by_map_id(map_id)
        
        if layer:
            # Add sync action
            sync_action = QAction("Synchronize", menu)
            sync_action.triggered.connect(lambda: self.on_sync_clicked(item_data, layer))
            menu.addAction(sync_action)
            
            # Add disconnect action
            disconnect_action = QAction("Disconnect from Layer", menu)
            disconnect_action.triggered.connect(lambda: self.on_disconnect_clicked(item_data, layer))
            menu.addAction(disconnect_action)
        else:
            # Add download action
            download_action = QAction("Download", menu)
            download_action.triggered.connect(lambda: self.on_download_clicked(item_data))
            menu.addAction(download_action)
            
            # Add tiling service action
            tiling_action = QAction("Add as Tiling Service", menu)
            tiling_action.triggered.connect(lambda: self.on_tiling_clicked(item_data))
            menu.addAction(tiling_action)
            
    def register_context_menu_action(self, item_type, name, callback):
        """
        Register a custom context menu action for a specific item type.
        
        Args:
            item_type: The type of item to add the action to
            name: The name of the action
            callback: The callback function to call when the action is triggered
        """
        if item_type not in self.custom_context_menu_actions:
            self.custom_context_menu_actions[item_type] = {}
            
        self.custom_context_menu_actions[item_type][name] = callback
        
    @handled_exceptions
    def on_download_clicked(self, map_data):
        """Download the selected map."""
        download_map(map_data, self.iface.mainWindow())
        
    @handled_exceptions
    def on_tiling_clicked(self, map_data):
        """Add the selected map as a tiling service."""
        add_map_as_tiling_service(map_data, self.iface.mainWindow())
        
    @handled_exceptions
    def on_download_all_clicked(self, folder_id):
        """Download all maps in the selected folder."""
        download_folder_maps(folder_id, self.iface.mainWindow())
        
    @handled_exceptions
    def on_tiling_all_clicked(self, folder_id):
        """Add all maps in the selected folder as tiling services."""
        add_folder_maps_as_tiling_services(folder_id, self.iface.mainWindow())
        
    @handled_exceptions
    def on_load_and_sync_clicked(self, folder_id):
        """Load and synchronize the selected folder."""
        load_and_sync_folder(folder_id, self.iface, self.iface.mainWindow())
        
    @handled_exceptions
    def on_sync_clicked(self, map_data, layer):
        """Synchronize the selected map with its connected layer."""
        # Get the sync status
        status = self.sync_manager.get_layer_sync_status(layer)
        
        # Determine the sync direction based on the status
        if status == 'local_modified':
            # Upload local changes to MapHub
            self.sync_manager.synchronize_layer(layer, 'upload')
        elif status == 'remote_newer':
            # Download remote changes from MapHub
            self.sync_manager.synchronize_layer(layer, 'download')
        elif status == 'style_changed_local':
            # Upload local style changes to MapHub
            self.sync_manager.synchronize_layer(layer, 'upload', style_only=True)
        elif status == 'style_changed_remote':
            # Download remote style changes from MapHub
            self.sync_manager.synchronize_layer(layer, 'download', style_only=True)
        elif status == 'style_changed_both':
            # Show conflict resolution dialog
            QMessageBox.warning(self.iface.mainWindow(), "Style Conflict", 
                               "Both local and remote styles have changed. Please use the Synchronize Layers dialog to resolve the conflict.")
        elif status == 'file_missing':
            # Download the file again
            self.sync_manager.synchronize_layer(layer, 'download')
        elif status == 'in_sync':
            # Show message that the layer is already in sync
            QMessageBox.information(self.iface.mainWindow(), "Already in Sync", 
                                   f"The layer '{layer.name()}' is already in sync with MapHub.")
        else:
            # Show error message
            QMessageBox.warning(self.iface.mainWindow(), "Synchronization Error", 
                               f"Cannot synchronize layer '{layer.name()}' with status '{status}'.")
                               
        # Refresh the map item
        self.refresh_browser()
        
    @handled_exceptions
    def on_disconnect_clicked(self, map_data, layer):
        """Disconnect the selected map from its connected layer."""
        # Confirm disconnection
        result = QMessageBox.question(self.iface.mainWindow(), "Disconnect from MapHub", 
                                     f"Are you sure you want to disconnect the layer '{layer.name()}' from MapHub?",
                                     QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
                                     
        if result == QMessageBox.Yes:
            # Disconnect the layer
            self.sync_manager.disconnect_layer(layer)
            
            # Refresh the map item
            self.refresh_browser()
            
    @handled_exceptions
    def on_refresh_clicked(self, checked=False):
        """Refresh the browser."""
        self.refresh_browser()
        
    @handled_exceptions
    def on_synchronize_layers_clicked(self, checked=False):
        """Synchronize layers with MapHub."""
        # Create and show the synchronization dialog
        dialog = SynchronizeLayersDialog(self.iface, self.iface.mainWindow())
        dialog.exec_()