"""
Test script for the folder selection scroll bar functionality.

This script tests the fixed height and scroll bar functionality in the folder selection area by:
1. Opening the BatchConnectLayersDialog
2. Verifying the ProjectNavigationWidget has a fixed height and shows a scroll bar when needed

To run this test:
1. Open QGIS
2. Load some layers that are not connected to MapHub
3. Run this script from the QGIS Python console
"""

from qgis.utils import iface
from PyQt5.QtWidgets import QDialog

# Import the dialog
from ui.dialogs.BatchConnectLayersDialog import BatchConnectLayersDialog
from ui.widgets.ProjectNavigationWidget import ProjectNavigationWidget

def test_folder_scroll():
    """Test the folder selection scroll bar functionality."""
    # Get all layers from the current project
    layers = iface.mapCanvas().layers()
    
    # Filter for layers that are not connected to MapHub
    not_connected_layers = [layer for layer in layers if layer.customProperty("maphub/map_id") is None]
    
    if not not_connected_layers:
        print("No unconnected layers found. Please add some layers to the project that are not connected to MapHub.")
        return
    
    # Create and show the BatchConnectLayersDialog
    dialog = BatchConnectLayersDialog(iface, not_connected_layers)
    
    # Get the ProjectNavigationWidget from the dialog
    workspace_nav_widget = dialog.workspace_nav_widget
    project_nav_widget = workspace_nav_widget.project_nav_widget
    
    # Verify the scroll area has a fixed height
    scroll_area = project_nav_widget.scroll_area
    fixed_height = scroll_area.height()
    
    print(f"Scroll area height: {fixed_height}px")
    print(f"Is scroll area visible: {scroll_area.isVisible()}")
    print(f"Scroll area vertical scrollbar policy: {scroll_area.verticalScrollBarPolicy()}")
    
    # Show the dialog for manual verification
    dialog.exec_()
    
    print("Test completed. Please verify that:")
    print("1. The folder selection area has a fixed height")
    print("2. A scroll bar appears when there are many folders")
    print("3. You can scroll through the folders when there are too many to display at once")

if __name__ == "__main__":
    test_folder_scroll()