"""
Test script for the MapHub synchronization manager.

This script demonstrates how to use the MapHubSyncManager class to:
1. Find layers connected to MapHub
2. Check synchronization status
3. Synchronize layers with MapHub

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_sync_manager.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject

from utils.sync_manager import MapHubSyncManager


def test_sync_manager():
    """Test the MapHubSyncManager functionality."""
    # Initialize the sync manager
    sync_manager = MapHubSyncManager(iface)
    
    # Get all connected layers
    connected_layers = sync_manager.get_connected_layers()
    print(f"Found {len(connected_layers)} layers connected to MapHub:")
    
    # If no connected layers, show a message
    if not connected_layers:
        print("No layers connected to MapHub. Download a map first.")
        return
    
    # Print information about each connected layer
    for i, layer in enumerate(connected_layers):
        map_id = layer.customProperty("maphub/map_id")
        folder_id = layer.customProperty("maphub/folder_id")
        last_sync = layer.customProperty("maphub/last_sync")
        local_path = layer.customProperty("maphub/local_path")
        
        print(f"\nLayer {i+1}: {layer.name()}")
        print(f"  Map ID: {map_id}")
        print(f"  Folder ID: {folder_id}")
        print(f"  Last Sync: {last_sync}")
        print(f"  Local Path: {local_path}")
        
        # Check synchronization status
        status = sync_manager.get_layer_sync_status(layer)
        print(f"  Sync Status: {status}")
        
        # Ask if user wants to synchronize this layer
        if status != "in_sync" and status != "not_connected":
            response = input(f"Do you want to synchronize layer '{layer.name()}' (y/n)? ")
            if response.lower() == 'y':
                print(f"Synchronizing layer '{layer.name()}'...")
                sync_manager.synchronize_layer(layer)
                
                # Check status again after synchronization
                new_status = sync_manager.get_layer_sync_status(layer)
                print(f"  New Sync Status: {new_status}")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_sync_manager()
else:
    print("Please run this script from the QGIS Python console.")