"""
Test script for the updated style change detection in MapHub QGIS plugin.

This script demonstrates how the on-demand style hash calculation works:
1. It finds a layer connected to MapHub
2. It checks the synchronization status
3. It modifies the layer style
4. It checks the synchronization status again to verify style changes are detected

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_style_detection.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject, QgsSymbol, QgsSingleSymbolRenderer
from PyQt5.QtGui import QColor

from utils.sync_manager import MapHubSyncManager


def test_style_detection():
    """Test the on-demand style change detection."""
    # Initialize the sync manager
    sync_manager = MapHubSyncManager(iface)
    
    # Get all connected layers
    connected_layers = sync_manager.get_connected_layers()
    print(f"Found {len(connected_layers)} layers connected to MapHub:")
    
    # If no connected layers, show a message
    if not connected_layers:
        print("No layers connected to MapHub. Download a map first.")
        return
    
    # Use the first connected vector layer for testing
    test_layer = None
    for layer in connected_layers:
        if layer.type() == 0:  # Vector layer
            test_layer = layer
            break
    
    if not test_layer:
        print("No vector layers found for testing. Please download a vector map first.")
        return
    
    print(f"\nUsing layer '{test_layer.name()}' for testing style detection")
    
    # Check initial synchronization status
    initial_status = sync_manager.get_layer_sync_status(test_layer)
    print(f"Initial sync status: {initial_status}")
    
    # Modify the layer style
    print("Modifying layer style...")
    if test_layer.renderer().type() == "singleSymbol":
        # Get the symbol
        symbol = test_layer.renderer().symbol()
        
        # Change the color to a random color
        import random
        r, g, b = random.randint(0, 255), random.randint(0, 255), random.randint(0, 255)
        symbol.setColor(QColor(r, g, b))
        
        # Update the layer
        test_layer.triggerRepaint()
        
        print(f"Changed symbol color to RGB({r}, {g}, {b})")
    else:
        print(f"Layer has a {test_layer.renderer().type()} renderer, creating a new single symbol renderer")
        symbol = QgsSymbol.defaultSymbol(test_layer.geometryType())
        symbol.setColor(QColor(255, 0, 0))  # Red
        renderer = QgsSingleSymbolRenderer(symbol)
        test_layer.setRenderer(renderer)
        test_layer.triggerRepaint()
    
    # Check synchronization status after style change
    new_status = sync_manager.get_layer_sync_status(test_layer)
    print(f"New sync status after style change: {new_status}")
    
    # Verify if style change was detected
    if new_status == "style_changed":
        print("SUCCESS: Style change was correctly detected!")
    else:
        print(f"WARNING: Style change was not detected. Status is '{new_status}' instead of 'style_changed'")
        print("This could be because:")
        print("1. The remote map doesn't have style information")
        print("2. There was an error checking the remote style")
        print("3. The style change detection is not working correctly")
    
    # Ask if user wants to synchronize this layer
    if new_status != "in_sync":
        response = input(f"Do you want to synchronize layer '{test_layer.name()}' (y/n)? ")
        if response.lower() == 'y':
            print(f"Synchronizing layer '{test_layer.name()}'...")
            sync_manager.synchronize_layer(test_layer)
            
            # Check status again after synchronization
            final_status = sync_manager.get_layer_sync_status(test_layer)
            print(f"Final sync status after synchronization: {final_status}")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_style_detection()
else:
    print("Please run this script from the QGIS Python console.")