"""
Test script for the Connect button functionality in the SynchronizeLayersDialog.

This script demonstrates how to:
1. Open the SynchronizeLayersDialog
2. Verify that non-connected layers have Connect buttons
3. Test the connection functionality

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_connect_button.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject
from PyQt5.QtWidgets import QPushButton

from ui.dialogs.SynchronizeLayersDialog import SynchronizeLayersDialog


def test_connect_button():
    """Test the Connect button functionality in the SynchronizeLayersDialog."""
    print("Testing Connect button functionality...")
    
    # Check if there are any layers in the project
    all_layers = QgsProject.instance().mapLayers().values()
    if not all_layers:
        print("No layers found in the current project. Please add some layers first.")
        return
    
    # Count non-connected layers
    non_connected_count = 0
    for layer in all_layers:
        if not layer.customProperty("maphub/map_id"):
            non_connected_count += 1
    
    print(f"Found {len(all_layers)} layers in the project, {non_connected_count} not connected to MapHub.")
    
    if non_connected_count == 0:
        print("All layers are already connected to MapHub. Please add a non-connected layer to test the Connect button.")
        return
    
    # Create and show the dialog
    dialog = SynchronizeLayersDialog(iface, iface.mainWindow())
    
    # Check if Connect buttons are present
    connect_buttons_found = False
    for i in range(dialog.layersTree.topLevelItemCount()):
        item = dialog.layersTree.topLevelItem(i)
        widget = dialog.layersTree.itemWidget(item, 3)
        if isinstance(widget, QPushButton) and widget.text() == "Connect":
            connect_buttons_found = True
            print(f"Found Connect button for layer: {item.text(0)}")
    
    if not connect_buttons_found:
        print("No Connect buttons found in the dialog. This is unexpected.")
    else:
        print("\nConnect buttons found. You can now:")
        print("1. Click on a Connect button to open the UploadMapDialog")
        print("2. Upload the layer to MapHub")
        print("3. Verify that the layer is now connected to MapHub")
        print("4. Check that the layer appears as connected in the dialog after refreshing")
    
    # Show the dialog
    result = dialog.exec_()
    
    if result:
        print("Dialog accepted - synchronization performed")
    else:
        print("Dialog rejected - synchronization cancelled")
    
    # Check if any layers were connected
    newly_connected = 0
    for layer in all_layers:
        if layer.customProperty("maphub/map_id") and not layer.customProperty("maphub/map_id") == "":
            newly_connected += 1
    
    if newly_connected > (len(all_layers) - non_connected_count):
        print(f"Success! {newly_connected - (len(all_layers) - non_connected_count)} layers were connected to MapHub.")
    else:
        print("No new layers were connected to MapHub.")
    
    print("\nTest completed.")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_connect_button()
else:
    print("Please run this script from the QGIS Python console.")