#!/usr/bin/env python3
"""
Test script for the MapBrowserDockWidget implementation.
This script simulates the QGIS environment and tests the async functionality.
"""

import os
import sys
import asyncio
import logging
from PyQt5.QtWidgets import QApplication, QMainWindow
from PyQt5.QtCore import QSettings

# Configure logging
logging.basicConfig(level=logging.DEBUG, 
                   format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

# Add the parent directory to the path so we can import the plugin modules
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

# Mock QGIS interfaces
class MockInterface:
    def __init__(self):
        self._main_window = QMainWindow()
        
    def mainWindow(self):
        return self._main_window
        
    def mapCanvas(self):
        return MockMapCanvas()
        
class MockMapCanvas:
    def __init__(self):
        self._layers = []
        
    def layers(self):
        return self._layers

# Mock MapHubSyncManager
class MockSyncManager:
    def __init__(self):
        pass
        
    def find_layer_by_map_id(self, map_id):
        return None
        
    def get_layer_sync_status(self, layer):
        return 'in_sync'

# Main test function
async def test_browser():
    # Create QApplication
    app = QApplication(sys.argv)
    
    # Create mock interface
    iface = MockInterface()
    
    # Import the MapBrowserDockWidget
    try:
        from ui.widgets.MapBrowserDockWidgetAsync import MapBrowserDockWidget
        logger.info("Successfully imported MapBrowserDockWidgetAsync")
    except ImportError as e:
        logger.error(f"Failed to import MapBrowserDockWidgetAsync: {e}")
        return
    
    # Create the dock widget
    try:
        dock = MapBrowserDockWidget(iface)
        logger.info("Successfully created MapBrowserDockWidget")
        
        # Mock the sync manager
        dock.sync_manager = MockSyncManager()
        
        # Show the dock widget
        dock.show()
        logger.info("Showing dock widget")
        
        # Wait for the initial refresh to complete
        await asyncio.sleep(2)
        logger.info("Initial refresh should be complete")
        
        # Test refresh
        await dock.refresh_browser_async()
        logger.info("Manual refresh completed")
        
        # Keep the application running
        app.exec_()
        
    except Exception as e:
        logger.error(f"Error during test: {e}")

# Run the test
if __name__ == "__main__":
    # Set up QSettings for the test
    QSettings().setValue("MapHubPlugin/api_key", "test_api_key")
    
    # Run the async test
    asyncio.run(test_browser())