import os.path

from qgis._core import QgsProject
from qgis.core import QgsProjectBadLayerHandler, QgsProject

from .utils import get_maphub_client, get_default_download_location


def get_project_folder_id() -> str:
    """
    Get the folder ID associated with the current project.
    
    Returns:
        str: The folder ID, or None if not found
    """
    project = QgsProject.instance()
    folder_id, _ = project.readEntry("maphub", "folder_id", "")
    return folder_id if folder_id else None


def folder_has_project(folder_id):
    """
    Check if the folder already has a QGIS project.

    Args:
        folder_id: The ID of the folder to check

    Returns:
        bool: True if the folder has a project, False otherwise
    """
    return get_maphub_client().folder.get_is_project(folder_id)


def save_project_to_maphub(folder_id):
    if not folder_id:
        raise Exception("Cloud not infer folder_id from project.")

    # Get the current project
    project = QgsProject.instance()

    # Set the project title to the folder name.
    folder_info = get_maphub_client().folder.get_folder(folder_id)["folder"]
    project.setTitle(folder_info['name'])

    # Check if the project has been saved
    if project.fileName() == "":
        # Project has never been saved
        base_folder = get_default_download_location()
        local_path = os.path.join(base_folder, f"{folder_id}.qgz")
        
        # Save the project to the temporary file
        project.write(local_path)
    else:
        # Check if the project is stored remotely (e.g., in a PostgreSQL database)
        file_name = project.fileName()
        if file_name.startswith(('postgresql:', 'mysql:', 'oracle:', 'mssql:')) or not os.path.exists(file_name):
            # Project is stored remotely, save to a temporary file
            base_folder = get_default_download_location()
            local_path = os.path.join(base_folder, f"{folder_id}.qgz")
            
            # Save the project to the temporary file
            project.write(local_path)
        else:
            # Project is stored locally, save to its current location
            local_path = file_name
            project.write()

    # Upload the project to MapHub
    get_maphub_client().folder.put_qgis_project(folder_id, local_path)


class MapHubBadLayerHandler(QgsProjectBadLayerHandler):
    """Custom handler for bad layers that attempts to download them from MapHub"""

    def __init__(self):
        super().__init__()

    def handleBadLayers(self, layers):
        """
        Handle bad layers by attempting to download them from MapHub

        Args:
            layers: Iterable of QDomNode objects representing bad layers
        """
        pass

        # TODO make this handler download missing layer data instead of the fix layers function


def load_maphub_project(folder_id):
    base_folder = get_default_download_location()
    project_path = os.path.join(base_folder, f"{folder_id}.qgz")

    # Download the project from MapHub
    get_maphub_client().folder.get_qgis_project(folder_id, project_path)

    # Load the project into QGIS
    project = QgsProject.instance()
    project.setBadLayerHandler(MapHubBadLayerHandler())
    project.read(project_path)

    # Set the project title to the folder name.
    folder_info = get_maphub_client().folder.get_folder(folder_id)["folder"]
    project.setTitle(folder_info['name'])

    # Apply custom properties
    project.writeEntry("maphub", "folder_id", folder_id)
