"""
Test script for the UploadMapDialog with the new layer connection functionality.

This script creates and shows the UploadMapDialog to verify that
the layer is automatically connected to the uploaded map.

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_upload_connect.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject
from ui.dialogs.UploadMapDialog import UploadMapDialog

def test_upload_connect():
    """Test the UploadMapDialog with the new layer connection functionality."""
    print("Testing UploadMapDialog with layer connection functionality...")
    
    # Get the first layer in the project
    layers = QgsProject.instance().mapLayers().values()
    if not layers:
        print("No layers found in the project. Please add a layer first.")
        return
    
    # Create and show the dialog
    dialog = UploadMapDialog(iface, iface.mainWindow())
    
    # Show instructions
    print("\nInstructions:")
    print("1. Select a layer in the dialog")
    print("2. Enter a map name")
    print("3. Select a workspace and folder")
    print("4. Click OK to upload the map")
    print("5. After upload, check if the layer has MapHub connection properties:")
    print("   - maphub/map_id")
    print("   - maphub/folder_id")
    print("   - maphub/workspace_id")
    print("   - maphub/last_sync")
    print("   - maphub/local_path")
    
    # Show the dialog
    result = dialog.exec_()
    
    if result:
        print("\nChecking layer connection properties...")
        # Get the selected layer
        layer = dialog.comboBox_layer.currentData()
        if layer:
            # Check if the layer has MapHub connection properties
            map_id = layer.customProperty("maphub/map_id")
            folder_id = layer.customProperty("maphub/folder_id")
            workspace_id = layer.customProperty("maphub/workspace_id")
            last_sync = layer.customProperty("maphub/last_sync")
            local_path = layer.customProperty("maphub/local_path")
            
            print(f"Layer: {layer.name()}")
            print(f"  Map ID: {map_id}")
            print(f"  Folder ID: {folder_id}")
            print(f"  Workspace ID: {workspace_id}")
            print(f"  Last Sync: {last_sync}")
            print(f"  Local Path: {local_path}")
            
            if map_id:
                print("\nSUCCESS: Layer is connected to MapHub!")
            else:
                print("\nFAILURE: Layer is not connected to MapHub.")
        else:
            print("No layer selected.")
    else:
        print("Dialog cancelled.")

if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_upload_connect()
else:
    print("Please run this script from the QGIS Python console.")