"""
Test script for the redesigned SynchronizeLayersDialog in MapHub QGIS plugin.

This script demonstrates how to:
1. Open the SynchronizeLayersDialog
2. Verify that all layers are displayed
3. Test the synchronization functionality

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_sync_dialog.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject

from ui.dialogs.SynchronizeLayersDialog import SynchronizeLayersDialog
from ui.dialogs.ConfirmSyncDialog import ConfirmSyncDialog


def test_sync_dialog():
    """Test the SynchronizeLayersDialog functionality."""
    print("Testing SynchronizeLayersDialog...")
    
    # Check if there are any layers in the project
    all_layers = QgsProject.instance().mapLayers().values()
    if not all_layers:
        print("No layers found in the current project. Please add some layers first.")
        return
    
    # Count connected layers
    connected_count = 0
    for layer in all_layers:
        if layer.customProperty("maphub/map_id"):
            connected_count += 1
    
    print(f"Found {len(all_layers)} layers in the project, {connected_count} connected to MapHub.")
    
    # Create and show the dialog
    dialog = SynchronizeLayersDialog(iface, iface.mainWindow())
    result = dialog.exec_()
    
    if result:
        print("Dialog accepted - synchronization performed")
    else:
        print("Dialog rejected - synchronization cancelled")
    
    print("\nTo test the confirmation dialog directly:")
    
    # Test the confirmation dialog directly
    if all_layers:
        test_layer = list(all_layers.values())[0]
        confirm_dialog = ConfirmSyncDialog(test_layer.name(), "Test synchronization action", iface.mainWindow())
        confirm_result = confirm_dialog.exec_()
        
        if confirm_result:
            print(f"Confirmation dialog accepted for layer '{test_layer.name()}'")
        else:
            print(f"Confirmation dialog rejected for layer '{test_layer.name()}'")
    
    print("\nTest completed. Please verify that:")
    print("1. The SynchronizeLayersDialog showed all layers in the project")
    print("2. Connected layers had appropriate status indicators")
    print("3. Non-connected layers were grayed out with Connect buttons")
    print("4. The Select All/None buttons worked correctly")
    print("5. The confirmation dialog showed the correct layer name and action")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_sync_dialog()
else:
    print("Please run this script from the QGIS Python console.")