"""
Test script for verifying that the stretcher remains at the bottom when switching folders.

This script tests that the stretcher is correctly maintained at the bottom of the layout
when navigating between folders in the ProjectNavigationWidget.

To run this test:
1. Open QGIS
2. Load some layers that are not connected to MapHub
3. Run this script from the QGIS Python console using:
   exec(open('/path/to/test_stretcher_persistence.py').read())
"""

from qgis.utils import iface
from PyQt5.QtWidgets import QDialog, QVBoxLayout, QSpacerItem, QSizePolicy

# Import the dialog
from ui.dialogs.BatchConnectLayersDialog import BatchConnectLayersDialog

def test_stretcher_persistence():
    """Test that the stretcher remains at the bottom when switching folders."""
    # Get all layers from the current project
    layers = iface.mapCanvas().layers()
    
    # Filter for layers that are not connected to MapHub
    not_connected_layers = [layer for layer in layers if layer.customProperty("maphub/map_id") is None]
    
    if not not_connected_layers:
        print("No unconnected layers found. Please add some layers to the project that are not connected to MapHub.")
        return
    
    # Create the BatchConnectLayersDialog
    dialog = BatchConnectLayersDialog(iface, not_connected_layers)
    
    # Get the ProjectNavigationWidget from the dialog
    workspace_nav_widget = dialog.workspace_nav_widget
    project_nav_widget = workspace_nav_widget.project_nav_widget
    
    # Function to check if the last item is a stretcher
    def check_stretcher():
        list_layout = project_nav_widget.list_layout
        item_count = list_layout.count()
        
        if item_count > 0:
            last_item = list_layout.itemAt(item_count - 1)
            is_stretch = last_item.spacerItem() is not None
            print(f"Total items in layout: {item_count}")
            print(f"Is the last item a stretch: {is_stretch}")
            return is_stretch
        else:
            print("No items in layout")
            return False
    
    # Check initial state
    print("Initial state:")
    initial_has_stretcher = check_stretcher()
    
    # Show the dialog
    dialog.show()
    
    print("\nTest instructions:")
    print("1. Navigate to different folders using the folder items")
    print("2. After each navigation, check if the stretcher is still at the bottom")
    print("3. Verify that items are aligned at the top and don't stretch to fill the space")
    print("\nExpected behavior:")
    print("- The stretcher should remain at the bottom after each folder navigation")
    print("- Folder items should be aligned at the top of the scroll area")
    print("- There should be empty space at the bottom when there are few items")
    
    # Keep the dialog open for manual testing
    dialog.exec_()
    
    print("\nTest completed.")

if __name__ == "__main__":
    test_stretcher_persistence()