"""
Test script for verifying the singleton pattern implementation of MapHubLayerDecorator.

This script tests that:
1. Multiple calls to get_instance() return the same instance
2. Direct instantiation doesn't create new instances
3. Icons don't accumulate after multiple operations

To run this test:
1. Open QGIS
2. Open the Python Console
3. Run: exec(open('/path/to/test_singleton_decorator.py').read())
"""

from qgis.core import QgsProject
from qgis.utils import iface, plugins
from PyQt5.QtWidgets import QMessageBox

# Import the MapHub plugin classes
from utils.layer_decorator import MapHubLayerDecorator

def count_indicators_for_layer(layer_id):
    """Count the number of indicators attached to a layer"""
    layer_tree_view = iface.layerTreeView()
    if not layer_tree_view:
        return 0
        
    # Find the layer node
    root = QgsProject.instance().layerTreeRoot()
    layer_node = root.findLayer(layer_id)
    if not layer_node:
        return 0
    
    # Count indicators
    indicators = layer_tree_view.indicators(layer_node)
    return len(indicators)

def test_singleton_pattern():
    """Test that the singleton pattern is working correctly"""
    # Get the first instance
    instance1 = MapHubLayerDecorator.get_instance(iface)
    
    # Get a second instance - should be the same object
    instance2 = MapHubLayerDecorator.get_instance(iface)
    
    # Try direct instantiation - should not create a new instance
    instance3 = MapHubLayerDecorator(iface)
    
    # Check if all instances are the same object
    is_singleton = (instance1 is instance2) and (instance1 is instance3)
    
    if is_singleton:
        print("Singleton pattern test PASSED: All instances are the same object")
    else:
        print("Singleton pattern test FAILED: Instances are different objects")
        if instance1 is not instance2:
            print("  - get_instance() returned different objects")
        if instance1 is not instance3:
            print("  - Direct instantiation created a new object")
    
    return is_singleton

def test_icon_accumulation():
    """Test that icons don't accumulate after multiple operations"""
    # Get all layers
    layers = QgsProject.instance().mapLayers().values()
    
    # Find layers connected to MapHub
    maphub_layers = []
    for layer in layers:
        if layer.customProperty("maphub/map_id"):
            maphub_layers.append(layer)
    
    if not maphub_layers:
        print("No MapHub-connected layers found. Please connect at least one layer to MapHub.")
        return False
    
    # Get the decorator instance
    decorator = MapHubLayerDecorator.get_instance(iface)
    
    # Get initial indicator counts
    initial_counts = {}
    for layer in maphub_layers:
        initial_counts[layer.id()] = count_indicators_for_layer(layer.id())
        print(f"Initial count for layer '{layer.name()}': {initial_counts[layer.id()]}")
    
    # Create multiple decorator instances (should all be the same object)
    decorators = []
    for i in range(5):
        decorators.append(MapHubLayerDecorator.get_instance(iface))
        
    # Update icons multiple times using different "instances"
    for i, dec in enumerate(decorators):
        print(f"\nUpdating icons with decorator {i+1}...")
        dec.update_layer_icons()
        
        # Check counts after update
        after_update_counts = {}
        for layer in maphub_layers:
            after_update_counts[layer.id()] = count_indicators_for_layer(layer.id())
            print(f"After update {i+1}, count for layer '{layer.name()}': {after_update_counts[layer.id()]}")
        
        # Verify counts didn't change
        for layer_id, initial_count in initial_counts.items():
            current_count = after_update_counts[layer_id]
            layer_name = QgsProject.instance().mapLayer(layer_id).name()
            
            if initial_count != current_count:
                print(f"WARNING: Count changed for layer '{layer_name}': {initial_count} -> {current_count}")
    
    # Final verification
    final_counts = {}
    for layer in maphub_layers:
        final_counts[layer.id()] = count_indicators_for_layer(layer.id())
    
    # Check if counts remained the same
    all_passed = True
    results = []
    for layer_id, initial_count in initial_counts.items():
        final_count = final_counts[layer_id]
        layer_name = QgsProject.instance().mapLayer(layer_id).name()
        
        if initial_count != final_count:
            all_passed = False
            results.append(f"Layer '{layer_name}': FAILED - Initial: {initial_count}, Final: {final_count}")
        else:
            results.append(f"Layer '{layer_name}': PASSED - Count remained at {initial_count}")
    
    print("\nIcon accumulation test results:")
    for result in results:
        print(f"  {result}")
    
    return all_passed

def run_tests():
    """Run all tests and show results"""
    print("Testing MapHubLayerDecorator singleton pattern...")
    singleton_test_passed = test_singleton_pattern()
    
    print("\nTesting icon accumulation prevention...")
    icon_test_passed = test_icon_accumulation()
    
    # Show overall results
    if singleton_test_passed and icon_test_passed:
        QMessageBox.information(None, "Tests Passed", 
                               "All tests passed successfully!\n\n" + 
                               "- Singleton pattern is working correctly\n" +
                               "- Icons don't accumulate after multiple operations")
    else:
        failed_tests = []
        if not singleton_test_passed:
            failed_tests.append("Singleton pattern test")
        if not icon_test_passed:
            failed_tests.append("Icon accumulation test")
            
        QMessageBox.warning(None, "Tests Failed", 
                           f"The following tests failed:\n\n" + 
                           "\n".join(f"- {test}" for test in failed_tests) +
                           "\n\nSee the Python console for details.")

# Run the tests
run_tests()