"""
Test script for the field exclusion implementation in MapHub QGIS Plugin.

This script creates a test layer with QVariantList fields and attempts to export it
using the MapHubSyncManager.add_layer method to verify that problematic fields
are correctly excluded during export.

To run this script:
1. Open QGIS Python Console
2. Run: exec(open('/path/to/field_exclusion_test.py').read())
"""

from qgis.core import QgsVectorLayer, QgsField, QgsFeature, QgsGeometry, QgsPointXY, QgsProject
from PyQt5.QtCore import QVariant
import tempfile
import os

# Import MapHub plugin modules
from maphub.utils.sync_manager import MapHubSyncManager
from maphub.utils.utils import get_maphub_client

def create_test_layer():
    """Create a test layer with QVariantList fields."""
    print("Creating test layer with QVariantList fields...")
    
    # Create a memory layer
    layer = QgsVectorLayer("Point", "test_layer_with_lists", "memory")
    
    # Add normal fields
    layer.dataProvider().addAttributes([
        QgsField("name", QVariant.String),
        QgsField("description", QVariant.String),
        QgsField("value", QVariant.Int)
    ])
    
    # Add a field that will contain lists (this will be a QVariantList)
    layer.dataProvider().addAttributes([
        QgsField("coords_list", QVariant.List),
        QgsField("tags_list", QVariant.List)
    ])
    
    layer.updateFields()
    
    # Add features with list values
    for i in range(5):
        feature = QgsFeature(layer.fields())
        feature.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(i, i)))
        feature.setAttribute("name", f"Test Point {i}")
        feature.setAttribute("description", f"Description for point {i}")
        feature.setAttribute("value", i * 10)
        feature.setAttribute("coords_list", [i, i*2, i*3, i*4, i*5])  # This will be stored as QVariantList
        feature.setAttribute("tags_list", [f"tag{i}", f"category{i}", f"type{i}"])  # This will be stored as QVariantList
        
        layer.dataProvider().addFeature(feature)
    
    # Add the layer to the project
    QgsProject.instance().addMapLayer(layer)
    
    print(f"Created layer with {layer.featureCount()} features and {layer.fields().count()} fields")
    print("Fields:")
    for field in layer.fields():
        print(f"  - {field.name()}: {field.typeName()}")
    
    return layer

def test_export_layer(layer):
    """Test exporting the layer using MapHubSyncManager."""
    print("\nTesting layer export...")
    
    # Create a temporary directory for the test
    with tempfile.TemporaryDirectory() as temp_dir:
        # Create a temporary file path
        temp_file = os.path.join(temp_dir, "test_export.gpkg")
        
        try:
            # Initialize the sync manager
            sync_manager = MapHubSyncManager(None)  # Pass None as iface
            
            # Get a folder ID for testing
            # Note: This requires authentication with MapHub
            # You may need to set up authentication before running this test
            client = get_maphub_client()
            workspaces = client.workspaces.list_workspaces()
            if not workspaces:
                print("No workspaces found. Please authenticate with MapHub first.")
                return
            
            workspace_id = workspaces[0]['id']
            folders = client.folders.list_folders(workspace_id)
            if not folders:
                print("No folders found in the first workspace.")
                return
            
            folder_id = folders[0]['id']
            
            # Export the layer
            print(f"Exporting layer to folder {folder_id}...")
            sync_manager.add_layer(
                layer=layer,
                map_name="Test Layer Export",
                folder_id=folder_id,
                public=False
            )
            
            print("Layer exported successfully!")
            print("Check the QGIS message bar for warnings about excluded fields.")
            
        except Exception as e:
            print(f"Error exporting layer: {str(e)}")

def run_test():
    """Run the complete test."""
    print("=== Field Exclusion Implementation Test ===")
    
    # Create test layer
    layer = create_test_layer()
    
    # Test exporting the layer
    test_export_layer(layer)
    
    print("\nTest completed.")

# Run the test when the script is executed
run_test()