import asyncio
import logging
import os
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QTreeWidgetItem

from ...utils.utils import get_maphub_client


class SortableTreeWidgetItem(QTreeWidgetItem):
    """
    A QTreeWidgetItem that can be sorted alphabetically.
    """
    def __lt__(self, other):
        # Get item data
        item_data = self.data(0, Qt.UserRole)
        other_data = other.data(0, Qt.UserRole)
        
        # Handle placeholder items (always at the end)
        if item_data and item_data.get('type') == 'placeholder':
            return False
        if other_data and other_data.get('type') == 'placeholder':
            return True
            
        # Handle error items (always at the end)
        if item_data and item_data.get('type') == 'error':
            return False
        if other_data and other_data.get('type') == 'error':
            return True
            
        # Sort by type first (workspaces, then folders, then maps)
        type_order = {'workspace': 0, 'folder': 1, 'map': 2}
        
        item_type = item_data.get('type') if item_data else ''
        other_type = other_data.get('type') if other_data else ''
        
        if item_type in type_order and other_type in type_order:
            if type_order[item_type] != type_order[other_type]:
                return type_order[item_type] < type_order[other_type]
                
        # Then sort alphabetically by name
        return self.text(0).lower() < other.text(0).lower()


class TreeNode:
    """Base class for all tree nodes"""
    def __init__(self, item_id, name, parent_item=None):
        self.id = item_id
        self.name = name
        self.parent_item = parent_item
        self.tree_item = None  # Will hold reference to QTreeWidgetItem
        self.logger = logging.getLogger(__name__)
        
    async def load_children(self):
        """Load children for this node - to be implemented by subclasses"""
        pass
        
    def create_tree_item(self, parent_item):
        """Create a QTreeWidgetItem for this node"""
        self.tree_item = SortableTreeWidgetItem(parent_item)
        self.tree_item.setText(0, self.name)
        self.tree_item.setData(0, Qt.UserRole, {'type': self.get_type(), 'id': self.id})
        return self.tree_item
    
    def get_type(self):
        """Return the type of this node"""
        return "node"


class WorkspaceNode(TreeNode):
    """Represents a workspace in the tree"""
    def __init__(self, workspace_id, name, parent_item=None):
        super().__init__(workspace_id, name, parent_item)
        
    def get_type(self):
        return "workspace"
    
    def create_tree_item(self, parent_item):
        """Create a QTreeWidgetItem for this workspace"""
        tree_item = super().create_tree_item(parent_item)
        # Add workspace icon
        icon_path = os.path.join(os.path.dirname(__file__), '..', '..', 'icons', 'workspace.svg')
        tree_item.setIcon(0, QIcon(icon_path))
        return tree_item
        
    async def load_children(self):
        """Load root folder for this workspace"""
        try:
            client = get_maphub_client()
            root_folder = await asyncio.to_thread(client.folder.get_root_folder, self.id)
            folder_id = root_folder["folder"]["id"]
            return [FolderNode(folder_id, root_folder["folder"]["name"], self.tree_item)]
        except Exception as e:
            self.logger.error(f"Error loading workspace children: {str(e)}")
            return []


class FolderNode(TreeNode):
    """Represents a folder in the tree"""
    def __init__(self, folder_id, name, parent_item=None):
        super().__init__(folder_id, name, parent_item)
        
    def get_type(self):
        return "folder"
    
    def create_tree_item(self, parent_item):
        """Create a QTreeWidgetItem for this folder"""
        tree_item = super().create_tree_item(parent_item)
        # Add folder icon
        icon_path = os.path.join(os.path.dirname(__file__), '..', '..', 'icons', 'folder.svg')
        tree_item.setIcon(0, QIcon(icon_path))
        return tree_item
        
    async def load_children(self):
        """Load subfolders and maps for this folder"""
        try:
            client = get_maphub_client()
            folder_details = await asyncio.to_thread(client.folder.get_folder, self.id)
            
            children = []
            
            # Add subfolders
            for subfolder in folder_details.get("subfolders", []):
                children.append(FolderNode(subfolder["id"], subfolder["name"], self.tree_item))
                
            # Add maps
            for map_item in folder_details.get("maps", []):
                children.append(MapNode(map_item["id"], map_item["name"], self.tree_item, map_item))
                
            return children
        except Exception as e:
            self.logger.error(f"Error loading folder children: {str(e)}")
            return []


class MapNode(TreeNode):
    """Represents a map in the tree"""
    def __init__(self, map_id, name, parent_item=None, map_data=None):
        super().__init__(map_id, name, parent_item)
        self.map_data = map_data
        
    def get_type(self):
        return "map"
        
    async def load_children(self):
        """Maps don't have children"""
        return []
        
    def create_tree_item(self, parent_item):
        """Create a QTreeWidgetItem for this map with additional data"""
        tree_item = super().create_tree_item(parent_item)
        
        # Add map-specific icon based on file format
        icon_name = 'vector_map.svg'
        if self.map_data and self.map_data.get('file_format') in ['tif', 'tiff', 'png', 'jpg', 'jpeg']:
            icon_name = 'raster_map.svg'
        
        icon_path = os.path.join(os.path.dirname(__file__), '..', '..', 'icons', icon_name)
        tree_item.setIcon(0, QIcon(icon_path))
        
        # Add map-specific data
        if self.map_data:
            tree_item.setData(0, Qt.UserRole, {
                'type': 'map',
                'id': self.id,
                'name': self.name,
                'description': self.map_data.get('description', ''),
                'created_at': self.map_data.get('created_at', ''),
                'updated_at': self.map_data.get('updated_at', ''),
                'file_format': self.map_data.get('file_format', ''),
                'version_id': self.map_data.get('version_id', ''),
                'folder_id': self.map_data.get('folder_id', '')
            })
            
        return tree_item