"""
Test script for the folder synchronization functionality in MapHub QGIS plugin.

This script demonstrates how to:
1. Access the MapBrowserDockWidget
2. Find a folder in the tree
3. Trigger the folder synchronization dialog
4. Verify the dialog functionality

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_folder_sync.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject
from PyQt5.QtCore import Qt

from ui.widgets.MapBrowserDockWidget import MapBrowserDockWidget
from ui.dialogs.FolderSyncDialog import FolderSyncDialog


def find_folder_in_tree(dock_widget, folder_name=None):
    """
    Find a folder in the MapBrowserDockWidget tree.
    
    Args:
        dock_widget: The MapBrowserDockWidget
        folder_name: Optional folder name to search for
        
    Returns:
        Tuple of (folder_item, folder_id) if found, (None, None) otherwise
    """
    # Get the root item
    root = dock_widget.tree_widget.invisibleRootItem()
    
    # Search for a folder
    for i in range(root.childCount()):
        workspace_item = root.child(i)
        
        # Expand the workspace to load its contents
        if workspace_item.childCount() == 1 and workspace_item.child(0).text(0) == "Loading...":
            dock_widget.on_item_expanded(workspace_item)
            # Wait for loading to complete
            import time
            time.sleep(2)
        
        # Check each child of the workspace
        for j in range(workspace_item.childCount()):
            child = workspace_item.child(j)
            item_data = child.data(0, Qt.UserRole)
            
            if item_data and item_data.get('type') == 'folder':
                # If folder_name is specified, check if it matches
                if folder_name:
                    if child.text(0) == folder_name:
                        return child, item_data.get('id')
                else:
                    # Return the first folder found
                    return child, item_data.get('id')
    
    return None, None


def test_folder_sync():
    """Test the folder synchronization functionality."""
    print("Testing folder synchronization functionality...")
    
    # Get or create the MapBrowserDockWidget
    dock_widget = None
    for dock in iface.mainWindow().findChildren(MapBrowserDockWidget):
        dock_widget = dock
        break
    
    if not dock_widget:
        dock_widget = MapBrowserDockWidget(iface, iface.mainWindow())
        iface.addDockWidget(Qt.LeftDockWidgetArea, dock_widget)
        print("Created new MapBrowserDockWidget")
    else:
        print("Using existing MapBrowserDockWidget")
    
    # Find a folder in the tree
    folder_item, folder_id = find_folder_in_tree(dock_widget)
    
    if not folder_item:
        print("No folders found in the MapBrowserDockWidget. Please expand a workspace first.")
        return
    
    print(f"Found folder: {folder_item.text(0)} (ID: {folder_id})")
    
    # Test the folder synchronization dialog directly
    print("Opening FolderSyncDialog directly...")
    dialog = FolderSyncDialog(folder_id, iface, iface.mainWindow())
    result = dialog.exec_()
    
    if result:
        print("Dialog accepted")
    else:
        print("Dialog rejected")
    
    # Test the context menu action
    print("\nTo test the context menu action:")
    print("1. Right-click on a folder in the MapHub Browser")
    print("2. Select 'Synchronize Folder' from the context menu")
    print("3. Verify that the folder synchronization dialog appears")
    print("4. Test the dialog functionality")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_folder_sync()
else:
    print("Please run this script from the QGIS Python console.")