"""
Test script for the UI integration of the MapHub synchronization features.

This script demonstrates how to:
1. Initialize the MapHubLayerDecorator for visual indicators
2. Initialize the MapHubLayerMenuProvider for context menu integration
3. Test the synchronization dialog

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_ui_integration.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject

from utils.sync_manager import MapHubSyncManager
from utils.layer_decorator import MapHubLayerDecorator
from utils.layer_menu_provider import MapHubLayerMenuProvider


def test_ui_integration():
    """Test the UI integration of the MapHub synchronization features."""
    print("Testing UI integration of MapHub synchronization features...")
    
    # Initialize the sync manager
    sync_manager = MapHubSyncManager(iface)
    
    # Initialize the layer decorator
    layer_decorator = MapHubLayerDecorator(iface)
    
    # Initialize the layer menu provider
    layer_menu_provider = MapHubLayerMenuProvider(iface, sync_manager)
    
    # Get connected layers
    connected_layers = sync_manager.get_connected_layers()
    print(f"Found {len(connected_layers)} layers connected to MapHub:")
    
    # If no connected layers, show a message
    if not connected_layers:
        print("No layers connected to MapHub. Download a map first.")
        return
    
    # Update layer icons
    layer_decorator.update_layer_icons()
    print("Updated layer icons with MapHub status indicators.")
    
    # Print information about each connected layer
    for i, layer in enumerate(connected_layers):
        map_id = layer.customProperty("maphub/map_id")
        folder_id = layer.customProperty("maphub/folder_id")
        last_sync = layer.customProperty("maphub/last_sync")
        local_path = layer.customProperty("maphub/local_path")
        
        print(f"\nLayer {i+1}: {layer.name()}")
        print(f"  Map ID: {map_id}")
        print(f"  Folder ID: {folder_id}")
        print(f"  Last Sync: {last_sync}")
        print(f"  Local Path: {local_path}")
        
        # Check synchronization status
        status = sync_manager.get_layer_sync_status(layer)
        print(f"  Sync Status: {status}")
    
    print("\nUI integration test completed.")
    print("You should now see:")
    print("1. Visual indicators in the layer panel for connected layers")
    print("2. Context menu options for connected layers (right-click on a layer)")
    print("3. Status indicators in the MapHub Browser for connected maps")
    print("\nYou can also test the Synchronize button in the MapHub toolbar.")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_ui_integration()
else:
    print("Please run this script from the QGIS Python console.")