#!/usr/bin/env python3
"""
Test script to verify the style hash history functionality in the MapHub QGIS plugin.

This script simulates different style change scenarios and verifies that the plugin
correctly identifies whether changes were made locally, remotely, or both.
"""

import hashlib
from PyQt5.QtXml import QDomDocument

def create_test_style(include_maphub_props=True):
    """Create a test style XML document."""
    xml_content = """
    <qgis version="3.22.0-Białowieża">
      <customproperties>
        <Option type="Map">
          <Option type="int" value="0" name="embeddedWidgets/count"/>
          {maphub_props}
          <Option type="invalid" name="variableNames"/>
          <Option type="invalid" name="variableValues"/>
        </Option>
      </customproperties>
      <renderer-v2 type="singleSymbol">
        <symbols>
          <symbol type="fill" name="0" clip_to_extent="1" alpha="1" force_rhr="0">
            <data_defined_properties>
              <Option type="Map">
                <Option type="QString" value="" name="name"/>
                <Option name="properties"/>
                <Option type="QString" value="collection" name="type"/>
              </Option>
            </data_defined_properties>
            <layer class="SimpleFill" enabled="1" locked="0" pass="0">
              <Option type="Map">
                <Option type="QString" value="3x:0,0,0,0,0,0" name="border_width_map_unit_scale"/>
                <Option type="QString" value="{fill_color}" name="color"/>
                <Option type="QString" value="bevel" name="joinstyle"/>
                <Option type="QString" value="0,0" name="offset"/>
                <Option type="QString" value="3x:0,0,0,0,0,0" name="offset_map_unit_scale"/>
                <Option type="QString" value="MM" name="offset_unit"/>
                <Option type="QString" value="0,0,0,255" name="outline_color"/>
                <Option type="QString" value="solid" name="outline_style"/>
                <Option type="QString" value="0.26" name="outline_width"/>
                <Option type="QString" value="MM" name="outline_width_unit"/>
                <Option type="QString" value="solid" name="style"/>
              </Option>
            </layer>
          </symbol>
        </symbols>
      </renderer-v2>
    </qgis>
    """
    
    maphub_props_str = """
          <Option type="QString" value="1f921f98-41d8-40e3-be26-f75269857573" name="maphub/folder_id"/>
          <Option type="QString" value="2025-08-28T14:06:09.625890" name="maphub/last_sync"/>
          <Option type="QString" value="/home/user/data/test_layer.gpkg" name="maphub/local_path"/>
          <Option type="QString" value="390b883f-6733-449e-8fef-a6d3ac695d5b" name="maphub/map_id"/>
          <Option type="QString" value="abc123" name="maphub/last_style_hash"/>
    """ if include_maphub_props else ""
    
    return xml_content.format(maphub_props=maphub_props_str, fill_color="255,0,0,255")

def filter_maphub_properties(style_xml):
    """Filter out MapHub-specific properties from style XML."""
    doc = QDomDocument()
    doc.setContent(style_xml)
    
    # Filter out MapHub properties
    root = doc.documentElement()
    custom_props = root.elementsByTagName("customproperties").item(0)
    if not custom_props.isNull():
        options = custom_props.firstChildElement("Option")
        if not options.isNull():
            option_list = options.childNodes()
            for i in range(option_list.count() - 1, -1, -1):
                option = option_list.item(i).toElement()
                if option.attribute("name").startswith("maphub/"):
                    options.removeChild(option_list.item(i))
    
    return doc.toString()

def get_style_hash(style_xml):
    """Get hash of filtered style XML."""
    filtered_xml = filter_maphub_properties(style_xml)
    return hashlib.md5(filtered_xml.encode()).hexdigest()

def test_style_hash_history():
    """Test the style hash history functionality."""
    print("Testing style hash history functionality:")
    print("-" * 50)
    
    # Create initial style (red fill)
    initial_style = create_test_style()
    initial_hash = get_style_hash(initial_style)
    print(f"Initial style hash: {initial_hash}")
    
    # Simulate last synced hash
    last_synced_hash = initial_hash
    print(f"Last synced hash: {last_synced_hash}")
    
    # Create local modified style (green fill)
    local_style = create_test_style().replace("255,0,0,255", "0,255,0,255")
    local_hash = get_style_hash(local_style)
    print(f"Local modified style hash: {local_hash}")
    
    # Create remote modified style (blue fill)
    remote_style = create_test_style().replace("255,0,0,255", "0,0,255,255")
    remote_hash = get_style_hash(remote_style)
    print(f"Remote modified style hash: {remote_hash}")
    
    # Test scenarios
    print("\nTesting style change detection scenarios:")
    print("-" * 50)
    
    # Scenario 1: No changes
    print("\nScenario 1: No changes")
    if initial_hash == last_synced_hash:
        print("✓ Correctly detected no changes")
    else:
        print("✗ Failed to detect no changes")
    
    # Scenario 2: Local changes only
    print("\nScenario 2: Local changes only")
    if local_hash != last_synced_hash and remote_hash == last_synced_hash:
        print("✓ Correctly detected local changes only")
        print("  Status should be: style_changed_local")
    else:
        print("✗ Failed to detect local changes only")
    
    # Scenario 3: Remote changes only
    print("\nScenario 3: Remote changes only")
    if local_hash == last_synced_hash and remote_hash != last_synced_hash:
        print("✓ Correctly detected remote changes only")
        print("  Status should be: style_changed_remote")
    else:
        print("✗ Failed to detect remote changes only")
    
    # Scenario 4: Both local and remote changes
    print("\nScenario 4: Both local and remote changes")
    if local_hash != last_synced_hash and remote_hash != last_synced_hash:
        print("✓ Correctly detected changes on both sides")
        print("  Status should be: style_changed_both")
    else:
        print("✗ Failed to detect changes on both sides")
    
    # Scenario 5: No last synced hash
    print("\nScenario 5: No last synced hash")
    print("  Status should be: style_changed (can't determine which side changed)")
    
    print("\nSummary:")
    print("-" * 50)
    print("The style hash history approach allows the plugin to determine whether")
    print("style changes were made locally, remotely, or both, by comparing the")
    print("current local and remote style hashes with the last synced hash.")
    print("This enables more intelligent conflict resolution strategies.")

if __name__ == "__main__":
    test_style_hash_history()