"""
Test script for verifying that icons don't accumulate after multiple push operations.

This script simulates multiple push operations and checks if icons accumulate.

To run this test:
1. Open QGIS
2. Open the Python Console
3. Run: exec(open('/path/to/test_push_icons.py').read())
"""

from qgis.core import QgsProject
from qgis.utils import iface, plugins
from PyQt5.QtWidgets import QMessageBox

def count_indicators_for_layer(layer_id):
    """Count the number of indicators attached to a layer"""
    layer_tree_view = iface.layerTreeView()
    if not layer_tree_view:
        return 0
        
    # Find the layer node
    root = QgsProject.instance().layerTreeRoot()
    layer_node = root.findLayer(layer_id)
    if not layer_node:
        return 0
    
    # Count indicators
    indicators = layer_tree_view.indicators(layer_node)
    return len(indicators)

def test_push_operations():
    """Test multiple push operations to ensure icons don't accumulate"""
    # Get all layers
    layers = QgsProject.instance().mapLayers().values()
    
    # Find a layer connected to MapHub
    maphub_layers = []
    for layer in layers:
        if layer.customProperty("maphub/map_id"):
            maphub_layers.append(layer)
    
    if not maphub_layers:
        QMessageBox.warning(None, "Test Failed", 
                           "No MapHub-connected layers found. Please connect at least one layer to MapHub.")
        return
    
    # Get the MapHub plugin
    if 'MapHubPlugin' not in plugins:
        QMessageBox.warning(None, "Test Failed", 
                           "MapHub plugin not found or not loaded.")
        return
    
    maphub_plugin = plugins['MapHubPlugin']
    sync_manager = maphub_plugin.sync_manager
    
    # Get initial indicator counts
    initial_counts = {}
    for layer in maphub_layers:
        initial_counts[layer.id()] = count_indicators_for_layer(layer.id())
        print(f"Initial count for layer '{layer.name()}': {initial_counts[layer.id()]}")
    
    # Simulate multiple push operations
    for i in range(3):
        print(f"\nSimulating push operation {i+1}...")
        
        # Update layer icons before each operation
        maphub_plugin.layer_decorator.update_layer_icons()
        
        # Check counts after update
        after_update_counts = {}
        for layer in maphub_layers:
            after_update_counts[layer.id()] = count_indicators_for_layer(layer.id())
            print(f"After update {i+1}, count for layer '{layer.name()}': {after_update_counts[layer.id()]}")
        
        # Verify counts didn't change
        for layer_id, initial_count in initial_counts.items():
            current_count = after_update_counts[layer_id]
            layer_name = QgsProject.instance().mapLayer(layer_id).name()
            
            if initial_count != current_count:
                print(f"WARNING: Count changed for layer '{layer_name}': {initial_count} -> {current_count}")
    
    # Final verification
    final_counts = {}
    for layer in maphub_layers:
        final_counts[layer.id()] = count_indicators_for_layer(layer.id())
    
    # Check if counts remained the same
    all_passed = True
    results = []
    for layer_id, initial_count in initial_counts.items():
        final_count = final_counts[layer_id]
        layer_name = QgsProject.instance().mapLayer(layer_id).name()
        
        if initial_count != final_count:
            all_passed = False
            results.append(f"Layer '{layer_name}': FAILED - Initial: {initial_count}, Final: {final_count}")
        else:
            results.append(f"Layer '{layer_name}': PASSED - Count remained at {initial_count}")
    
    # Show results
    if all_passed:
        QMessageBox.information(None, "Test Passed", 
                               "Icon count remained consistent after multiple updates.\n\n" + 
                               "\n".join(results))
    else:
        QMessageBox.warning(None, "Test Failed", 
                           "Icon count changed after multiple updates.\n\n" + 
                           "\n".join(results))

# Run the test
test_push_operations()