"""
Test script for the refactored layer connection functionality in MapHub QGIS plugin.

This script demonstrates how to:
1. Use the MapHubSyncManager to connect a layer to MapHub
2. Verify that the layer is properly connected
3. Disconnect the layer from MapHub

Usage:
    Run this script from the QGIS Python console:
    
    ```python
    import os
    os.chdir('/path/to/maphub-qgis')
    exec(open('test_layer_connection.py').read())
    ```
"""

from qgis.utils import iface
from qgis.core import QgsProject

from utils.sync_manager import MapHubSyncManager


def test_layer_connection():
    """Test the layer connection functionality."""
    print("Testing layer connection functionality...")
    
    # Get the first layer in the project
    layers = QgsProject.instance().mapLayers().values()
    if not layers:
        print("No layers found in the project. Please add a layer first.")
        return
    
    test_layer = list(layers)[0]
    print(f"Using layer: {test_layer.name()}")
    
    # Initialize the sync manager
    sync_manager = MapHubSyncManager(iface)
    
    # Check if the layer is already connected
    if test_layer.customProperty("maphub/map_id"):
        print("Layer is already connected to MapHub. Disconnecting first...")
        sync_manager.disconnect_layer(test_layer)
    
    # Connect the layer to MapHub with test values
    print("Connecting layer to MapHub...")
    sync_manager.connect_layer(
        test_layer,
        "test-map-id",
        "test-folder-id",
        test_layer.source()
    )
    
    # Verify that the layer is connected
    print("\nVerifying layer connection...")
    map_id = test_layer.customProperty("maphub/map_id")
    folder_id = test_layer.customProperty("maphub/folder_id")
    workspace_id = test_layer.customProperty("maphub/workspace_id")
    last_sync = test_layer.customProperty("maphub/last_sync")
    local_path = test_layer.customProperty("maphub/local_path")
    
    print(f"  Map ID: {map_id}")
    print(f"  Folder ID: {folder_id}")
    print(f"  Workspace ID: {workspace_id}")
    print(f"  Last Sync: {last_sync}")
    print(f"  Local Path: {local_path}")
    
    if map_id == "test-map-id" and folder_id == "test-folder-id" and workspace_id == "test-workspace-id":
        print("\nSUCCESS: Layer was properly connected to MapHub!")
    else:
        print("\nFAILURE: Layer was not properly connected to MapHub.")
    
    # Disconnect the layer
    print("\nDisconnecting layer from MapHub...")
    sync_manager.disconnect_layer(test_layer)
    
    # Verify that the layer is disconnected
    print("\nVerifying layer disconnection...")
    map_id = test_layer.customProperty("maphub/map_id")
    
    if not map_id:
        print("SUCCESS: Layer was properly disconnected from MapHub!")
    else:
        print("FAILURE: Layer was not properly disconnected from MapHub.")


if __name__ == "__console__":
    # This will be true when run from the QGIS Python console
    test_layer_connection()
else:
    print("Please run this script from the QGIS Python console.")