# Layer-Based Synchronization in MapHub QGIS Plugin

## Overview

The MapHub QGIS Plugin now uses a layer-based synchronization approach that stores connection information directly in QGIS layer properties. This approach provides several benefits:

- Clear visual representation of which layers are connected to MapHub
- Direct synchronization of individual layers
- No hidden metadata or complex folder structure required
- Integration with the QGIS layer panel and MapHub browser

## How It Works

### Connection Information

When a map is downloaded from MapHub, the following information is stored in the layer's custom properties:

- `maphub/map_id`: The ID of the map on MapHub
- `maphub/folder_id`: The ID of the folder containing the map
- `maphub/workspace_id`: The ID of the workspace containing the folder
- `maphub/last_sync`: Timestamp of the last synchronization
- `maphub/local_path`: Path to the local file

This information is used to track the relationship between local layers and remote maps, and to determine when synchronization is needed.

### Style Change Detection

The plugin uses a hash history approach to detect style changes and identify their source:

1. When checking synchronization status, the current layer style is exported to QGIS format
2. MapHub-specific properties (like `maphub/last_sync`) are filtered out to prevent false positives
3. A hash is calculated from the filtered QGIS style representation
4. The plugin retrieves the remote map's style and calculates a hash from its filtered QGIS representation
5. If the hashes differ, the plugin compares both with the last synced style hash to determine which side changed:
   - If the local style hash differs from the last synced hash, local style has changed
   - If the remote style hash differs from the last synced hash, remote style has changed
   - If both differ from the last synced hash, both sides have changed

During synchronization, the current style hash is stored as the "last synced hash" in the layer's custom properties. This allows the plugin to track style changes and determine whether they were made locally, remotely, or both.

This approach ensures that style changes are detected accurately and their source is identified, allowing for appropriate resolution strategies. By filtering out MapHub-specific properties that change frequently but don't affect the visual appearance, the plugin avoids false style change detections.

### Synchronization Status

The plugin checks the synchronization status of connected layers on demand. The possible statuses are:

- `not_connected`: Layer is not connected to MapHub
- `file_missing`: Local file is missing
- `local_modified`: Local file has been modified since last sync
- `remote_newer`: Remote map has been updated since last sync
- `style_changed`: Style has changed since last sync (legacy status)
- `style_changed_local`: Local style has changed since last sync
- `style_changed_remote`: Remote style has changed since last sync
- `style_changed_both`: Both local and remote styles have changed since last sync
- `remote_error`: Error checking remote status
- `in_sync`: Layer is in sync with MapHub

### Synchronization Actions

The plugin provides the following synchronization actions:

- **Synchronize**: Automatically determines the synchronization direction based on the status
- **Upload to MapHub**: Uploads local changes to MapHub, renames the file to include version information, and deletes the old file
- **Update from MapHub**: Downloads remote changes from MapHub
- **Upload Style to MapHub**: Uploads local style changes to MapHub
- **Download Style from MapHub**: Downloads remote style changes from MapHub
- **Resolve Style Conflict**: Shows a dialog to resolve style conflicts when both sides have changed
- **Disconnect from MapHub**: Removes the connection information from the layer

### Visual Indicators

The plugin provides visual indicators to show which maps are connected to local layers:

- In the MapHub Browser, connected maps are shown with bold text
- In the QGIS layer panel, connected layers have visual indicators showing their connection and synchronization status
- Each connected layer displays a single indicator that shows its current state:
  - Chain icon: Layer is connected to MapHub and in sync
  - Green (upload): Local changes need to be uploaded to MapHub
  - Blue (download): Remote changes need to be downloaded from MapHub
  - Purple (style): Style changes detected
  - Red (error): Local file is missing
  - Orange (warning): Error checking remote status

## File Management

### File Naming and Storage

When maps are downloaded from MapHub or pushed to MapHub, the plugin uses a consistent file naming convention:

- Files are named using the pattern `{map_id}_{version_id}.{extension}`
- Files are stored in the default download location (configurable in settings)
- This naming convention ensures that files are uniquely identified by their map ID and version ID

### File Handling During Push Operations

When you push a layer to MapHub (upload local changes):

1. The plugin uploads the file to MapHub, creating a new version
2. After successful upload, the plugin:
   - Creates a new file with the naming convention `{map_id}_{version_id}.{extension}` in the default download location
   - Copies the content from the original file to the new file
   - Updates the layer's source path to point to the new file
   - Deletes the original file to avoid duplication
3. For shapefiles, all related files (.dbf, .shx, .prj, etc.) are also copied and the originals are deleted

This approach ensures that:
- Files are consistently named with their map_id and version_id
- The file system remains clean with only the current version of each file
- Disk space is used efficiently by avoiding duplicate files

## Using the Synchronization Features

### Downloading Maps

When you download a map from MapHub, it is automatically connected to the local layer. The connection information is stored in the layer's properties.

### Synchronizing Layers

There are multiple ways to synchronize layers with MapHub:

#### Using the MapHub Browser

1. In the MapHub Browser, right-click on a connected map
2. Select "Synchronize"
3. The plugin will automatically determine the synchronization direction based on the status

#### Using the Layer Context Menu

1. In the QGIS layer panel, right-click on a connected layer
2. Select "Synchronize with MapHub" or one of the specific actions:
   - "Upload to MapHub" (for locally modified layers)
   - "Update from MapHub" (for layers with remote changes)
   - "Upload Style to MapHub" (for layers with local style changes)
   - "Download Style from MapHub" (for layers with remote style changes)
   - "Resolve Style Conflict" (for layers with style changes on both sides)

#### Using the Synchronize Button

1. Click the "Synchronize Layers with MapHub" button in the MapHub toolbar
2. In the dialog that appears, select the layers you want to synchronize
3. Click "Synchronize Selected"

The plugin will automatically determine the appropriate synchronization action based on each layer's status.

### Connecting Layers to MapHub

You can connect existing layers to MapHub using the Synchronize Layers dialog:

1. Click the "Synchronize Layers with MapHub" button in the MapHub toolbar
2. In the dialog, non-connected layers appear in the "LAYERS TO CONNECT" section
3. Select the checkboxes next to the layers you want to connect
4. Click "Synchronize Selected"
5. If no folder is associated with the current project, you'll be prompted to select a destination folder
6. The selected layers will be uploaded to MapHub and automatically connected

This allows you to connect existing layers to MapHub without having to download them first, and you can connect multiple layers in a single operation.

### Synchronizing Folders

You can synchronize all maps in a folder at once:

1. In the MapHub Browser, right-click on a folder
2. Select "Synchronize Folder"
3. In the dialog that appears:
   - All maps in the folder are listed with their synchronization status
   - For maps already connected to local layers, you can see their status and appropriate actions
   - For maps not connected to local layers, you can download them
   - For maps with style conflicts, you can choose to keep local style or use remote style
4. Select the maps you want to synchronize using the checkboxes
5. Click "Synchronize Selected"

The folder synchronization dialog provides batch operations for multiple maps, making it easy to keep entire folders in sync with your local QGIS project.

### Disconnecting Layers

There are two ways to disconnect a layer from MapHub:

#### Using the MapHub Browser

1. In the MapHub Browser, right-click on a connected map
2. Select "Disconnect from Layer"
3. Confirm the disconnection

#### Using the Layer Context Menu

1. In the QGIS layer panel, right-click on a connected layer
2. Select "Disconnect from MapHub"
3. Confirm the disconnection

Disconnecting a layer does not delete the layer or the file, it only removes the connection information from the layer properties.

## Technical Details

### MapHubSyncManager

The `MapHubSyncManager` class provides the core functionality for layer-based synchronization:

- `get_connected_layers()`: Get all layers connected to MapHub
- `find_layer_by_map_id(map_id)`: Find a layer by its MapHub map ID
- `get_layer_sync_status(layer)`: Get synchronization status for a layer
- `synchronize_layer(layer, direction, style_only=False)`: Synchronize a layer with its MapHub counterpart, with option to only sync style changes
- `connect_layer(layer, map_id, folder_id, workspace_id, local_path)`: Connect a layer to a MapHub map
- `disconnect_layer(layer)`: Disconnect a layer from MapHub

### MapHubLayerDecorator

The `MapHubLayerDecorator` class adds visual indicators to QGIS layers in the layer panel:

- Implements the Singleton pattern to ensure only one instance exists throughout the plugin's lifecycle
- `get_instance(iface)`: Returns the singleton instance of the decorator
- `update_layer_icons()`: Updates layer icons with MapHub status indicators
- Adds a single indicator to each layer connected to MapHub
- Uses different icons for different synchronization statuses
- Automatically updates when layers are added or removed

### MapHubLayerMenuProvider

The `MapHubLayerMenuProvider` class adds MapHub-specific actions to the layer context menu:

- Adds "Synchronize with MapHub" action to connected layers
- Adds status-specific actions based on the layer's synchronization status
- Provides batch operations for multiple selected layers

### Integration with MapBrowserDockWidget

The `MapBrowserDockWidget` class has been enhanced to support layer-based synchronization:

- Shows visual indicators for connected maps
- Provides synchronization actions in the context menu
- Refreshes the display after synchronization actions

### Synchronization Dialogs

#### Layer Synchronization Dialog

The `SynchronizeLayersDialog` provides a comprehensive interface for batch synchronization of layers:

- Shows all layers in the project, not just connected ones
- Displays non-connected layers in a dedicated "LAYERS TO CONNECT" section with enabled checkboxes
- Shows separate columns for local and remote status with appropriate icons
- Provides style resolution options for layers with style conflicts
- Includes Select All/None buttons for easier batch operations
- Automatically determines the appropriate synchronization action for each layer
- Allows connecting existing layers to MapHub by selecting their checkboxes and clicking "Synchronize Selected"

#### Confirmation Dialog

The `ConfirmSyncDialog` provides a simple confirmation interface for single layer synchronization:

- Shows when synchronizing a single layer via the context menu
- Displays the layer name and the action to be performed
- Requires explicit confirmation before proceeding with synchronization
- Provides a consistent user experience for all synchronization operations

#### Folder Synchronization Dialog

The `FolderSyncDialog` provides a comprehensive interface for synchronizing all maps in a folder:

- Shows all maps in a folder with their synchronization status
- Handles both connected and non-connected maps
- Provides style resolution options for maps with style conflicts
- Allows downloading maps that aren't connected to local layers
- Includes "Select All" and "Select None" buttons for batch operations

### Integration with Main Plugin

The synchronization components are integrated into the main plugin:

- The `MapHubPlugin` class initializes the layer decorator and menu provider
- The plugin automatically updates layer icons when layers are added or removed
- The "Synchronize Layers with MapHub" button is added to the MapHub toolbar
- All components are properly cleaned up when the plugin is unloaded

### Centralized Error Handling

The plugin uses a centralized error handling system:

- The `ErrorManager` class provides standardized error handling across the plugin
- API exceptions are handled with appropriate messages based on status code
- Detailed error information is available for debugging
- All error messages follow a consistent format and style

### Status Icon Management

The plugin uses a centralized status icon management system:

- The `StatusIconManager` class provides consistent status icons and tooltips
- Icons are reused across different components (layer panel, dialogs, etc.)
- Status tooltips provide clear information about synchronization status
- The singleton pattern ensures consistent icon usage throughout the plugin

### Progress Reporting

The plugin provides progress reporting for long-running operations:

- The `ProgressDialog` class provides a reusable progress dialog
- Batch operations show progress with cancellation options
- Users receive clear feedback during synchronization operations
- Error handling is integrated with progress reporting

### Test Scripts

The plugin includes several test scripts to verify the functionality:

- `test_sync_manager.py`: Tests the core synchronization functionality
- `test_style_detection.py`: Tests the on-demand style change detection
- `test_ui_integration.py`: Tests the UI integration of the synchronization features

These scripts can be run from the QGIS Python console to verify that the synchronization features are working correctly.

## Future Enhancements

In future versions, the plugin will provide:

- Enhanced visual indicators with more detailed status information
- More advanced style conflict resolution options with preview capabilities
- Recursive synchronization options for nested folders
- Automatic background synchronization with notification system
- Integration with version control systems for more advanced workflows
- ~~Progress reporting for long-running synchronization operations~~ (Implemented in version 0.7)
- Conflict detection and resolution for concurrent edits

## Migrating from the Old Approach

The old git-like synchronization approach (clone/pull/push) is still available but is being phased out in favor of the new layer-based approach. Users are encouraged to use the new approach for better transparency and control.

Starting from version 0.7, the old git-like methods (clone, pull, push) show deprecation warnings to guide users toward the new layer-based approach. These methods will be removed in a future version.

If you have existing projects using the old approach, you can continue to use it for now, but all new features will be focused on the layer-based approach. We recommend migrating to the new approach as soon as possible to take advantage of the improved features and user experience.