# Layer Connection Refactoring in MapHub QGIS Plugin

## Overview

The layer connection functionality in the MapHub QGIS Plugin has been refactored to centralize the logic in the `MapHubSyncManager` class. This refactoring improves code maintainability, reduces duplication, and ensures consistent behavior across the plugin.

## Changes Made

### 1. New Method in MapHubSyncManager

A new `connect_layer` method has been added to the `MapHubSyncManager` class:

```
# In utils/sync_manager.py

def connect_layer(self, layer, map_id, folder_id, workspace_id, local_path):
    """
    Connect a layer to a MapHub map.
    
    Args:
        layer: The QGIS layer
        map_id: The MapHub map ID
        folder_id: The MapHub folder ID
        workspace_id: The MapHub workspace ID
        local_path: The local file path
    """
    # Store MapHub connection information in layer properties
    layer.setCustomProperty("maphub/map_id", str(map_id))
    layer.setCustomProperty("maphub/folder_id", str(folder_id))
    layer.setCustomProperty("maphub/workspace_id", str(workspace_id))
    layer.setCustomProperty("maphub/last_sync", datetime.now().isoformat())
    layer.setCustomProperty("maphub/local_path", local_path)
    
    self.iface.messageBar().pushInfo("MapHub", f"Layer '{layer.name()}' connected to MapHub.")
```

### 2. Refactored map_operations.py

The `download_map` function in `map_operations.py` has been updated to use the new `connect_layer` method:

Before refactoring, the code directly set custom properties on the layer:

```
# Store MapHub connection information in layer properties
layer.setCustomProperty("maphub/map_id", str(map_data['id']))
layer.setCustomProperty("maphub/folder_id", str(map_data.get('folder_id', '')))
layer.setCustomProperty("maphub/workspace_id", str(map_data.get('workspace_id', '')))
layer.setCustomProperty("maphub/last_sync", datetime.now().isoformat())
layer.setCustomProperty("maphub/local_path", file_path)
```

After refactoring, the code uses the centralized method:

```
# Connect the layer to MapHub
from .sync_manager import MapHubSyncManager
sync_manager = MapHubSyncManager(iface)
sync_manager.connect_layer(
    layer,
    map_data['id'],
    map_data.get('folder_id', ''),
    map_data.get('workspace_id', ''),
    file_path
)
```

### 3. Refactored UploadMapDialog.py

The `upload_map` method in `UploadMapDialog.py` has been updated to use the new `connect_layer` method:

Before refactoring, the code directly set custom properties on the layer:

```
# Set the layer's custom properties to connect it to MapHub
layer.setCustomProperty("maphub/map_id", str(map_id))
layer.setCustomProperty("maphub/folder_id", str(self.selected_folder_id))
layer.setCustomProperty("maphub/workspace_id", str(result.get('workspace_id', '')))
layer.setCustomProperty("maphub/last_sync", datetime.now().isoformat())
layer.setCustomProperty("maphub/local_path", source_path)
```

After refactoring, the code uses the centralized method:

```
# Connect the layer to MapHub
from ...utils.sync_manager import MapHubSyncManager
sync_manager = MapHubSyncManager(self.iface)
sync_manager.connect_layer(
    layer,
    map_id,
    self.selected_folder_id,
    result.get('workspace_id', ''),
    source_path
)
```

### 4. Updated Documentation

The documentation in `docs/layer_sync.md` has been updated to include the new `connect_layer` method in the `MapHubSyncManager` class.

### 5. Added Test Script

A new test script `test_layer_connection.py` has been created to verify the refactored functionality. The script tests both connecting and disconnecting a layer using the `MapHubSyncManager`.

## Benefits

- **Centralized Logic**: All layer connection logic is now in one place, making it easier to maintain and update.
- **Reduced Duplication**: The same code is no longer duplicated across multiple files.
- **Consistent Behavior**: All parts of the plugin now use the same method to connect layers, ensuring consistent behavior.
- **Improved Testability**: The centralized logic is easier to test, as demonstrated by the new test script.

## Future Enhancements

- **Connection Events**: Add events or signals that can be triggered when a layer is connected or disconnected.
- **Batch Connection**: Add support for connecting multiple layers at once.
- **Connection Validation**: Add validation to ensure that the connection information is valid before connecting a layer.