# Fix for Scenario 2: Runtime Error in MapHub Plugin

## Issues Fixed

This update addresses two issues in the MapHub QGIS Plugin:

1. **Issue 1: Downloaded layers not created as MapHubPluginLayer**
   - In `map_operations.py`, the `connect_layer` parameter was set to `False` when downloading maps, which prevented the creation of MapHubPluginLayer instances.
   - Fixed by changing the parameter to `True` to ensure proper layer creation.

2. **Issue 2: Runtime error when loading and synchronizing folders**
   - Error message: `RuntimeError: wrapped C/C++ object of type QgsVectorLayer has been deleted`
   - This occurred in the `load_and_sync_folder` function when trying to access layers that had been removed during the synchronization process.
   - Fixed by:
     - Creating a copy of the connected layers list to avoid modification issues during iteration
     - Checking if each layer is still valid and in the project before attempting to synchronize it
     - Adding safer error handling to gracefully handle cases where layer objects have been deleted

## Implementation Details

### Fix for Issue 1

In `map_operations.py`, changed:
```python
connect_layer=False  # Ensure the layer is connected
```
to:
```python
connect_layer=True  # Ensure the layer is connected
```

This ensures that downloaded maps are properly created as MapHubPluginLayer instances.

### Fix for Issue 2

In `map_operations.py`, modified the `load_and_sync_folder` function to:

1. Create a copy of the connected layers list:
   ```python
   # Create a copy of the connected layers list to avoid issues with deleted layers
   layers_to_sync = connected_layers.copy()
   ```

2. Check if each layer is still valid before synchronizing:
   ```python
   # Check if the layer is still valid and in the project
   if not QgsProject.instance().mapLayer(layer.id()):
       continue
   ```

3. Add safer error handling:
   ```python
   try:
       # Try to get the layer name safely
       layer_name = layer.name() if hasattr(layer, 'name') else "Unknown"
       errors.append(f"Error synchronizing layer {layer_name}: {str(e)}")
   except RuntimeError:
       # Handle case where layer object has been deleted
       errors.append(f"Error synchronizing layer (layer was deleted): {str(e)}")
   ```

## Testing

To verify the fixes:

1. **Test Issue 1**: Download a map from MapHub and verify it's created as a MapHubPluginLayer
   - Check the layer type in the Python console: `print(type(iface.activeLayer()).__name__)`
   - Should print "MapHubPluginLayer"

2. **Test Issue 2**: Load a folder with multiple maps and verify no runtime errors occur
   - Drag and drop a folder from the MapHub Browser to the map canvas
   - Verify that all maps are loaded and synchronized without errors