	# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TileLoaderDialog
                                 A QGIS plugin
 Download georeferenced parts of tile map servives
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2022-05-13
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Pavel Pereverzev
        email                : pasha004@yandex.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import math 
import requests 
import json
import time 

from osgeo import gdal, osr
from PIL import Image
Image.MAX_IMAGE_PIXELS = None
from concurrent.futures import ThreadPoolExecutor, as_completed, wait

from PyQt5.QtWidgets import QWidget, QComboBox, QLineEdit, QGridLayout, QLabel, QPushButton, \
    QMessageBox, QProgressBar, QCheckBox, QSpinBox, QFileDialog

from PyQt5 import QtCore
from PyQt5.QtGui import *
from PyQt5.QtCore import *

from qgis._core import *
from qgis.utils import iface
from qgis.gui import QgsMapToolEmitPoint, QgsRubberBand, QgsMapTool
from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsGeometry, QgsPointXY, QgsRasterLayer
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets

current_folder = (os.path.dirname(os.path.realpath(__file__)))
cfg_file = os.path.join(current_folder, 'cfg.json')
global_counter = 0

headers = {
    'accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9',
    'User-Agent': 'M',
    'accept-encoding': 'gzip, deflate',
    'accept-language': 'ru-RU,ru;q=0.9,en-US;q=0.8,en;q=0.7,uk;q=0.6'
}
out_format = 'jpeg'
large_pic_warning_message = 'Image dimensions will be:\n • width: {}\n • height: {}\n\nFile size may be too big. To reduce size you may change zoom level.\n\nAre you sure you want to continue?'

# I had to remove OSM url due to their request. 
# You are free to edit this dict to get tiles from another sources
dict_sources = {
    "Google BaseMap": {"url":"https://mt1.google.com/vt/lyrs=m&x={0}&y={1}&z={2}", "zmax":21},
    "Google Terrain": {"url":"https://mt1.google.com/vt/lyrs=p&x={0}&y={1}&z={2}", "zmax":20},
    "Google Traffic": {"url":"https://mt1.google.com/vt?lyrs=h@159000000,traffic|seconds_into_week:-1&style=3&x={0}&y={1}&z={2}", "zmax":20},
    "Google Satellite": {"url":"https://mt1.google.com/vt/lyrs=s&x={0}&y={1}&z={2}", "zmax":20},
    "Google Hybrid": {"url":"https://mt1.google.com/vt/lyrs=y&x={0}&y={1}&z={2}", "zmax":20},
    "ESRI BaseMap": {"url":"https://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{2}/{1}/{0}", "zmax":20},
    "ESRI Terrain": {"url":"https://server.arcgisonline.com/ArcGIS/rest/services/World_Terrain_Base/MapServer/tile/{2}/{1}/{0}", "zmax":20},
    "ESRI Satellite": {"url":"https://services.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{2}/{1}/{0}", "zmax":20},
    "Yandex BaseMap": {"url":"https://core-renderer-tiles.maps.yandex.net/tiles?l=map&v=21.07.13-0-b210701140430&x={0}&y={1}&z={2}&scale=1&lang=ru_RU", "zmax":21},
    "Yandex Satellite": {"url":"https://core-sat.maps.yandex.net/tiles?l=sat&v=3.1105.0&x={0}&y={1}&z={2}&scale=1.5&lang=ru_RU", "zmax":21},
    "Bing BaseMap": {"url":"https://t0.ssl.ak.dynamic.tiles.virtualearth.net/comp/ch/{0}?mkt=ru-RU&it=G,LC,BX,RL&shading=t&n=z&og=1852&cstl=vbp2&o=jpeg", "zmax":19},
    "Bing Satellite": {"url":"https://t1.ssl.ak.tiles.virtualearth.net/tiles/a{0}.jpeg?g=12225&n=z&prx=1", "zmax":18},
    "Bing Hybrid": {"url":"https://t1.ssl.ak.dynamic.tiles.virtualearth.net/comp/ch/{0}?mkt=ru-RU&it=A,G,RL&shading=t&n=z&og=1852&o=jpeg", "zmax":18},
    "MapZen": {"url":"https://s3.amazonaws.com/elevation-tiles-prod/terrarium/{2}/{0}/{1}.png", "zmax":18},
}


def write_cfg(folder, zoom, source):
    # writing config
    config = {
        "last_folder":folder, 
        "last_zoom":zoom, 
        "last_source":source
    }
    with open(cfg_file, "w", encoding = 'utf-8') as d:
        json.dump(config, d, indent=4, ensure_ascii=False)
    return


def read_cfg():
    # reading config
    if not os.path.isfile(cfg_file):
        write_cfg("", 17, "Google BaseMap")
    with open(cfg_file, 'r') as fp:
        data = json.load(fp)
    return data


def bing_coor_convert(tile_x, tile_y, zoom):
    # convert tile coordinates to quadkey
    quadkey = ''
    for i in range(zoom, 0, -1):
        digit = 0
        mask = 1<< (i - 1)
        if(tile_x & mask)!= 0:
            digit+=1
        if (tile_y & mask)!= 0:
            digit+=2
        quadkey+=str(digit)
    return quadkey


def bing_coor_convert_reverse(quadkey, zoom):
    # convert quadkey to tile coordinates
    tileX = tileY = 0
    levelOfDetail = len(quadkey)
    for i in range(zoom, 0, -1):
        mask = 1<< (i - 1)
        curkey = quadkey[levelOfDetail - i]
        if curkey == '0':
            pass
        elif curkey == '1':
            tileX += mask
        elif curkey == '2':
            tileY += mask
        elif curkey == '3':
            tileX += mask
            tileY += mask
        else:
            pass 
    return tileX, tileY


def yandex_coor_convert(longitude, latitude):
    # convert world lon lat to MercatorProjectionYandex
    d = longitude * math.pi/180
    m = latitude *  math.pi/180
    l = 6378137
    k = 0.0818191908426 
    f = k * math.sin(m)
    h = math.tan(math.pi/4 + m/2)
    j = math.pow(math.tan(math.pi/4 + math.asin(f)/2), k)
    i = h / j
    pnt_x_m  = l*d
    pnt_y_m = l*math.log(i)

    lon = (pnt_x_m/20037508.34) * 180
    lat = (pnt_y_m/20037508.34) * 180
    lat = 180 / math.pi * (2 * math.atan(math.exp(lat * math.pi / 180)) - math.pi/ 2)
    return ( lon, lat)


def yandex_coor_convert_reverse(x, y, zoom):
    # convert Yandex tile XY to world long lat to 
    RAD_DEG = 180 / math.pi
    x = x*256.0
    y = y*256.0
    a = 6378137
    c1 = 0.00335655146887969
    c2 = 0.00000657187271079536
    c3 = 0.00000001764564338702
    c4 = 0.00000000005328478445
    z1 = (23 - zoom)
    mercX = (x * math.pow(2, z1))/53.5865938 - 20037508.342789
    mercY = 20037508.342789 - (y * math.pow(2, z1)) / 53.5865938
    g = math.pi/ 2 - 2 * math.atan(1 / math.exp(mercY / a))
    z = g + c1 * math.sin(2 * g) + c2 * math.sin(4 * g) + c3 * math.sin(6 * g) + c4 * math.sin(8 * g)
    Y =  z * RAD_DEG
    X =  mercX / a * RAD_DEG
    return(X, Y)


def getxy(longitude, latitude, zoom, tileSize):
    # get tile xy coordinates from world lon lat
    sinLatitude = math.sin(latitude * math.pi/180)
    pixelX = ((longitude + 180) / 360) * tileSize * math.pow(2, zoom)
    pixelY = (0.5 - math.log((1 + sinLatitude) / (1 - sinLatitude)) / (4 * math.pi)) * tileSize * math.pow(2, zoom)
    tileX = math.floor(pixelX / tileSize)
    tileY = math.floor(pixelY / tileSize)
    
    return(tileX, tileY)


def getxy_reverse(tilelX, tileY, zoom, tileSize):
    # convert tile XY to world coodinates
    longitude = tilelX/math.pow(2, zoom)*360 - 180
    n = math.pi-2*math.pi*tileY/math.pow(2, zoom)
    latitude = 180/math.pi* math.atan(0.5 * (math.exp(n) - math.exp(-1*n)))
    
    return(longitude, latitude)


class rband(QgsMapToolEmitPoint):
    """rubberband object"""
    def __init__(self, canvas, app):
        self.canvas = canvas
        self.app = app
        self.isEmittingPoint = False
        self.start_pos_x = None
        self.start_pos_y = None
        self.end_pos_x = None
        self.end_pos_y = None
        
        QgsMapToolEmitPoint.__init__(self, self.canvas)
        self.rubberBand = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        self.rubberBand.setColor(QColor(55,150,200,150))
        self.rubberBand.setWidth(3)
        self.rubberBand.reset()
    
    def canvasPressEvent(self, e):
        # start drawing handler
        self.startPoint = self.toMapCoordinates(e.pos())
        self.endPoint = self.startPoint
        self.isEmittingPoint = True
        self.showRect(self.startPoint, self.endPoint)
    
    def canvasMoveEvent(self, e):
        # draw rectangle on move
        if not self.isEmittingPoint:
            return
        self.endPoint = self.toMapCoordinates(e.pos())
        self.showRect(self.startPoint, self.endPoint)
    
    def canvasReleaseEvent(self, e):
        # finish rectangle, miximize main widget
        self.isEmittingPoint = False
        self.app.setWindowState(Qt.WindowNoState)
        self.app.btn_save_frame.setDisabled(False)
        self.app.activate_dl()
        self.deactivate()
    
    def showRect(self, startPoint, endPoint):
        # draw rectangle 
        self.rubberBand.reset(QgsWkbTypes.PolygonGeometry)
        if startPoint.x() == endPoint.x() or startPoint.y() == endPoint.y():
            return
        pnts = [
            (startPoint.x(), startPoint.y()),
            (startPoint.x(), endPoint.y()),
            (endPoint.x(), endPoint.y()),
            (endPoint.x(), startPoint.y()),
            (startPoint.x(), startPoint.y())
        ]
        polygon_coors = [QgsPointXY(p[0], p[1]) for p in pnts]
        geom_polygon = QgsGeometry().fromPolygonXY([polygon_coors])
        
        self.start_pos_x = min(pnts, key=lambda v: v[0])[0]
        self.start_pos_y = max(pnts, key=lambda v: v[1])[1]

        self.end_pos_x = max(pnts, key=lambda v: v[0])[0]
        self.end_pos_y = min(pnts, key=lambda v: v[1])[1]

        self.rubberBand.setToGeometry(geom_polygon)
        
    def deactivate(self):
        QgsMapTool.deactivate(self)
        self.deactivated.emit()


class MapTileLoader(QWidget):
    """ Main widget"""
    def __init__(self, parent=None):
        super(MapTileLoader, self).__init__(parent)
        self.setWindowFlags(self.windowFlags() | QtCore.Qt.WindowStaysOnTopHint)
        self.resize(400, 0)
        self.setWindowTitle("MapTileLoader")
        self.draw_tool = None 
        self.tr = None
        self.tr_reversed = None
        self.grid = QGridLayout()
        self.grid.setSpacing(10)
        self.setLayout(self.grid)
        self.work_folder = ''

        # widgets
        self.label_source = QLabel("Source:")
        self.source_cmbx = QComboBox()

        self.label_zoom = QLabel("Zoom level:")
        self.zoom_slider = QSpinBox()
        self.zoom_slider.setMinimum(1)
        self.zoom_slider.setMaximum(24)
        self.zoom_slider.setValue(11)
        
        self.label_path = QLabel("Save to:")
        self.path_line = QLineEdit()
        self.btn_folder_select = QPushButton("...")
        self.btn_folder_select.setMaximumWidth(30)
        
        self.label_add_rect = QLabel("Frame:")
        self.btn_add_rect = QPushButton("+")
        self.btn_save_frame = QPushButton("Save frame")
        self.btn_load_frame = QPushButton("Load frame")
        
        self.btn_load = QPushButton("Download")
        self.add_checkbox = QCheckBox("Add image on load")
        
        self.pbar = QProgressBar(self)
        
        # setting up interface
        self.grid.addWidget(self.label_source,       0, 1, 1, 5)
        self.grid.addWidget(self.source_cmbx,        1, 1, 1, 5)

        self.grid.addWidget(self.label_zoom,         2, 1, 1, 5)
        self.grid.addWidget(self.zoom_slider,        3, 1, 1, 5)
        
        self.grid.addWidget(self.label_add_rect,     4, 1, 1, 1)
        self.grid.addWidget(self.btn_add_rect,       5, 1, 1, 1)
        self.grid.addWidget(self.btn_save_frame,     5, 2, 1, 2)
        self.grid.addWidget(self.btn_load_frame,     5, 4, 1, 2)

        self.grid.addWidget(self.label_path,         6, 1, 1, 5)
        self.grid.addWidget(self.path_line,          7, 1, 1, 4)
        self.grid.addWidget(self.btn_folder_select,  7, 5, 1, 1)
        
        self.grid.addWidget(self.btn_load,           8, 1, 1, 5)
        self.grid.addWidget(self.add_checkbox,       9, 1, 1, 5)
        self.grid.addWidget(self.pbar,               10, 1, 1, 5)
        
        # adding map sources
        self.source_cmbx.addItems(list(dict_sources.keys()))

        # setting fucntions to widget signals
        self.source_cmbx.currentTextChanged.connect(self.set_max_zoom)
        self.btn_add_rect.clicked.connect(self.draw_rect)
        self.btn_save_frame.clicked.connect(self.save_frame)
        self.btn_load_frame.clicked.connect(self.load_frame)
        self.path_line.textChanged.connect(self.activate_dl)
        self.btn_folder_select.clicked.connect(self.select_folder)
        self.btn_load.clicked.connect(self.load)

        # disabling buttons until frame is drawn and path is printed
        self.add_checkbox.setChecked(True)
        self.btn_save_frame.setDisabled(True)
        self.btn_load.setDisabled(True)

        self.proj_transforms()
        self.set_config()
        self.show()


    def ready_message(self, info_text):
        # custom information message
        msg = QMessageBox()
        msg.information(self, "Success!", info_text)
        return


    def warning_message(self, err_text):
        # custom warning message
        msg = QMessageBox()
        msg.warning(self, "Warning", err_text)
        return
    
    def warning_question(self, w, h):
        dlg = QtWidgets.QMessageBox(self) 
        dlg.setWindowTitle('Large output size warning')
        dlg.setText(large_pic_warning_message.format(int(round(w)), int(round(h))))
        dlg.setStandardButtons(QtWidgets.QMessageBox.No | QtWidgets.QMessageBox.Yes)
        dlg.button(QtWidgets.QMessageBox.Yes).setText("Continue")
        dlg.button(QtWidgets.QMessageBox.No).setText("Cancel")
        return dlg.exec_()  
    
    
    def set_config(self):
        # set config on previous use
        config = read_cfg()
        idx_source = list(dict_sources.keys()).index(config['last_source'])
        zoom = config['last_zoom']
        folder = config['last_folder']
        self.source_cmbx.setCurrentIndex(idx_source)
        self.zoom_slider.setValue(zoom)
        self.path_line.setText(folder)

    
    def set_max_zoom(self):
        # set maximum tile source level
        current_source = self.source_cmbx.currentText()
        max_zoom = dict_sources[current_source]['zmax']
        self.zoom_slider.setMaximum(max_zoom)
        
    
    def select_folder(self):
        # seelct folder for output rasters
        result = QFileDialog.getExistingDirectory(self, 'Select Folder')
        if result:
            self.path_line.setText(result)
        return
    

    def activate_dl(self):
        # check if rectangle is drawn and path is pointed
        # if yes then make download button enabled
        cur_path = self.path_line.text()
        if not cur_path or not self.draw_tool:
            self.btn_load.setDisabled(True)
        else:
            self.btn_load.setDisabled(False)
        return

    
    def check_folder(self):
        # check if folder exists
        path = self.path_line.text()
        if os.path.isdir(path):
            self.work_folder = path
            return True
        return False


    def draw_rect(self):
        # running rect tool
        if self.draw_tool:
            self.draw_tool.rubberBand.reset()
            self.draw_tool = None
        self.draw_tool = rband(iface.mapCanvas(), self)
        iface.mapCanvas().setMapTool(self.draw_tool) 
        self.showMinimized()
    
    
    def proj_transforms(self):
        # set crs transforms
        current_crs = QgsProject.instance().crs() 
        sourceCrs = QgsCoordinateReferenceSystem(current_crs)
        destCrs = QgsCoordinateReferenceSystem(4326)
        destCrs_3857 = QgsCoordinateReferenceSystem(3857)
        self.tr = QgsCoordinateTransform(sourceCrs, destCrs, QgsProject.instance())
        self.tr_3857 = QgsCoordinateTransform(destCrs, destCrs_3857, QgsProject.instance())
        self.tr_reversed = QgsCoordinateTransform(destCrs, sourceCrs , QgsProject.instance())


    def rband_coors(self):
        # transforming rectangle coordinates into project crs
        if self.draw_tool:
            pnt_start = QgsPointXY(self.draw_tool.start_pos_x, self.draw_tool.start_pos_y)
            pnt_end = QgsPointXY(self.draw_tool.end_pos_x, self.draw_tool.end_pos_y) 
            pnt_geom_start = QgsGeometry().fromPointXY(pnt_start)
            pnt_geom_end  = QgsGeometry().fromPointXY(pnt_end)
            
            pnt_geom_start.transform(self.tr)
            pnt_geom_end.transform(self.tr)
            return pnt_geom_start, pnt_geom_end

    
    def load(self):
        # initializing download
        if self.draw_tool:
            self.pnt_geom_start, self.pnt_geom_end = self.rband_coors()
            self.get_raster(self.pnt_geom_start, self.pnt_geom_end, self.zoom_slider.value())


    def download_image(self, session, url, path_file):
        # tile load
        attempts = 0
        while True:
            if attempts == 10:
                break 
            try:
                request = session.get(url, headers=headers, timeout=2, verify=False)
                if request.status_code == 200:
                    with open(path_file, 'wb') as out:
                        out.write(request.content)
                    break
                else:
                    return path_file

            except:
                attempts+=1
                time.sleep(1)
        return path_file

    
    def fetch(self, session, url, path_file):
        # tile download
        self.download_image(session, url, path_file)
        return 

   
    def stitch_image(self, list_images, img):
        # open each tile and collect them on PIL canvas
        for img_data in list_images:
            off_x = img_data[0]
            off_y = img_data[1]
            t_height = img_data[2]
            img_path = img_data[3]
            if os.path.isfile(img_path):
                im_saved = Image.open(img_path)
                img.paste(im_saved, (off_x, t_height-off_y))
        return 


    def get_raster(self, pnt_start, pnt_end, zoom):
        # get tiles and stitch them into single georeferenced image
        tileSize = 256
        img_size_limit = 20000
        current_source = self.source_cmbx.currentText()

        folder_valid = self.check_folder()
        if not folder_valid:
            self.warning_message("Folder does not exist")
            return
        
        # output filename will contain only datetime instead of coordinates
        current_datetime = time.localtime()
        str_datetime = time.strftime("%Y-%m-%d_%H-%M-%S", current_datetime)
        f_name_pref = '{}'.format(str_datetime)

        # filenames and paths
        tile_url = dict_sources[current_source]['url']
        file_pil = os.path.join(self.work_folder, 'pil_img.jpeg')
        file_pil_clip = os.path.join(self.work_folder, 'pil_img_clip.jpeg')
        final_file_out = os.path.join(self.work_folder,'{}_{}.tif'.format(current_source, f_name_pref))


        lon_start, lat_start = pnt_start.asPoint().x(), pnt_start.asPoint().y()
        lon_end, lat_end = pnt_end.asPoint().x(), pnt_end.asPoint().y()

        # copies of rband points to convert them into 3857
        pnt_copy_start = QgsGeometry().fromPointXY(QgsPointXY(pnt_start.asPoint().x(), pnt_start.asPoint().y()))
        pnt_copy_end = QgsGeometry().fromPointXY(QgsPointXY(pnt_end.asPoint().x(), pnt_end.asPoint().y()))
        pnt_copy_start.transform(self.tr_3857)
        pnt_copy_end.transform(self.tr_3857)

        # 3857 coordinates for georeferencing picture
        # there are two reasons for that:
        # 1 - result image can be opened with original appearance in viewers/editors
        # 2 - raster layer will be displayed in better quality rather than
        # if it was georeferenced with degrees (EPSG:4326)
        lon_start_3857, lat_start_3857 = pnt_copy_start.asPoint().x(), pnt_copy_start.asPoint().y()
        lon_end_3857, lat_end_3857 = pnt_copy_end.asPoint().x(), pnt_copy_end.asPoint().y()
        
        # yandex is projected in 3395
        if 'yandex' in current_source.lower():
            lon_start, lat_start = yandex_coor_convert(lon_start, lat_start)
            lon_end, lat_end = yandex_coor_convert(lon_end, lat_end)

        # coordinates of tiles
        xy_start = getxy(lon_start, lat_start, zoom, tileSize)
        xy_end = getxy(lon_end, lat_end, zoom, tileSize)

        # yandex
        if 'yandex' in current_source.lower():
            dt = yandex_coor_convert_reverse(xy_start[0], xy_start[1], zoom)
            et = yandex_coor_convert_reverse(xy_end[0]+1, xy_end[1]+1, zoom)

        # degrees coordinates of tiles
        dt = getxy_reverse(xy_start[0], xy_start[1], zoom, 256)
        et = getxy_reverse(xy_end[0]+1, xy_end[1]+1, zoom, 256)
        
        # width and height of rband in degrees and 
        # ratios to get crop coodrdinates 
        width_degrees = et[0] - dt[0]
        height_degrees = dt[1] - et[1]
        diff_w_start = (lon_start - dt[0])/width_degrees
        diff_w_end = (lon_end - dt[0])/width_degrees
        diff_h_start = (dt[1] - lat_start)/height_degrees
        diff_h_end = (dt[1]- lat_end)/height_degrees

        # min and max tile coordinates
        xmin = xy_start[0]
        ymin = xy_start[1]
        xmax = xy_end[0]
        ymax = xy_end[1]
        
        # total width and height of request field in pixels
        total_width = (xmax+1 - xmin)*tileSize
        total_height = (ymax+1 - ymin)*tileSize

        # coordiantes for cropping
        crop_bounds= [
            total_width*diff_w_start,
            total_height*diff_h_start,
            total_width*diff_w_end,
            total_height*diff_h_end
        ]

        # after image is cropped there will be new width and height
        cropped_width = total_width*diff_w_end - total_width*diff_w_start
        cropped_height = total_height*diff_h_end - total_height*diff_h_start

        # image size is greater than 20000 pixels, ask user
        # if he is ready for that
        if any(dim_img > img_size_limit for dim_img in (cropped_height, cropped_width)):
            answer = self.warning_question(cropped_width,cropped_height)
            if answer == QMessageBox.No:
                return

        # blank image template
        new_im = Image.new('RGB', (total_width, total_height))

        # looping tile net
        offset_y = tileSize
        urls_stack = []
        for y_step in range(ymax, ymin-1, -1):
            offset_x = 0
            for x_step in range(xmin, xmax+1, 1):
                if 'bing' in current_source.lower():
                    quadkey = bing_coor_convert(x_step, y_step, zoom)
                    lnk_download = tile_url.format(quadkey)
                else:
                    lnk_download = tile_url.format(x_step, y_step, zoom)
                file_out = os.path.join(self.work_folder, 'out_{}_{}.jpeg'.format(x_step, y_step))
                urls_stack.append([lnk_download, offset_x, offset_y, total_height, file_out])
                offset_x += tileSize
            offset_y += tileSize
            offset_x = 0

        # progress bar counts
        self.step_pbar = 100/len(urls_stack) if urls_stack else 100
        self.step_counter = 0
       
        # multithread thing to download all tiles
        session = requests.Session()
        with ThreadPoolExecutor(None) as executor:
            futures = [executor.submit(self.fetch, session, url[0], url[4])  for url in urls_stack]
            for _ in as_completed(futures):
                self.step_counter+=self.step_pbar
                if abs(self.pbar.value())-int(self.step_counter)==-1:
                    self.pbar.setValue(int(self.step_counter))
                    QtCore.QCoreApplication.processEvents()
        
        # stitching tiles
        self.stitch_image([[url[1], url[2], url[3], url[4]] for url in urls_stack], new_im)

        # save raw image
        new_im.save(file_pil, quality=100)

        # crop image to rband bounds
        new_im.crop((crop_bounds[0], crop_bounds[1], crop_bounds[2], crop_bounds[3])).save(file_pil_clip, quality=100)

        # georeferencing points
        gcps = [
            gdal.GCP(lon_start_3857, lat_start_3857, 0, 0, 0),
            gdal.GCP(lon_end_3857,   lat_start_3857, 0, cropped_width, 0),
            gdal.GCP(lon_end_3857,   lat_end_3857,   0, cropped_width, cropped_height),
            gdal.GCP(lon_start_3857, lat_end_3857,   0, 0, cropped_height)
        ]

        # georeferencing
        ds = gdal.Translate(final_file_out, file_pil_clip)
        sr = osr.SpatialReference()
        sr.ImportFromEPSG(3857) 
        ds.SetGCPs(gcps, sr.ExportToWkt())
        ds = None

        # add raster to project
        if self.add_checkbox.isChecked():
            rlayer = QgsRasterLayer(final_file_out, '{}_{}'.format(current_source, f_name_pref))
            QgsProject.instance().addMapLayer(rlayer)  

        # remove all auxiliary files  
        files_to_remove = [url[4] for url in  urls_stack] + [file_pil, file_pil_clip]
        for file in files_to_remove:
            if os.path.isfile(file):
                # pass 
                os.remove(file)

        # save last map and zoom choice
        write_cfg(self.work_folder, zoom, current_source)
        self.pbar.setValue(0)
        QtCore.QCoreApplication.processEvents()
        self.ready_message("Image from {} is ready.".format(current_source))
        return        
    
    
    def save_frame(self):
        # save current drawn rectangle to *.xtnt file
        if self.draw_tool:
            self.pnt_geom_start, self.pnt_geom_end = self.rband_coors()
            name = QFileDialog.getSaveFileName(self, 'Save Frame', "", "xtnt (*.xtnt)")
            if name[0]:
                data = [(p.asPoint().x(), p.asPoint().y()) for p in [self.pnt_geom_start, self.pnt_geom_end]]
                data_str = "\n".join(["{},{}".format(p[0], p[1])  for p in data])
                with open(name[0], 'w') as wfile:
                    wfile.write(data_str)
        return

            
    def load_frame(self):
        # load rectangle coordinates from *.xtnt file
        f_open = QFileDialog.getOpenFileName(self, "Select Frame", "", "*.xtnt")
        if f_open[0]:
            lines = []
            with open(f_open[0], 'r') as rfile:
                lines = [line.rstrip() for line in rfile]
            uw_pnt = [float(p) for p in lines[0].split(',')]
            le_pnt = [float(p) for p in lines[1].split(',')]
            
            start_point = QgsPointXY(uw_pnt[0], uw_pnt[1])
            end_point = QgsPointXY(le_pnt[0], le_pnt[1])
            
            start_point_geom = QgsGeometry().fromPointXY(start_point)
            end_point_geom = QgsGeometry().fromPointXY(end_point)
            
            start_point_geom.transform(self.tr_reversed)
            end_point_geom.transform(self.tr_reversed)

            start_point_projected = start_point_geom.asPoint()
            end_point_projected = end_point_geom.asPoint()

            if self.draw_tool:
                self.draw_tool.rubberBand.setToGeometry(geom_polygon)
            else:
                self.draw_tool = rband(iface.mapCanvas(), self)
                iface.mapCanvas().setMapTool(self.draw_tool) 
                self.draw_tool.showRect(start_point_projected, end_point_projected)
                self.draw_tool.startPoint = start_point_projected
                self.draw_tool.endPoint = end_point_projected
            self.activate_dl()
        return
            
        
    def closeEvent(self, event):
        # remove all rubberbands
        if self.draw_tool:
            self.draw_tool.rubberBand.reset()
            self.draw_tool.deactivate()
            iface.mapCanvas().unsetMapTool(self.draw_tool) 
            
