# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TileLoaderDialog
                                 A QGIS plugin
 Download georeferenced parts of tile map servives
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2022-05-13
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Pavel Pereverzev
        email                : pasha004@yandex.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import math 
import requests 
import time 
import json
 
from osgeo import gdal, osr
from PIL import Image

from PyQt5.QtWidgets import QApplication, QWidget, QComboBox, QHBoxLayout, QLineEdit, QGridLayout, QLabel, QPushButton, \
    QMessageBox, QProgressBar, QCheckBox, QSpinBox, QFileDialog

from PyQt5 import QtCore
from PyQt5.QtGui import *
from PyQt5.QtCore import *

from qgis._core import *
from qgis.utils import iface
from qgis.gui import QgsMapToolEmitPoint, QgsRubberBand, QgsMapTool
from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsPoint, QgsFeature, QgsGeometry, QgsVectorLayer, QgsPointXY, QgsRasterLayer
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets


current_folder = (os.path.dirname(os.path.realpath(__file__)))
cfg_file = os.path.join(current_folder, 'cfg.json')

headers = {
    'accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.9',
    'User-Agent': 'M',
    'accept-encoding': 'gzip, deflate',
    'accept-language': 'ru-RU,ru;q=0.9,en-US;q=0.8,en;q=0.7,uk;q=0.6'
}
out_format = 'jpeg'

dict_sources = {
    "Google BaseMap": {"url":"https://mt1.google.com/vt/lyrs=m&x={0}&y={1}&z={2}", "zmax":21},
    "Google Terrain": {"url":"https://mt1.google.com/vt/lyrs=p&x={0}&y={1}&z={2}", "zmax":20},
    "Google Traffic": {"url":"https://mt1.google.com/vt?lyrs=h@159000000,traffic|seconds_into_week:-1&style=3&x={0}&y={1}&z={2}", "zmax":20},
    "Google Satellite": {"url":"https://khms0.google.com/kh/v=923?x={0}&y={1}&z={2}", "zmax":20},
    "Google Hybrid": {"url":"https://mt1.google.com/vt/lyrs=y&x={0}&y={1}&z={2}", "zmax":20},
    "OSM": {"url":"http://tile.openstreetmap.org/{2}/{0}/{1}.png", "zmax":19},
    "ESRI BaseMap": {"url":"https://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{2}/{1}/{0}", "zmax":20},
    "ESRI Terrain": {"url":"https://server.arcgisonline.com/ArcGIS/rest/services/World_Terrain_Base/MapServer/tile/{2}/{1}/{0}", "zmax":20},
    "ESRI Satellite": {"url":"https://services.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{2}/{1}/{0}", "zmax":20},
    "Yandex BaseMap": {"url":"https://core-renderer-tiles.maps.yandex.net/tiles?l=map&v=21.07.13-0-b210701140430&x={0}&y={1}&z={2}&scale=1&lang=ru_RU", "zmax":21},
    "Yandex Satellite": {"url":"https://core-sat.maps.yandex.net/tiles?l=sat&v=3.1016.0&x={0}&y={1}&z={2}&scale=1.5&lang=ru_RU", "zmax":19},
    "Bing BaseMap": {"url":"https://t0.ssl.ak.dynamic.tiles.virtualearth.net/comp/ch/{0}?mkt=ru-RU&it=G,LC,BX,RL&shading=t&n=z&og=1852&cstl=vbp2&o=jpeg", "zmax":19},
    "Bing Satellite": {"url":"https://t1.ssl.ak.tiles.virtualearth.net/tiles/a{0}.jpeg?g=12225&n=z&prx=1", "zmax":18},
    "Bing Hybrid": {"url":"https://t1.ssl.ak.dynamic.tiles.virtualearth.net/comp/ch/{0}?mkt=ru-RU&it=A,G,RL&shading=t&n=z&og=1852&o=jpeg", "zmax":18},
    "MapZen": {"url":"https://s3.amazonaws.com/elevation-tiles-prod/terrarium/{2}/{0}/{1}.png", "zmax":18}
}



def write_cfg(folder, zoom, source):
    # writing config
    config = {
        "last_folder":folder, 
        "last_zoom":zoom, 
        "last_source":source
    }
    with open(cfg_file, "w", encoding = 'utf-8') as d:
        json.dump(config, d, indent=4, ensure_ascii=False)
    return


def read_cfg():
    # reading config
    if not os.path.isfile(cfg_file):
        write_cfg("", 17, "Google BaseMap")
    with open(cfg_file, 'r') as fp:
        data = json.load(fp)
    return data


def bing_coor_convert(tile_x, tile_y, zoom):
    # convert tile coordinates to quadkey
    quadkey = ''
    for i in range(zoom, 0, -1):
        digit = 0
        mask = 1<< (i - 1)
        if(tile_x & mask)!= 0:
            digit+=1
        if (tile_y & mask)!= 0:
            digit+=2
        quadkey+=str(digit)
    return quadkey


def bing_coor_convert_reverse(quadkey, zoom):
    # convert quadkey to tile coordinates
    tileX = tileY = 0
    levelOfDetail = len(quadkey)
    for i in range(zoom, 0, -1):
        mask = 1<< (i - 1)
        curkey = quadkey[levelOfDetail - i]
        if curkey == '0':
            pass
        elif curkey == '1':
            tileX += mask
        elif curkey == '2':
            tileY += mask
        elif curkey == '3':
            tileX += mask
            tileY += mask
        else:
            pass 
    return tileX, tileY


def yandex_coor_convert(longitude, latitude):
    # convert world lon lat to MercatorProjectionYandex
    d = longitude * math.pi/180
    m = latitude *  math.pi/180
    l = 6378137
    k = 0.0818191908426 
    f = k * math.sin(m)
    h = math.tan(math.pi/4 + m/2)
    j = math.pow(math.tan(math.pi/4 + math.asin(f)/2), k)
    i = h / j
    pnt_x_m  = l*d
    pnt_y_m = l*math.log(i)

    lon = (pnt_x_m/20037508.34) * 180
    lat = (pnt_y_m/20037508.34) * 180
    lat = 180 / math.pi * (2 * math.atan(math.exp(lat * math.pi / 180)) - math.pi/ 2)
    return ( lon, lat)


def yandex_coor_convert_reverse(x, y, zoom):
    # convert Yandex tile XY to world long lat to 
    RAD_DEG = 180 / math.pi
    x = x*256.0
    y = y*256.0
    a = 6378137
    c1 = 0.00335655146887969
    c2 = 0.00000657187271079536
    c3 = 0.00000001764564338702
    c4 = 0.00000000005328478445
    z1 = (23 - zoom)
    mercX = (x * math.pow(2, z1))/53.5865938 - 20037508.342789
    mercY = 20037508.342789 - (y * math.pow(2, z1)) / 53.5865938
    g = math.pi/ 2 - 2 * math.atan(1 / math.exp(mercY / a))
    z = g + c1 * math.sin(2 * g) + c2 * math.sin(4 * g) + c3 * math.sin(6 * g) + c4 * math.sin(8 * g)
    Y =  z * RAD_DEG
    X =  mercX / a * RAD_DEG
    return(X, Y)


def getxy(longitude, latitude, zoom, tileSize):
    # get tile xy coordinates from world lon lat
    sinLatitude = math.sin(latitude * math.pi/180)
    pixelX = ((longitude + 180) / 360) * tileSize * math.pow(2, zoom)
    pixelY = (0.5 - math.log((1 + sinLatitude) / (1 - sinLatitude)) / (4 * math.pi)) * tileSize * math.pow(2, zoom)
    tileX = math.floor(pixelX / tileSize)
    tileY = math.floor(pixelY / tileSize)
    
    return(tileX, tileY)


def getxy_reverse(tilelX, tileY, zoom, tileSize):
    # convert tile XY to world coodinates
    longitude = tilelX/math.pow(2, zoom)*360 - 180
    n = math.pi-2*math.pi*tileY/math.pow(2, zoom)
    latitude = 180/math.pi* math.atan(0.5 * (math.exp(n) - math.exp(-1*n)))
    
    return(longitude, latitude)


class rband(QgsMapToolEmitPoint):
    """rubberband object"""
    def __init__(self, canvas, app):
        self.canvas = canvas
        self.app = app
        self.isEmittingPoint = False
        self.start_pos_x = None
        self.start_pos_y = None
        self.end_pos_x = None
        self.end_pos_y = None
        
        QgsMapToolEmitPoint.__init__(self, self.canvas)
        self.rubberBand = QgsRubberBand(self.canvas, True)
        self.rubberBand.setColor(QColor(55,150,200,150))
        self.rubberBand.setWidth(3)
        self.rubberBand.reset()
    
    def canvasPressEvent(self, e):
        # start drawing handler
        self.startPoint = self.toMapCoordinates(e.pos())
        self.endPoint = self.startPoint
        self.isEmittingPoint = True
        self.showRect(self.startPoint, self.endPoint)
    
    def canvasMoveEvent(self, e):
        # draw rectangle on move
        if not self.isEmittingPoint:
            return
        self.endPoint = self.toMapCoordinates(e.pos())
        self.showRect(self.startPoint, self.endPoint)
    
    def canvasReleaseEvent(self, e):
        # finish rectangle, miximize main widget
        self.isEmittingPoint = False
        canvasGeom = self.rubberBand.asGeometry()
        self.app.setWindowState(Qt.WindowNoState)
        self.app.btn_save_frame.setDisabled(False)
        self.app.activate_dl()
        self.deactivate()
    
    def showRect(self, startPoint, endPoint):
        # draw rectangle 
        self.rubberBand.reset(QgsWkbTypes.PolygonGeometry)
        if startPoint.x() == endPoint.x() or startPoint.y() == endPoint.y():
            return
        pnts = [
            (startPoint.x(), startPoint.y()),
            (startPoint.x(), endPoint.y()),
            (endPoint.x(), endPoint.y()),
            (endPoint.x(), startPoint.y()),
            (startPoint.x(), startPoint.y())
        ]
        polygon_coors = [QgsPointXY(p[0], p[1]) for p in pnts]
        geom_polygon = QgsGeometry().fromPolygonXY([polygon_coors])
        
        self.start_pos_x = min(pnts, key=lambda v: v[0])[0]
        self.start_pos_y = max(pnts, key=lambda v: v[1])[1]

        self.end_pos_x = max(pnts, key=lambda v: v[0])[0]
        self.end_pos_y = min(pnts, key=lambda v: v[1])[1]

        self.rubberBand.setToGeometry(geom_polygon)
        
    def deactivate(self):
        QgsMapTool.deactivate(self)
        self.deactivated.emit()


class MapTileLoader(QWidget):
    """ Main widget"""
    def __init__(self, parent=None):
        super(MapTileLoader, self).__init__(parent)
        self.setWindowFlags(self.windowFlags() | QtCore.Qt.WindowStaysOnTopHint)
        self.resize(400, 0)
        self.setWindowTitle("MapTileLoader")
        self.draw_tool = None 
        self.tr = None
        self.tr_reversed = None
        self.grid = QGridLayout()
        self.grid.setSpacing(10)
        self.setLayout(self.grid)
        self.work_folder = ''

        # widgets
        self.label_source = QLabel("Source:")
        self.source_cmbx = QComboBox()

        self.label_zoom = QLabel("Zoom level:")
        self.zoom_slider = QSpinBox()
        self.zoom_slider.setMinimum(1)
        self.zoom_slider.setMaximum(24)
        self.zoom_slider.setValue(11)
        
        self.label_path = QLabel("Save to:")
        self.path_line = QLineEdit()
        self.btn_folder_select = QPushButton("...")
        self.btn_folder_select.setMaximumWidth(30)
        
        self.label_add_rect = QLabel("Frame:")
        self.btn_add_rect = QPushButton("+")
        self.btn_save_frame = QPushButton("Save frame")
        self.btn_load_frame = QPushButton("Load frame")
        
        self.btn_load = QPushButton("Download")
        self.add_checkbox = QCheckBox("Add image on load")
        
        self.pbar = QProgressBar()
        
        # setting up interface
        self.grid.addWidget(self.label_source,       0, 1, 1, 5)
        self.grid.addWidget(self.source_cmbx,        1, 1, 1, 5)

        self.grid.addWidget(self.label_zoom,         2, 1, 1, 5)
        self.grid.addWidget(self.zoom_slider,        3, 1, 1, 5)
        
        self.grid.addWidget(self.label_add_rect,     4, 1, 1, 1)
        self.grid.addWidget(self.btn_add_rect,       5, 1, 1, 1)
        self.grid.addWidget(self.btn_save_frame,     5, 2, 1, 2)
        self.grid.addWidget(self.btn_load_frame,     5, 4, 1, 2)

        self.grid.addWidget(self.label_path,         6, 1, 1, 5)
        self.grid.addWidget(self.path_line,          7, 1, 1, 4)
        self.grid.addWidget(self.btn_folder_select,  7, 5, 1, 1)
        
        self.grid.addWidget(self.btn_load,           8, 1, 1, 5)
        self.grid.addWidget(self.add_checkbox,       9, 1, 1, 5)
        self.grid.addWidget(self.pbar,               10, 1, 1, 5)
        
        # adding map sources
        self.source_cmbx.addItems(list(dict_sources.keys()))

        # setting fucntions to widget signals
        self.source_cmbx.currentTextChanged.connect(self.set_max_zoom)
        self.btn_add_rect.clicked.connect(self.draw_rect)
        self.btn_save_frame.clicked.connect(self.save_frame)
        self.btn_load_frame.clicked.connect(self.load_frame)
        self.path_line.textChanged.connect(self.activate_dl)
        self.btn_folder_select.clicked.connect(self.select_folder)
        self.btn_load.clicked.connect(self.load)

        # disabling buttons until frame is drawn and path is printed
        self.add_checkbox.setChecked(True)
        self.btn_save_frame.setDisabled(True)
        self.btn_load.setDisabled(True)
        
        self.proj_transforms()
        self.set_config()
        self.show()


    def warning_message(self, err_text):
        # custom warning message
        msg = QMessageBox()
        msg.warning(self, "Warning", err_text)
        return
    

    def set_config(self):
        # set config on previous use
        config = read_cfg()
        idx_source = list(dict_sources.keys()).index(config['last_source'])
        zoom = config['last_zoom']
        folder = config['last_folder']
        self.source_cmbx.setCurrentIndex(idx_source)
        self.zoom_slider.setValue(zoom)
        self.path_line.setText(folder)

    
    def set_max_zoom(self):
        # set maximum tile source level
        current_source = self.source_cmbx.currentText()
        max_zoom = dict_sources[current_source]['zmax']
        self.zoom_slider.setMaximum(max_zoom)
        
    
    def select_folder(self):
        # seelct folder for output rasters
        result = QFileDialog.getExistingDirectory(self, 'Select Folder')
        if result:
            self.path_line.setText(result)
        return
    

    def activate_dl(self):
        # check if rectangle is drawn and path is pointed
        # if yes then make download button enabled
        cur_path = self.path_line.text()
        if not cur_path or not self.draw_tool:
            self.btn_load.setDisabled(True)
        else:
            self.btn_load.setDisabled(False)
        return

    
    def check_folder(self):
        # check if folder exists
        path = self.path_line.text()
        if os.path.isdir(path):
            self.work_folder = path
            return True
        return False


    def draw_rect(self):
        # running rect tool
        if self.draw_tool:
            self.draw_tool.rubberBand.reset()
            self.draw_tool = None
        self.draw_tool = rband(iface.mapCanvas(), self)
        iface.mapCanvas().setMapTool(self.draw_tool) 
        self.showMinimized()
    
    
    def proj_transforms(self):
        current_crs = QgsProject.instance().crs() 
        sourceCrs = QgsCoordinateReferenceSystem(current_crs)
        destCrs = QgsCoordinateReferenceSystem(4326)
        self.tr = QgsCoordinateTransform(sourceCrs, destCrs, QgsProject.instance())
        self.tr_reversed = QgsCoordinateTransform(destCrs, sourceCrs , QgsProject.instance())


    def rband_coors(self):
        # transforming rectangle coordinates into project crs
        if self.draw_tool:
            pnt_start = QgsPointXY(self.draw_tool.start_pos_x, self.draw_tool.start_pos_y)
            pnt_end = QgsPointXY(self.draw_tool.end_pos_x, self.draw_tool.end_pos_y) 
            # print('old', self.draw_tool.startPoint, self.draw_tool.endPoint)
            # print('new', pnt_start, pnt_end)
            pnt_geom_start = QgsGeometry().fromPointXY(pnt_start)
            pnt_geom_end  = QgsGeometry().fromPointXY(pnt_end)
            
            pnt_geom_start.transform(self.tr)
            pnt_geom_end.transform(self.tr)
            return pnt_geom_start, pnt_geom_end

    
    def load(self):
        # initializing download
        if self.draw_tool:
            self.pnt_geom_start, self.pnt_geom_end = self.rband_coors()
            self.get_raster(self.pnt_geom_start, self.pnt_geom_end, self.zoom_slider.value())


    def get_raster(self, pnt_start, pnt_end, zoom):
        # get tiles and stitch them into one georeferenced image
        folder_valid = self.check_folder()
        if not folder_valid:
            self.warning_message("Folder does not exist")
            return

        lon_start, lat_start = pnt_start.asPoint().x(), pnt_start.asPoint().y()
        lon_end, lat_end = pnt_end.asPoint().x(), pnt_end.asPoint().y()
        f_name_pref = '{}-{}-{}-{}'.format(round(lon_start, 2), round(lat_start, 2),round(lon_end, 2), round(lat_end, 2)).replace('.', '_')
        
        tileSize = 256
        current_source = self.source_cmbx.currentText()
        tile_url = dict_sources[current_source]['url']

        file_pil = os.path.join(self.work_folder, 'pil_img.jpeg')
        final_file_out = os.path.join(self.work_folder,'{}_{}_pre_clip.tif'.format(current_source, f_name_pref))
        final_file_out_cropped = os.path.join(self.work_folder,'{}_{}.tif'.format(current_source, f_name_pref))
        
        # yandex
        if 'yandex' in current_source.lower():
            lon_start, lat_start = yandex_coor_convert(lon_start, lat_start)
            lon_end, lat_end = yandex_coor_convert(lon_end, lat_end)

        xy_start = getxy(lon_start, lat_start, zoom, tileSize)
        xy_end = getxy(lon_end, lat_end, zoom, tileSize)

        dt = getxy_reverse(xy_start[0], xy_start[1], zoom, 256)
        et = getxy_reverse(xy_end[0]+1, xy_end[1]+1, zoom, 256)

        # yandex
        if 'yandex' in current_source.lower():
            dt = yandex_coor_convert_reverse(xy_start[0], xy_start[1], zoom)
            et = yandex_coor_convert_reverse(xy_end[0]+1, xy_end[1]+1, zoom)
        
        xmin = xy_start[0]
        ymin = xy_start[1]
        xmax = xy_end[0]
        ymax = xy_end[1]
            
        session = requests.Session()
        total_width = (xmax+1 - xmin)*256
        total_height = (ymax+1 - ymin)*256
        new_im = Image.new('RGB', (total_width, total_height))
        
        counts = (ymax - (ymin-1)) * ((xmax+1) - xmin)
        step_pbar = 100/counts
        step_counter = 0
        files_list = [file_pil, final_file_out]
        offset_y = 256
        for y_step in range(ymax, ymin-1, -1):
            offset_x = 0
            for x_step in range(xmin, xmax+1, 1):
                if 'bing' in current_source.lower():
                    quadkey = bing_coor_convert(x_step, y_step, zoom)
                    lnk_download = tile_url.format(quadkey)
                else:
                    lnk_download = tile_url.format(x_step, y_step, zoom)
                
                file_out = os.path.join(self.work_folder, 'out_{}_{}.jpeg'.format(x_step, y_step))
                
                xy_orig = getxy_reverse(x_step, y_step, zoom, tileSize)
                if 'yandex' in current_source.lower():
                    xy_orig = yandex_coor_convert_reverse(x_step, y_step, zoom)
                
                step_counter+=step_pbar
                self.pbar.setValue(step_counter)
                
                request = session.get(lnk_download, headers=headers)
                if request.status_code == 404:
                    img = Image.new("RGB", (256, 256), (255, 255, 255))
                    img.save(file_out, "JPEG")
                else:
                    with open(file_out, 'wb') as out:
                        out.write(request.content)
                time.sleep(.2)
                files_list.append(file_out)
                
                im_saved = Image.open(file_out)
                new_im.paste(im_saved, (offset_x, total_height-offset_y))
                offset_x += 256
                QtCore.QCoreApplication.processEvents()
            offset_y += 256
            offset_x = 0
        new_im.save(file_pil, quality=100)
        gcps = [gdal.GCP(dt[0], dt[1], 0, 0, 0),
            gdal.GCP(et[0], dt[1], 0, total_width, 0),
            gdal.GCP(et[0], et[1], 0, total_width, total_height),
            gdal.GCP(dt[0], et[1], 0, 0, total_height)
        ]
        ds = gdal.Translate(final_file_out, file_pil)
        sr = osr.SpatialReference()
        sr.ImportFromEPSG(4326) 
        ds.SetGCPs(gcps, sr.ExportToWkt())
        ds = None
        window = (pnt_start.asPoint().x(), pnt_end.asPoint().y(), pnt_end.asPoint().x(), pnt_start.asPoint().y())
        cropped_img = gdal.Warp(final_file_out_cropped, final_file_out, outputBounds = window, cropToCutline = True, copyMetadata = True)
        cropped_img = None
        self.pbar.setValue(0)
        if self.add_checkbox.isChecked():
            rlayer = QgsRasterLayer(final_file_out_cropped, '{}_{}'.format(current_source, f_name_pref))
            QgsProject.instance().addMapLayer(rlayer)  
        for file in files_list:
            os.remove(file)
        write_cfg(self.work_folder, zoom, current_source)
        return        
    

    def save_frame(self):
        # save current drawn rectangle to *.xtnt file
        if self.draw_tool:
            self.pnt_geom_start, self.pnt_geom_end = self.rband_coors()
            name = QFileDialog.getSaveFileName(self, 'Save Frame', "", "xtnt (*.xtnt)")
            if name[0]:
                data = [(p.asPoint().x(), p.asPoint().y()) for p in [self.pnt_geom_start, self.pnt_geom_end]]
                data_str = "\n".join(["{},{}".format(p[0], p[1])  for p in data])
                with open(name[0], 'w') as wfile:
                    wfile.write(data_str)
        return

            
    def load_frame(self):
        # load rectangle coordinates from *.xtnt file
        f_open = QFileDialog.getOpenFileName(self, "Select Frame", "", "*.xtnt")
        if f_open[0]:
            lines = []
            with open(f_open[0], 'r') as rfile:
                lines = [line.rstrip() for line in rfile]
            uw_pnt = [float(p) for p in lines[0].split(',')]
            le_pnt = [float(p) for p in lines[1].split(',')]
            
            start_point = QgsPointXY(uw_pnt[0], uw_pnt[1])
            end_point = QgsPointXY(le_pnt[0], le_pnt[1])
            
            start_point_geom = QgsGeometry().fromPointXY(start_point)
            end_point_geom = QgsGeometry().fromPointXY(end_point)
            
            start_point_geom.transform(self.tr_reversed)
            end_point_geom.transform(self.tr_reversed)

            start_point_projected = start_point_geom.asPoint()
            end_point_projected = end_point_geom.asPoint()

            if self.draw_tool:
                self.draw_tool.rubberBand.setToGeometry(geom_polygon)
            else:
                self.draw_tool = rband(iface.mapCanvas(), self)
                iface.mapCanvas().setMapTool(self.draw_tool) 
                self.draw_tool.showRect(start_point_projected, end_point_projected)
                self.draw_tool.startPoint = start_point_projected
                self.draw_tool.endPoint = end_point_projected
            self.activate_dl()

        return
            
        
    def closeEvent(self, event):
        # remove all rubberbands
        if self.draw_tool:
            self.draw_tool.rubberBand.reset()
            self.draw_tool.deactivate()
            iface.mapCanvas().unsetMapTool(self.draw_tool) 