from typing import List
import math

from qgis.core import (
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingParameterDistance,
    QgsProcessingParameterNumber,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterBoolean,
    QgsField,
    QgsProcessingParameterFeatureSink,
    QgsFields,
    QgsFeature,
    QgsLineString,
    QgsPoint,
    QgsWkbTypes,
Qgis,
QgsMessageLog)

from qgis.PyQt.QtCore import QVariant

from los_tools.constants.field_names import FieldNames
from los_tools.tools.util_functions import get_horizon_lines_type

class FindVistaAlgorithm(QgsProcessingAlgorithm):

    INPUT_HORIZON_LINES_LAYER = "InputHorizonLinesLayer"
    MIN_DIST_CHANGE = "MinimalDistanceChange"
    MAX_DIST_CHANGE = "MaximalDistanceChange"
    MIN_WIDTH = "MinimalWidth"
    MAX_WIDTH = "MaximalWidth"
    OUTPUT_LAYER = "OutputLayer"

    def initAlgorithm(self, config=None):

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_HORIZON_LINES_LAYER,
                "Input Horizon Lines layer",
                [QgsProcessing.TypeVectorLine])
        )

        self.addParameter(
            QgsProcessingParameterDistance(
                self.MIN_DIST_CHANGE,
                "Minimal distance change",
                parentParameterName=self.INPUT_HORIZON_LINES_LAYER,
                defaultValue=100.0,
                minValue=1,
                optional=False)
        )

        self.addParameter(
            QgsProcessingParameterDistance(
                self.MAX_DIST_CHANGE,
                "Maximal distance change",
                parentParameterName=self.INPUT_HORIZON_LINES_LAYER,
                defaultValue=50.0,
                minValue=1,
                optional=False)
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MIN_WIDTH,
                "Minimal angle width",
                defaultValue=5.0,
                minValue=1,
                maxValue=180,
                optional=False)
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.MAX_WIDTH,
                "Maximal angle width",
                defaultValue=10.0,
                minValue=1,
                maxValue=180,
                optional=False)
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT_LAYER,
                "Output layer")
        )

    def checkParameterValues(self, parameters, context):

        input_horizon_layer = self.parameterAsSource(parameters, self.INPUT_HORIZON_LINES_LAYER, context)

        min_angle_width = self.parameterAsDouble(parameters, self.MIN_WIDTH, context)
        max_angle_width = self.parameterAsDouble(parameters, self.MAX_WIDTH, context)

        field_names = input_horizon_layer.fields().names()

        if FieldNames.HORIZON_TYPE not in field_names:
            msg = "Fields specific for horizon lines not found in current layer ({0}). " \
                  "Cannot export the layer as horizon lines.".format(FieldNames.HORIZON_TYPE)

            QgsMessageLog.logMessage(msg,
                                     "los_tools",
                                     Qgis.MessageLevel.Critical)
            return False, msg

        if max_angle_width <= min_angle_width:

            msg = "Maximal width of vista has to be bigger then minimal width."

            QgsMessageLog.logMessage(msg,
                                     "los_tools",
                                     Qgis.MessageLevel.Critical)
            return False, msg

        return super().checkParameterValues(parameters, context)

    def processAlgorithm(self, parameters, context, feedback):

        input_horizon_layer = self.parameterAsSource(parameters, self.INPUT_HORIZON_LINES_LAYER, context)
        min_dist_change = self.parameterAsDouble(parameters, self.MIN_DIST_CHANGE, context)
        max_dist_change = self.parameterAsDouble(parameters, self.MAX_DIST_CHANGE, context)
        min_angle_width = self.parameterAsDouble(parameters, self.MIN_WIDTH, context)
        max_angle_width = self.parameterAsDouble(parameters, self.MAX_WIDTH, context)

        fields = QgsFields()
        fields.append(QgsField(FieldNames.ID_OBSERVER, QVariant.Int))
        fields.append(QgsField(FieldNames.HORIZON_TYPE, QVariant.String))

        sink, dest_id = self.parameterAsSink(parameters,
                                             self.OUTPUT_LAYER,
                                             context,
                                             fields,
                                             QgsWkbTypes.MultiLineString,
                                             input_horizon_layer.sourceCrs())

        feature_count = input_horizon_layer.featureCount()
        total = 100.0 / feature_count if feature_count else 0

        horizon_lines_iterator = input_horizon_layer.getFeatures()

        horizon_type = get_horizon_lines_type(input_horizon_layer)

        for feature_number, horizon_line_feature in enumerate(horizon_lines_iterator):

            if feedback.isCanceled():
                break

            observer_x = horizon_line_feature.attribute(FieldNames.OBSERVER_X)
            observer_y = horizon_line_feature.attribute(FieldNames.OBSERVER_Y)
            point_observer = QgsPoint(observer_x, observer_y)

            horizon_line_geom = QgsLineString()
            horizon_line_geom.fromWkt(horizon_line_feature.geometry().asWkt())

            points: List[QgsPoint] = horizon_line_geom.points()

            previous_point = points[0]

            angle_points = abs(point_observer.azimuth(points[0]) - point_observer.azimuth(points[1]))

            i = 1

            while i < len(points):

                start_point: QgsPoint = points[i]

                found_vista = False

                QgsMessageLog.logMessage("{} - {} --- {}".format(i,
                                                                point_observer.distance(points[i-1]),
                                                                point_observer.distance(start_point)),
                                         "los_tools1",
                                         Qgis.Warning)

                if point_observer.distance(points[i-1]) < point_observer.distance(start_point):

                    start_angle = point_observer.azimuth(start_point)

                    max_mid_distance = -float('Inf')

                    for j in range(i+1, len(points)-1):

                        mid_point: QgsPoint = points[j]

                        diff_distance_positive = point_observer.distance(mid_point) - point_observer.distance(start_point)

                        if min_dist_change < diff_distance_positive:
                            break

                    for k in range(j+1, len(points)-1):

                        end_point: QgsPoint = points[k]

                        diff_distance_negative = point_observer.distance(mid_point) - point_observer.distance(end_point)

                        if min_dist_change < diff_distance_negative and \
                            min_angle_width <= abs(start_angle - point_observer.azimuth(end_point)) <= max_angle_width:

                            found_vista = True

                        if found_vista and point_observer.distance(end_point) < point_observer.distance(points[j+1]):
                            break

                if found_vista:
                    f = QgsFeature(fields)
                    f.setGeometry(QgsLineString([start_point, end_point]))
                    f.setAttribute(f.fieldNameIndex(FieldNames.ID_OBSERVER),
                                   horizon_line_feature.attribute(FieldNames.ID_OBSERVER))
                    f.setAttribute(f.fieldNameIndex(FieldNames.HORIZON_TYPE),
                                   horizon_line_feature.attribute(FieldNames.HORIZON_TYPE))
                    sink.addFeature(f)

                    i = k
                else:
                    i += 1

            feedback.setProgress(int(feature_number * total))

        return {self.OUTPUT_LAYER: dest_id}

    def name(self):
        return "findvista"

    def displayName(self):
        return "Find vista on horizon lines"

    def group(self):
        return "Horizons"

    def groupId(self):
        return "horizons"

    def createInstance(self):
        return FindVistaAlgorithm()

    def helpUrl(self):
        pass
        #return "https://jancaha.github.io/qgis_los_tools/tools/LoS%20Points/tool_extract_points_los/"