#! python3  # noqa: E265

"""
    Widget to locate any feature from any layer
"""

# standard
import os

# PyQGIS
from qgis.core import QgsExpression, QgsMapLayerProxyModel, QgsProject
from qgis.gui import QgisInterface
from qgis.PyQt import uic
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QWidget

from locator_grand_lyon.toolbelt.log_handler import PlgLogger

# project

# ############################################################################
# ########## Globals ###############
# ##################################


# ############################################################################
# ########## Classes ###############
# ##################################
class GenericLocatorWidget(QWidget):
    """Widget to locate a feature from any layer"""

    def __init__(self, parent: QWidget = None, iface: QgisInterface = None) -> None:
        super().__init__(parent)
        uic.loadUi(
            os.path.dirname(os.path.realpath(__file__)) + "/wdg_generic_locator.ui",
            self,
        )
        self.setWindowIcon(QIcon(":images/themes/default/mActionOpenTableVisible.svg"))
        self.log = PlgLogger().log

        self.iface = iface
        self._init_field_cbx()

        self.cbx_layer.setFilters(QgsMapLayerProxyModel.VectorLayer)

        self.cbx_layer.layerChanged.connect(self._init_field_cbx)
        self.cbx_field.fieldChanged.connect(self._init_wdg_picker)
        self.btn_unique_locate.clicked.connect(self.locate_unique_element)

    def _init_field_cbx(self) -> None:
        """Initialization of the field combobox"""
        layer = self.cbx_layer.currentLayer()
        enable = layer is not None
        self.cbx_field.setEnabled(enable)
        self.cbx_unique_attribute.setEnabled(enable)

        if enable:
            self.cbx_field.setToolTip("")
            self.cbx_unique_attribute.setToolTip("")
        else:
            self.cbx_field.setToolTip(self.tr("Aucune couche sélectionnée"))
            self.cbx_unique_attribute.setToolTip("Aucune couche sélectionnée")

        self.cbx_field.setLayer(layer)
        self.cbx_unique_attribute.setSourceLayer(layer)

        self._init_wdg_picker()

    def _init_wdg_picker(self) -> None:
        """Initialization of feature picker widget"""
        field = self.cbx_field.currentField()
        enable = field != ""
        self.btn_unique_locate.setEnabled(enable)

        if enable:
            self.cbx_unique_attribute.setToolTip("")
            self.btn_unique_locate.setToolTip("")
        else:
            self.cbx_unique_attribute.setToolTip(self.tr("Aucun champ sélectionné"))
            self.btn_unique_locate.setToolTip("Aucun champ sélectionné")

        if field != "":
            self.cbx_unique_attribute.setDisplayField(field)
            self.cbx_unique_attribute.setFilterExpression("")

    def locate_unique_element(self):
        """Zoom to unique attribute from layer field"""
        layer = self.cbx_layer.currentLayer()
        if layer is not None:
            layer.removeSelection()
            if not self.cbx_unique_attribute.isClearValueSelected():
                expression = QgsExpression.createFieldEqualityExpression(
                    self.cbx_field.currentField(),
                    self.cbx_unique_attribute.currentData(),
                )
                layer.selectByExpression(expression)
                self.iface.mapCanvas().zoomToSelected(layer)
            else:
                # Zoom sur l'étendue de la couche
                canvas = self.iface.mapCanvas()
                layer_extent = layer.extent()
                if layer_extent.isEmpty():
                    layer.updateExtents()
                    layer_extent = layer.extent()
                # transform extent
                layer_extent = canvas.mapSettings().layerExtentToOutputExtent(
                    layer, layer_extent
                )
                canvas.setExtent(layer_extent)
                canvas.refresh()
