# standard
import time

# PyQGIS
from qgis.core import QgsApplication, QgsVectorLayer
from qgis.gui import QgsFilterLineEdit
from qgis.PyQt.QtCore import QRegExp, Qt
from qgis.PyQt.QtGui import QRegExpValidator
from qgis.PyQt.QtWidgets import QApplication, QComboBox, QCompleter, QWidget

# project
from locator_grand_lyon.toolbelt.log_handler import PlgLogger


class UniqueLayerAttributesComboBox(QComboBox):
    """Widget to select unique layer field"""

    def __init__(self, parent: QWidget) -> None:
        super().__init__(parent)

        self.log = PlgLogger().log

        # Add completer for search
        completer = QCompleter(self.model())
        completer.setWidget(self)
        self.setEditable(True)
        self.setCompleter(completer)

        # Custom line edit for search
        self.filter_lineedit = QgsFilterLineEdit(None, self.tr("Aucune sélection"))
        self.filter_lineedit.setSelectOnFocus(True)
        self.filter_lineedit.setShowClearButton(True)

        self.setLineEdit(self.filter_lineedit)

        # No layer and field selected by default
        self.layer = None
        self.filter_layer = None
        self.field = None

    def isClearValueSelected(self) -> bool:
        """Check if clear value is selected

        :return: True if clear value selected, False otherwise
        :rtype: bool
        """
        return self.filter_lineedit.isNull()

    def setSourceLayer(self, layer: QgsVectorLayer) -> None:
        """Define source layer for feature search

        :param layer: source layer
        :type layer: QgsVectorLayer
        """
        if self.layer is not None:
            self.layer.destroyed.disconnect(self.clearSourceLayer)

        if layer is not None:
            self.layer = layer
            self.filter_layer = layer.clone()
            # Connect to layer destruction for clear
            self.layer.destroyed.connect(self.clearSourceLayer)
        else:
            self.clearSourceLayer()

    def clearSourceLayer(self) -> None:
        """Clear combobox for no source layer selection"""
        self.clear()
        if self.filter_layer is not None:
            del self.filter_layer
            self.filter_layer = None
        self.field = None
        self.layer = None

    def setDisplayField(self, field: str) -> None:
        """Define display unique field

        :param field: field name
        :type field: str
        """
        self.field = field

    def setFilterExpression(self, expression: str) -> None:
        """Define filter expression for unique field search

        :param expression: expression for search
        :type expression: str
        """
        if self.filter_layer is not None and self.field is not None:
            QApplication.setOverrideCursor(Qt.WaitCursor)
            time_start = time.perf_counter()
            # Get uniques value
            self.filter_layer.setSubsetString(expression)
            fields = self.filter_layer.fields()
            field_index = fields.indexFromName(self.field)
            qfield = fields.at(field_index)
            unique_value = list(self.filter_layer.uniqueValues(field_index))

            time_end = time.perf_counter()
            time_duration = time_end - time_start
            self.log(
                message=f"Filter unique layer field with expression {expression} in {time_duration:.3f} seconds",
                log_level=4,
            )

            # Sort for display
            unique_value.sort()
            unique_value_str = [qfield.displayString(val) for val in unique_value]

            QApplication.restoreOverrideCursor()

            # Add validator to use only available values
            regex = QRegExp("^(" + "|".join(unique_value_str) + ")$")
            validator = QRegExpValidator(regex, self)
            self.setValidator(validator)

            # Define combobox values
            time_start = time.perf_counter()
            self.clear()
            for i, value in enumerate(unique_value):
                self.addItem(unique_value_str[i], value)
            time_end = time.perf_counter()
            time_duration = time_end - time_start
            self.log(
                message=f"Unique layer field combobox populated in {time_duration:.3f} seconds for {len(unique_value)} values",
                log_level=4,
            )
