#! python3  # noqa: E265

"""
    Widget to locate cadastre feature
"""

# standard
import os

# PyQGIS
from qgis.core import QgsProject
from qgis.gui import QgisInterface, QgsFeatureListComboBox
from qgis.PyQt import uic
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QWidget

# project
from locator_grand_lyon.toolbelt.log_handler import PlgLogger
from locator_grand_lyon.toolbelt.preferences import PlgOptionsManager

# ############################################################################
# ########## Globals ###############
# ##################################


# ############################################################################
# ########## Classes ###############
# ##################################


class CadastreLocatorWidget(QWidget):
    """Widget to locate cadastre feature"""

    def __init__(self, parent: QWidget = None, iface: QgisInterface = None) -> None:
        super().__init__(parent)
        uic.loadUi(
            os.path.dirname(os.path.realpath(__file__)) + "/wdg_cadastre_locator.ui",
            self,
        )
        self.setWindowIcon(QIcon(":images/themes/default/mIconQgsProjectFile.svg"))
        self.log = PlgLogger().log

        self.iface = iface
        self._init_all_cbx_feature_list()

        self.cbx_feat_commune.currentFeatureChanged.connect(self.commune_changed)
        self.btn_parcelle.clicked.connect(self.locate_parcelle)
        self.btn_parcelle_historique.clicked.connect(self.locate_parcelle_historique)
        self.cbx_unique_section.currentTextChanged.connect(self.section_changed)

        QgsProject.instance().layersAdded.connect(self._project_layer_updated)
        QgsProject.instance().layersRemoved.connect(self._project_layer_updated)

    def _init_all_cbx_feature_list(self) -> None:
        """Init all QgsFeatureListComboBox with expected layer"""
        plg_settings = PlgOptionsManager.get_plg_settings()
        self._init_cbx_feature_list(
            self.cbx_feat_commune,
            layer_name=plg_settings.commune_layer,
            identifier_field=plg_settings.commune_id,
            display_expression=plg_settings.commune_display,
        )
        self._init_cbx_feature_list(
            self.cbx_feat_parcelle,
            layer_name=plg_settings.parcelle_layer,
            identifier_field=plg_settings.parcelle_id,
            display_expression=plg_settings.parcelle_display,
        )
        self._init_cbx_feature_list(
            self.cbx_feat_parcelle_historique,
            layer_name=plg_settings.parcelle_hist_layer,
            identifier_field=plg_settings.parcelle_hist_id,
            display_expression=plg_settings.parcelle_hist_display,
        )

        layer = QgsProject.instance().mapLayersByName(plg_settings.parcelle_layer)
        if len(layer):
            self.cbx_unique_section.setEnabled(True)
            self.cbx_unique_section.setDisplayField(plg_settings.parcelle_section)
            self.cbx_unique_section.setSourceLayer(layer[0])
            self.cbx_unique_section.setToolTip("")
        else:
            self.cbx_unique_section.setToolTip(
                self.tr("Couche '{}' non disponible").format(
                    plg_settings.parcelle_layer
                )
            )
            self.cbx_unique_section.setSourceLayer(None)
            self.cbx_unique_section.clearEditText()
            self.cbx_unique_section.setEnabled(False)

        layer = QgsProject.instance().mapLayersByName(plg_settings.parcelle_layer)
        self.btn_parcelle.setEnabled(len(layer) != 0)

        layer = QgsProject.instance().mapLayersByName(plg_settings.parcelle_hist_layer)
        self.btn_parcelle_historique.setEnabled(len(layer) != 0)

    def plugin_setting_updated(self):
        """Update widget for new plugin settings"""
        self._init_all_cbx_feature_list()

    def _init_cbx_feature_list(
        self,
        cbx_feat_list: QgsFeatureListComboBox,
        layer_name: str,
        identifier_field: str,
        display_expression: str,
    ) -> None:
        """Initialization of a QgsFeatureListComboBox.
        If the source layer is not available, component is disabled

        :param cbx_feat_list: component to initialize
        :type cbx_feat_list: QgsFeatureListComboBox
        :param layer_name: source layer name
        :type layer_name: str
        :param identifier_field: identifier field
        :type identifier_field: str
        :param display_expression: display expression
        :type display_expression: str
        """
        layer = QgsProject.instance().mapLayersByName(layer_name)
        if len(layer):
            cbx_feat_list.setSourceLayer(layer[0])
            cbx_feat_list.setIdentifierFields([identifier_field])
            cbx_feat_list.setDisplayExpression(display_expression)
            cbx_feat_list.setToolTip("")
            cbx_feat_list.setEnabled(True)
        else:
            cbx_feat_list.setToolTip(
                self.tr("Couche '{}' non disponible").format(layer_name)
            )
            cbx_feat_list.clearEditText()
            cbx_feat_list.setEnabled(False)

    def commune_changed(self) -> None:
        """Update section field filter for a specific commune"""
        plg_settings = PlgOptionsManager.get_plg_settings()
        id_values = self.cbx_feat_commune.identifierValues()
        if len(id_values):
            self.cbx_unique_section.currentTextChanged.disconnect(self.section_changed)
            self.cbx_unique_section.setFilterExpression(
                f"{plg_settings.parcelle_commune}='{id_values[0]}'"
            )
            self.cbx_unique_section.currentTextChanged.connect(self.section_changed)
            self.section_changed()

    def section_changed(self) -> None:
        """Update parcelle for a specific section and commune"""
        section = self.cbx_unique_section.currentText()
        plg_settings = PlgOptionsManager.get_plg_settings()
        id_values = self.cbx_feat_commune.identifierValues()
        if len(id_values) and section:
            self.cbx_feat_parcelle.setFilterExpression(
                f"{plg_settings.parcelle_commune}='{id_values[0]}' and {plg_settings.parcelle_section}='{section}'"
            )
            self.cbx_feat_parcelle_historique.setFilterExpression(
                f"{plg_settings.parcelle_hist_id} ILIKE '{id_values[0]}{section}%'"
            )

    def locate_parcelle(self) -> None:
        """Zoom to selected parcelle"""
        self._zoom_to_selected_feat(self.cbx_feat_parcelle)

    def locate_parcelle_historique(self) -> None:
        """Zoom to selected parcelle historique"""
        self._zoom_to_selected_feat(self.cbx_feat_parcelle_historique)

    def _zoom_to_selected_feat(self, cbx_feat_list: QgsFeatureListComboBox) -> None:
        """Zoom to selected feature of QgsFeatureListComboBox"""
        layer = cbx_feat_list.sourceLayer()
        if layer is not None:
            feat_request = cbx_feat_list.currentFeatureRequest()
            layer.selectByIds([i.id() for i in layer.getFeatures(feat_request)])
            canvas = self.iface.mapCanvas()
            canvas.zoomToSelected(layer)

    def _project_layer_updated(self) -> None:
        """Init QgsFeatureListComboBox for layer update"""
        self._init_all_cbx_feature_list()
