#! python3  # noqa: E265

"""
    Widget to locate road feature
"""

# standard
import os
from typing import List

# PyQGIS
from qgis.core import QgsGeometry, QgsProject
from qgis.gui import QgisInterface, QgsFeatureListComboBox
from qgis.PyQt import uic
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QWidget

# project
from locator_grand_lyon.toolbelt.log_handler import PlgLogger
from locator_grand_lyon.toolbelt.preferences import PlgOptionsManager

# ############################################################################
# ########## Globals ###############
# ##################################


# ############################################################################
# ########## Classes ###############
# ##################################


class RoadLocatorWidget(QWidget):
    """Widget to locate road feature"""

    def __init__(self, parent: QWidget = None, iface: QgisInterface = None) -> None:
        super().__init__(parent)
        uic.loadUi(
            os.path.dirname(os.path.realpath(__file__)) + "/wdg_road_locator.ui",
            self,
        )
        self.setWindowIcon(
            QIcon(":images/themes/default/mIconSnappingIntersection.svg")
        )
        self.log = PlgLogger().log

        self.iface = iface
        self._init_all_cbx_feature_list()

        self.cbx_feat_route.currentFeatureChanged.connect(self.road_changed)
        self.cbx_feat_intersection.currentFeatureChanged.connect(
            self.intersection_changed
        )
        self.btn_route.clicked.connect(self.locate_road)
        self.btn_intersection.clicked.connect(self.locate_intersection)

        QgsProject.instance().layersAdded.connect(self._project_layer_updated)
        QgsProject.instance().layersRemoved.connect(self._project_layer_updated)

        self.btn_route.setEnabled(False)
        self.btn_intersection.setEnabled(False)

    def _init_all_cbx_feature_list(self) -> None:
        """Init all QgsFeatureListComboBox with expected layer"""
        plg_settings = PlgOptionsManager.get_plg_settings()
        self._init_cbx_feature_list(
            self.cbx_feat_route,
            layer_name=plg_settings.road_layer_name,
            display_expression=plg_settings.road_search_display,
        )
        self._init_cbx_feature_list(
            self.cbx_feat_intersection,
            layer_name=plg_settings.road_layer_name,
            display_expression=plg_settings.road_search_display,
        )

        layer = QgsProject.instance().mapLayersByName(plg_settings.road_layer_name)
        self.btn_route.setEnabled(len(layer) != 0)
        self.btn_intersection.setEnabled(len(layer) != 0)

    def plugin_setting_updated(self):
        """Update widget for new plugin settings"""
        self._init_all_cbx_feature_list()

    def _init_cbx_feature_list(
        self,
        cbx_feat_list: QgsFeatureListComboBox,
        layer_name: str,
        display_expression: str,
    ) -> None:
        """Initialization of a QgsFeatureListComboBox.
        If the source layer is not available, component is disabled

        :param cbx_feat_list: component to initialize
        :type cbx_feat_list: QgsFeatureListComboBox
        :param layer_name: source layer name
        :type layer_name: str
        :param display_expression: display expression
        :type display_expression: str
        """
        layers = QgsProject.instance().mapLayersByName(layer_name)
        if len(layers):
            layer = layers[0]
            primary_key_attributes = layer.primaryKeyAttributes()
            field_names = [
                layer.fields().at(idx).name() for idx in primary_key_attributes
            ]
            cbx_feat_list.setSourceLayer(layer)
            cbx_feat_list.setIdentifierFields(field_names)
            cbx_feat_list.setDisplayExpression(display_expression)
            cbx_feat_list.setToolTip("")
            cbx_feat_list.setEnabled(True)
        else:
            cbx_feat_list.setToolTip(
                self.tr("Couche '{}' non disponible").format(layer_name)
            )
            cbx_feat_list.clearEditText()
            cbx_feat_list.setEnabled(False)

    def road_changed(self) -> None:
        """Update intersection filter for a specific road"""
        layer = self.cbx_feat_route.sourceLayer()
        if layer is None:
            self.btn_route.setEnabled(False)
            return

        feature_request = self.cbx_feat_route.currentFeatureRequest()
        feature_iterator = layer.getFeatures(feature_request)
        features = [feat for feat in feature_iterator]
        if len(features):
            full_geom = QgsGeometry.collectGeometry(
                [feat.geometry() for feat in features]
            )
            ids_str = [
                str(id) for id in self._get_selected_feature_id(self.cbx_feat_route)
            ]
            expression_str = f"intersects($geometry, geom_from_wkt('{full_geom.asWkt()}')) AND $id NOT IN({','.join(ids_str)})"

            self.cbx_feat_intersection.setFilterExpression(expression_str)
            self.btn_intersection.setEnabled(True)
            self.btn_route.setEnabled(True)
        else:
            self.cbx_feat_intersection.setCurrentIndex(
                self.cbx_feat_intersection.nullIndex()
            )
            self.btn_route.setEnabled(False)
            self.btn_intersection.setEnabled(False)

    def intersection_changed(self) -> None:
        """Enable or disable intersection location button"""
        layer = self.cbx_feat_intersection.sourceLayer()
        if layer is None:
            self.btn_intersection.setEnabled(False)
            return
        feature_request = self.cbx_feat_intersection.currentFeatureRequest()
        feature_iterator = layer.getFeatures(feature_request)
        features = [feat for feat in feature_iterator]
        if len(features):
            self.btn_intersection.setEnabled(True)
        else:
            self.btn_intersection.setEnabled(False)

    def locate_road(self) -> None:
        """Zoom to selected road"""
        self._zoom_to_selected_feat(self.cbx_feat_route)

    def locate_intersection(self) -> None:
        """Zoom to selected intersection"""
        layer = self.cbx_feat_intersection.sourceLayer()
        if layer is not None:
            ids = self._get_selected_feature_id(self.cbx_feat_route)
            ids += self._get_selected_feature_id(self.cbx_feat_intersection)
            layer.selectByIds(ids)
            canvas = self.iface.mapCanvas()
            canvas.zoomToSelected(layer)

    def _zoom_to_selected_feat(self, cbx_feat_list: QgsFeatureListComboBox) -> None:
        """Zoom to selected feature of QgsFeatureListComboBox"""
        layer = cbx_feat_list.sourceLayer()
        if layer is not None:
            layer.selectByIds(self._get_selected_feature_id(self.cbx_feat_route))
            canvas = self.iface.mapCanvas()
            canvas.zoomToSelected(layer)

    def _get_selected_feature_id(
        self, cbx_feat_list: QgsFeatureListComboBox
    ) -> List[int]:
        """Get selected feature id  of QgsFeatureListComboBox

        :param cbx_feat_list: feature combobox
        :type cbx_feat_list: QgsFeatureListComboBox
        :return: list of feature ids
        :rtype: List[int]
        """
        layer = cbx_feat_list.sourceLayer()
        if layer is not None:
            feat_request = cbx_feat_list.currentFeatureRequest()
            return [i.id() for i in layer.getFeatures(feat_request)]
        else:
            return []

    def _project_layer_updated(self) -> None:
        """Init QgsFeatureListComboBox for layer update"""
        self._init_all_cbx_feature_list()
