#! python3  # noqa: E265

"""
    Main plugin module.
"""

# standard
from functools import partial
from pathlib import Path

# PyQGIS
from qgis.core import QgsApplication, QgsSettings
from qgis.gui import QgisInterface
from qgis.PyQt.QtCore import QCoreApplication, QLocale, Qt, QTranslator, QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QAction, QDockWidget, QWidget

# project
from locator_grand_lyon.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
    __title__,
    __uri_homepage__,
)
from locator_grand_lyon.core.adress_locator_filter import AdressLocatorFilter
from locator_grand_lyon.core.road_locator_filter import RoadLocatorFilter
from locator_grand_lyon.gui.dlg_settings import PlgOptionsFactory
from locator_grand_lyon.gui.wdg_cadastre_locator import CadastreLocatorWidget
from locator_grand_lyon.gui.wdg_generic_locator import GenericLocatorWidget
from locator_grand_lyon.gui.wdg_road_locator import RoadLocatorWidget
from locator_grand_lyon.toolbelt import PlgLogger

# ############################################################################
# ########## Classes ###############
# ##################################


class LocatorGrandLyonPlugin:
    def __init__(self, iface: QgisInterface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class which \
        provides the hook by which you can manipulate the QGIS application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.log = PlgLogger().log
        self.docks = []
        self.widgets = []
        self.toolbar = self.iface.addToolBar(self.tr("Localisation Grand Lyon"))
        self.adresse_locator_filter = None
        self.road_locator_filter = None

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[
            0:2
        ]
        locale_path: Path = (
            DIR_PLUGIN_ROOT
            / "resources"
            / "i18n"
            / f"locator_grand_lyon_{self.locale}.qm"
        )
        self.log(message=f"Translation: {self.locale}, {locale_path}", log_level=4)
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

    def initGui(self):
        """Set up plugin UI elements."""

        # settings page within the QGIS preferences menu
        self.options_factory = PlgOptionsFactory()
        self.options_factory.settingsUpdated.connect(self._setting_updated)
        self.iface.registerOptionsWidgetFactory(self.options_factory)

        # -- Actions
        self.action_help = QAction(
            QgsApplication.getThemeIcon("mActionHelpContents.svg"),
            self.tr("Help"),
            self.iface.mainWindow(),
        )
        self.action_help.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.action_settings = QAction(
            QgsApplication.getThemeIcon("console/iconSettingsConsole.svg"),
            self.tr("Settings"),
            self.iface.mainWindow(),
        )
        self.action_settings.triggered.connect(
            lambda: self.iface.showOptionsDialog(
                currentPage="mOptionsPage{}".format(__title__)
            )
        )

        # -- Menu
        self.iface.addPluginToMenu(__title__, self.action_settings)
        self.iface.addPluginToMenu(__title__, self.action_help)

        # -- Help menu

        # documentation
        self.iface.pluginHelpMenu().addSeparator()
        self.action_help_plugin_menu_documentation = QAction(
            QIcon(str(__icon_path__)),
            f"{__title__} - Documentation",
            self.iface.mainWindow(),
        )
        self.action_help_plugin_menu_documentation.triggered.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.iface.pluginHelpMenu().addAction(
            self.action_help_plugin_menu_documentation
        )

        actions = [
            {
                "name": self.tr("Cadastre"),
                "widget": CadastreLocatorWidget(self.iface.mainWindow(), self.iface),
            },
            {
                "name": self.tr("Generic Locator"),
                "widget": GenericLocatorWidget(self.iface.mainWindow(), self.iface),
            },
            {
                "name": self.tr("Road Locator"),
                "widget": RoadLocatorWidget(self.iface.mainWindow(), self.iface),
            },
        ]

        for action in actions:
            self.add_action(action["name"], action["widget"])

        # locator
        if not self.adresse_locator_filter:
            self.adresse_locator_filter = AdressLocatorFilter(self.iface)
            self.iface.registerLocatorFilter(self.adresse_locator_filter)

        if not self.road_locator_filter:
            self.road_locator_filter = RoadLocatorFilter(self.iface)
            self.iface.registerLocatorFilter(self.road_locator_filter)

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        # -- Clean up menu
        self.iface.removePluginMenu(__title__, self.action_help)
        self.iface.removePluginMenu(__title__, self.action_settings)

        # -- Clean up preferences panel in QGIS settings
        self.iface.unregisterOptionsWidgetFactory(self.options_factory)

        # remove from QGIS help/extensions menu
        if self.action_help_plugin_menu_documentation:
            self.iface.pluginHelpMenu().removeAction(
                self.action_help_plugin_menu_documentation
            )

        # remove actions
        del self.action_settings
        del self.action_help

        # remove docks
        for _dock in self.docks:
            self.iface.removeDockWidget(_dock)
            _dock.deleteLater()
        self.docks.clear()
        self.widgets.clear()

        self.toolbar.deleteLater()

        # remove filter from locator
        if self.adresse_locator_filter:
            self.iface.deregisterLocatorFilter(self.adresse_locator_filter)
        if self.road_locator_filter:
            self.iface.deregisterLocatorFilter(self.road_locator_filter)

    def add_action(self, name: str, widget: QWidget):
        """Add action to plugin menu to display a widget

        :param name: widget name
        :type name: str
        :param widget: widget to display in QDockWidget
        :type widget: QWidget
        """

        self.widgets.append(widget)
        dock = QDockWidget(name, self.iface.mainWindow())
        dock.setWindowIcon(widget.windowIcon())
        dock.setWidget(widget)
        self.iface.addDockWidget(Qt.RightDockWidgetArea, dock)
        dock.close()
        self.docks.append(dock)
        dock.toggleViewAction().setIcon(widget.windowIcon())
        self.toolbar.addAction(dock.toggleViewAction())

    def _setting_updated(self):
        """Slot called when plugin settings are update"""
        func_name = "plugin_setting_updated"
        for widget in self.widgets:
            if hasattr(widget, func_name) and callable(getattr(widget, func_name)):
                widget.plugin_setting_updated()

    def run(self):
        """Main process.

        :raises Exception: if there is no item in the feed
        """
        try:
            self.log(
                message=self.tr("Everything ran OK."),
                log_level=3,
                push=False,
            )
        except Exception as err:
            self.log(
                message=self.tr("Houston, we've got a problem: {}".format(err)),
                log_level=2,
                push=True,
            )
