#! python3  # noqa: E265

"""
    Plugin settings form integrated into QGIS 'Options' menu.
"""

import os
import platform

# standard
from dataclasses import asdict
from functools import partial
from pathlib import Path
from urllib.parse import quote

# PyQGIS
from qgis.core import Qgis, QgsApplication
from qgis.gui import QgsFilterLineEdit, QgsOptionsPageWidget, QgsOptionsWidgetFactory
from qgis.PyQt import QtCore, uic
from qgis.PyQt.Qt import QUrl
from qgis.PyQt.QtGui import QDesktopServices, QIcon
from qgis.PyQt.QtWidgets import QWidget

# project
from locator_grand_lyon.__about__ import (
    __icon_path__,
    __title__,
    __uri_homepage__,
    __uri_tracker__,
    __version__,
)
from locator_grand_lyon.toolbelt import PlgLogger, PlgOptionsManager
from locator_grand_lyon.toolbelt.preferences import PlgSettingsStructure

# ############################################################################
# ########## Globals ###############
# ##################################

FORM_CLASS, _ = uic.loadUiType(
    Path(__file__).parent / "{}.ui".format(Path(__file__).stem)
)


# ############################################################################
# ########## Classes ###############
# ##################################


class ConfigOptionsPage(FORM_CLASS, QgsOptionsPageWidget):
    """Settings form embedded into QGIS 'options' menu."""

    settingsUpdated = QtCore.pyqtSignal()

    def __init__(self, parent):
        super().__init__(parent)
        self.log = PlgLogger().log
        self.plg_settings = PlgOptionsManager()

        # load UI and set objectName
        self.setupUi(self)
        self.setObjectName("mOptionsPage{}".format(__title__))

        report_context_message = quote(
            "> Reported from plugin settings\n\n"
            f"- operating system: {platform.system()} "
            f"{platform.release()}_{platform.version()}\n"
            f"- QGIS: {Qgis.QGIS_VERSION}"
            f"- plugin version: {__version__}\n"
        )

        # header
        self.lbl_title.setText(f"{__title__} - Version {__version__}")

        # customization
        self.btn_help.setIcon(QIcon(QgsApplication.iconPath("mActionHelpContents.svg")))
        self.btn_help.pressed.connect(
            partial(QDesktopServices.openUrl, QUrl(__uri_homepage__))
        )

        self.btn_report.setIcon(
            QIcon(QgsApplication.iconPath("console/iconSyntaxErrorConsole.svg"))
        )

        self.btn_report.pressed.connect(
            partial(
                QDesktopServices.openUrl,
                QUrl(
                    f"{__uri_tracker__}new?issuable_template=bug_report&"
                    "issue[title]=[BUG]&"
                    f"issue[description]={report_context_message}"
                ),
            )
        )

        self.btn_reset.setIcon(QIcon(QgsApplication.iconPath("mActionUndo.svg")))
        self.btn_reset.pressed.connect(self.reset_settings)

        # load previously saved settings
        self.load_settings()

    @property
    def attribute_widget_dict(self) -> dict[str, QgsFilterLineEdit]:
        """Get attribute widget dict.
        This is a dict of widget used for settings attribute definition

        :return: _description_
        :rtype: dict[str, QgsFilterLineEdit]
        """
        return {
            "adress_layer_name": self.edit_adress_layer,
            "adress_search_column_name": self.edit_adress_search_column,
            "road_layer_name": self.edit_road_layer,
            "road_search_column_name": self.edit_road_search_column,
            "commune_layer": self.edit_commune_layer,
            "commune_id": self.edit_commune_id,
            "commune_display": self.edit_commune_display,
            "parcelle_layer": self.edit_parcelle_layer,
            "parcelle_id": self.edit_parcelle_id,
            "parcelle_display": self.edit_parcelle_display,
            "parcelle_hist_layer": self.edit_parcelle_hist_layer,
            "parcelle_hist_id": self.edit_parcelle_hist_id,
            "parcelle_hist_display": self.edit_parcelle_hist_display,
            "parcelle_section": self.edit_parcelle_section,
            "parcelle_commune": self.edit_parcelle_commune,
            "road_search_display": self.edit_road_search_display,
        }

    def apply(self):
        """Called to permanently apply the settings shown in the options page (e.g. \
        save them to QgsSettings objects). This is usually called when the options \
        dialog is accepted."""
        settings = self.plg_settings.get_plg_settings()

        # misc
        settings.debug_mode = self.opt_debug.isChecked()
        settings.version = __version__

        # Define attribute
        for attribute, widget in self.attribute_widget_dict.items():
            setattr(settings, attribute, widget.value())

        # dump new settings into QgsSettings
        self.plg_settings.save_from_object(settings)

        if __debug__:
            self.log(
                message="DEBUG - Settings successfully saved.",
                log_level=4,
            )
        self.settingsUpdated.emit()

    def set_attribute_with_env_variable(
        self,
        attribute: str,
        settings: PlgSettingsStructure,
        default_settings: PlgSettingsStructure,
        widget: QgsFilterLineEdit,
    ) -> None:
        """Set attribut in widget with check if environnement variable is used

        :param attribute: settings attribute name
        :type attribute: str
        :param settings: settings containing attribute
        :type settings: PlgSettingsStructure
        :param default_settings: default settings containing attribute
        :type settings: PlgSettingsStructure
        :param widget: widget to store attribute
        :type widget: QgsFilterLineEdit
        """
        value = asdict(settings)[attribute]
        default_value = asdict(default_settings)[attribute]
        widget.setValue(value)
        widget.setDefaultValue(default_value)
        widget.setClearMode(QgsFilterLineEdit.ClearMode.ClearToDefault)

        env_variable = PlgSettingsStructure.env_variable_used(attribute)
        if env_variable and os.getenv(env_variable):
            widget.setEnabled(False)
            widget.setToolTip(
                self.tr("Value defined from '{}' environnement variable").format(
                    env_variable
                )
            )
        else:
            widget.setEnabled(True)

    def load_settings(self):
        """Load options from QgsSettings into UI form."""
        settings = self.plg_settings.get_plg_settings()

        default_settings = PlgSettingsStructure()

        # global
        self.opt_debug.setChecked(settings.debug_mode)
        self.lbl_version_saved_value.setText(settings.version)

        # Define widget attribute
        for attribute, widget in self.attribute_widget_dict.items():
            self.set_attribute_with_env_variable(
                attribute, settings, default_settings, widget
            )

    def reset_settings(self):
        """Reset settings to default values (set in preferences.py module)."""
        default_settings = PlgSettingsStructure()

        # dump default settings into QgsSettings
        self.plg_settings.save_from_object(default_settings)

        # update the form
        self.load_settings()


class PlgOptionsFactory(QgsOptionsWidgetFactory):
    settingsUpdated = QtCore.pyqtSignal()
    """Factory for options widget."""

    def __init__(self):
        """Constructor."""
        super().__init__()

    def icon(self) -> QIcon:
        """Returns plugin icon, used to as tab icon in QGIS options tab widget.

        :return: _description_
        :rtype: QIcon
        """
        return QIcon(str(__icon_path__))

    def createWidget(self, parent) -> ConfigOptionsPage:
        """Create settings widget.

        :param parent: Qt parent where to include the options page.
        :type parent: QObject

        :return: options page for tab widget
        :rtype: ConfigOptionsPage
        """
        widget = ConfigOptionsPage(parent)
        widget.settingsUpdated.connect(lambda: self.settingsUpdated.emit())
        return widget

    def title(self) -> str:
        """Returns plugin title, used to name the tab in QGIS options tab widget.

        :return: plugin title from about module
        :rtype: str
        """
        return __title__

    def helpId(self) -> str:
        """Returns plugin help URL.

        :return: plugin homepage url from about module
        :rtype: str
        """
        return __uri_homepage__
