#! python3  # noqa: E265

"""
    Road Locator Filter.
"""

# standard library

# PyQGIS
from qgis.core import QgsExpression, QgsExpressionContext, QgsFeature, QgsLocatorFilter
from qgis.gui import QgisInterface
from qgis.utils import iface

# project
from locator_grand_lyon.core.vector_layer_locator_filter import VectorLayerLocatorFilter
from locator_grand_lyon.toolbelt import PlgLogger, PlgOptionsManager

# ############################################################################
# ########## Classes ###############
# ##################################


class RoadLocatorFilter(VectorLayerLocatorFilter):
    """QGIS Locator Filter subclass.

    :param iface: An interface instance that will be passed to this class which \
    provides the hook by which you can manipulate the QGIS application at run time.
    :type iface: QgisInterface
    """

    def __init__(self, iface: QgisInterface = iface):
        self.iface = iface
        self.log = PlgLogger().log

        super(VectorLayerLocatorFilter, self).__init__(iface)

    def priority(self) -> QgsLocatorFilter.Priority:
        """Filter priority. Controls the order of results in the locator.

        :return: filter priority
        :rtype: QgsLocatorFilter.Priority
        """
        return QgsLocatorFilter.Priority.High

    def get_search_layer_name(self) -> str:
        """Get search layer name

        :return: search layer name
        :rtype: str
        """
        plg_settings = PlgOptionsManager.get_plg_settings()
        return plg_settings.road_layer_name

    def get_search_column(self) -> str:
        """Define column used for search

        :return: search column
        :rtype: str
        """
        plg_settings = PlgOptionsManager.get_plg_settings()
        return plg_settings.road_search_column_name

    def get_display_string(self, feature: QgsFeature) -> str:
        """Get result display string from feature

        :param feature: feature
        :type feature: QgsFeature
        :return: display string
        :rtype: str
        """

        plg_settings = PlgOptionsManager.get_plg_settings()
        context = QgsExpressionContext()
        context.setFeature(feature)
        result = QgsExpression(plg_settings.road_search_display).evaluate(context)
        return result

    def name(self) -> str:
        """Returns the unique name for the filter. This should be an untranslated \
        string identifying the filter.

        :return: filter unique name
        :rtype: str
        """
        return self.__class__.__name__

    def clone(self) -> QgsLocatorFilter:
        """Creates a clone of the filter. New requests are always executed in a clone \
        of the original filter.

        :return: clone of the actual filter
        :rtype: QgsLocatorFilter
        """
        return RoadLocatorFilter(self.iface)

    def displayName(self) -> str:
        """Returns a translated, user-friendly name for the filter.

        :return: user-friendly name to be displayed
        :rtype: str
        """
        return self.tr("Road geocoder")

    def prefix(self) -> str:
        """Returns the search prefix character(s) for this filter. Prefix a search with \
        these characters will restrict the locator search to only include results from \
        this filter.

        :return: search prefix for the filter
        :rtype: str
        """
        return "route"
