#! python3  # noqa: E265

"""
    Road Intersection Locator Filter.
"""

# standard library

# PyQGIS
from typing import List, Union

from qgis.core import QgsExpression, QgsFeatureIterator, QgsGeometry, QgsLocatorFilter
from qgis.gui import QgisInterface
from qgis.utils import iface

# project
from locator_grand_lyon.core.road_locator_filter import RoadLocatorFilter
from locator_grand_lyon.toolbelt import PlgLogger, PlgOptionsManager

# ############################################################################
# ########## Classes ###############
# ##################################


class RoadIntersectionLocatorFilter(RoadLocatorFilter):

    """QGIS Locator Filter subclass.

    :param iface: An interface instance that will be passed to this class which \
    provides the hook by which you can manipulate the QGIS application at run time.
    :type iface: QgisInterface
    """

    def __init__(
        self,
        iface: QgisInterface = iface,
        intersection_ids=[],
        intersection_geometry: Union[QgsGeometry, None] = None,
    ):
        self.iface = iface
        self.log = PlgLogger().log

        self.intersection_ids = intersection_ids
        self.intersection_geometry = intersection_geometry

        super(RoadLocatorFilter, self).__init__(iface)

    def set_intersection_id(self, intersection_ids: List[int]) -> None:
        """Define intersection ids from previous search

        :param intersection_ids: previous search feature id
        :type intersection_ids: List[int]
        """
        self.intersection_ids = intersection_ids

    def set_intersection_geometry(
        self, intersection_geometry: Union[QgsGeometry, None]
    ) -> None:
        """Define intersection geometry for search

        :param intersection_geometry: intersection geometry for search
        :type intersection_geometry: QgsGeometry
        """
        self.intersection_geometry = intersection_geometry

    def name(self) -> str:
        """Returns the unique name for the filter. This should be an untranslated \
        string identifying the filter.

        :return: filter unique name
        :rtype: str
        """
        return self.__class__.__name__

    def clone(self) -> QgsLocatorFilter:
        """Creates a clone of the filter. New requests are always executed in a clone \
        of the original filter.

        :return: clone of the actual filter
        :rtype: QgsLocatorFilter
        """
        return RoadIntersectionLocatorFilter(
            self.iface, self.intersection_ids, self.intersection_geometry
        )

    def displayName(self) -> str:
        """Returns a translated, user-friendly name for the filter.

        :return: user-friendly name to be displayed
        :rtype: str
        """
        return self.tr("Road Intersection geocoder")

    def prefix(self) -> str:
        """Returns the search prefix character(s) for this filter. Prefix a search with \
        these characters will restrict the locator search to only include results from \
        this filter.

        :return: search prefix for the filter
        :rtype: str
        """
        return "intersection"

    def get_filter_expression(self, search: str) -> QgsExpression:
        """Get layer filter expression from search value

        Add a search from intersection geometry if defined

        :param search: search value
        :type search: str
        :return: expression for feature search
        :rtype: QgsExpression
        """
        if self.intersection_geometry is not None:
            search_column = self.get_search_column()
            # Define a QgsFeatureRequest with a filter expression
            expression_str = f"{search_column} ILIKE '%{search}%' AND intersects($geometry, geom_from_wkt('{self.intersection_geometry.asWkt()}'))"
            expression = QgsExpression(expression_str)
        else:
            expression = super().get_filter_expression(search)
        return expression

    def trigger_result_from_features(self, features: QgsFeatureIterator) -> None:
        """Trigger action result from found featues
        Features are selected from QgsVectorLayer
        """
        if self.intersection_geometry is not None:
            layer = self.get_search_layer()
            if layer is not None:
                layer.removeSelection()
                ids = [feat.id() for feat in features]
                ids += self.intersection_ids
                layer.selectByIds(ids)

                found_geom = QgsGeometry.collectGeometry(
                    [feat.geometry() for feat in features]
                )

                intersection = found_geom.intersection(self.intersection_geometry)
                center = (
                    self.iface.mapCanvas()
                    .mapSettings()
                    .layerToMapCoordinates(layer, intersection.centroid().asPoint())
                )
                # finally zoom actually
                self.iface.mapCanvas().setCenter(center)
                self.iface.mapCanvas().zoomScale(500)
        else:
            super().trigger_result_from_features(features)
