#! python3  # noqa: E265

"""
    Plugin settings.
"""


# standard
import os
from dataclasses import asdict, dataclass, fields

# PyQGIS
from qgis.core import QgsSettings

# package
import locator_grand_lyon.toolbelt.log_handler as log_hdlr
from locator_grand_lyon.__about__ import __title__, __version__

# ############################################################################
# ########## Classes ###############
# ##################################


@dataclass
class PlgEnvVariableSettings:
    """Plugin settings from environnement variable"""

    # Locator
    adress_layer_name: str = "QGIS_MGL_LOCATOR_ADRESS_LAYER_NAME"
    adress_search_column_name: str = "QGIS_MGL_LOCATOR_ADRESS_SEARCH_COLUMN_NAME"
    road_layer_name: str = "QGIS_MGL_LOCATOR_ROAD_LAYER_NAME"
    road_search_column_name: str = "QGIS_MGL_LOCATOR_ROAD_SEARCH_COLUMN_NAME"
    road_search_display: str = "QGIS_MGL_LOCATOR_ROAD_SEARCH_DISPLAY"

    # Cadastre
    commune_layer: str = "QGIS_MGL_LOCATOR_COMMUNE_LAYER"
    commune_id: str = "QGIS_MGL_LOCATOR_COMMUNE_ID"
    commune_display: str = "QGIS_MGL_LOCATOR_COMMUNE_DISPLAY"

    parcelle_layer: str = "QGIS_MGL_LOCATOR_PARCELLE_LAYER"
    parcelle_id: str = "QGIS_MGL_LOCATOR_PARCELLE_ID"
    parcelle_display: str = "QGIS_MGL_LOCATOR_PARCELLE_DISPLAY"

    parcelle_hist_layer: str = "QGIS_MGL_LOCATOR_PARCELLE_HIST_LAYER"
    parcelle_hist_id: str = "QGIS_MGL_LOCATOR_PARCELLE_HIST_ID"
    parcelle_hist_display: str = "QGIS_MGL_LOCATOR_PARCELLE_HIST_DISPLAY"

    # Cadastre filtering
    parcelle_section: str = "QGIS_MGL_LOCATOR_PARCELLE_SECTION"
    parcelle_commune: str = "QGIS_MGL_LOCATOR_PARCELLE_COMMUNE"


@dataclass
class PlgSettingsStructure:
    """Plugin settings structure and defaults values."""

    # global
    debug_mode: bool = False
    version: str = __version__

    # Locator
    min_search_length: int = 3
    adress_layer_name: str = "numerovoie"
    adress_search_column_name: str = "adressecomplete"
    road_layer_name: str = "voie"
    road_search_column_name: str = "nom"
    road_search_display: str = "concat(nom,' (',commune,')')"

    # Cadastre
    commune_layer: str = "commune"
    commune_id: str = "id_com"
    commune_display: str = "texte"

    parcelle_layer: str = "parcelle"
    parcelle_id: str = "id_par"
    parcelle_display: str = "parcelle"

    parcelle_hist_layer: str = "parcellehistorique"
    parcelle_hist_id: str = "idparcelle"
    parcelle_hist_display: str = "numero"

    # Cadastre filtering
    parcelle_section: str = "section"
    parcelle_commune: str = "insee"

    @staticmethod
    def env_variable_used(
        attribute: str,
        env_var_settings: PlgEnvVariableSettings = PlgEnvVariableSettings(),
    ) -> str:
        """Get environnement variable used for environnement variable settings

        :param attribute: attribute to check
        :type attribute: str
        :param env_var_settings: environnement variable settings, defaults to PlgEnvVariableSettings()
        :type env_var_settings: PlgEnvVariableSettings, optional
        :return: environnement variable used
        :rtype: str
        """
        settings_env_variable = asdict(env_var_settings)
        return settings_env_variable.get(attribute, "")


class PlgOptionsManager:
    @staticmethod
    def get_plg_settings() -> PlgSettingsStructure:
        """Load and return plugin settings as a dictionary. \
        Useful to get user preferences across plugin logic.

        :return: plugin settings
        :rtype: PlgSettingsStructure
        """
        # get dataclass fields definition
        settings_fields = fields(PlgSettingsStructure)

        # retrieve settings from QGIS/Qt
        settings = QgsSettings()
        settings.beginGroup(__title__)

        # map settings values to preferences object
        li_settings_values = []
        for i in settings_fields:
            value = settings.value(key=i.name, defaultValue=i.default, type=i.type)
            # If environnement variable used, get value from environnement variable
            env_variable = PlgSettingsStructure.env_variable_used(i.name)
            if env_variable:
                value = os.getenv(env_variable, value)
            li_settings_values.append(value)

        # instanciate new settings object
        options = PlgSettingsStructure(*li_settings_values)

        settings.endGroup()

        return options

    @staticmethod
    def get_value_from_key(key: str, default=None, exp_type=None):
        """Load and return plugin settings as a dictionary. \
        Useful to get user preferences across plugin logic.

        :return: plugin settings value matching key
        """
        if not hasattr(PlgSettingsStructure, key):
            log_hdlr.PlgLogger.log(
                message="Bad settings key. Must be one of: {}".format(
                    ",".join(PlgSettingsStructure._fields)
                ),
                log_level=1,
            )
            return None

        settings = QgsSettings()
        settings.beginGroup(__title__)

        try:
            out_value = settings.value(key=key, defaultValue=default, type=exp_type)
        except Exception as err:
            log_hdlr.PlgLogger.log(
                message="Error occurred trying to get settings: {}.Trace: {}".format(
                    key, err
                )
            )
            out_value = None

        settings.endGroup()

        return out_value

    @classmethod
    def set_value_from_key(cls, key: str, value) -> bool:
        """Set plugin QSettings value using the key.

        :param key: QSettings key
        :type key: str
        :param value: value to set
        :type value: depending on the settings
        :return: operation status
        :rtype: bool
        """
        if not hasattr(PlgSettingsStructure, key):
            log_hdlr.PlgLogger.log(
                message="Bad settings key. Must be one of: {}".format(
                    ",".join(PlgSettingsStructure._fields)
                ),
                log_level=2,
            )
            return False

        settings = QgsSettings()
        settings.beginGroup(__title__)

        try:
            settings.setValue(key, value)
            out_value = True
        except Exception as err:
            log_hdlr.PlgLogger.log(
                message="Error occurred trying to set settings: {}.Trace: {}".format(
                    key, err
                )
            )
            out_value = False

        settings.endGroup()

        return out_value

    @classmethod
    def save_from_object(cls, plugin_settings_obj: PlgSettingsStructure):
        """Load and return plugin settings as a dictionary. \
        Useful to get user preferences across plugin logic.

        :return: plugin settings value matching key
        """
        settings = QgsSettings()
        settings.beginGroup(__title__)

        for k, v in asdict(plugin_settings_obj).items():
            cls.set_value_from_key(k, v)

        settings.endGroup()
