#! python3  # noqa: E265

"""
    Locator Filter.
"""

# standard library
from typing import Union

# PyQGIS
from qgis.core import (
    QgsExpression,
    QgsFeature,
    QgsFeatureIterator,
    QgsFeatureRequest,
    QgsFeedback,
    QgsGeometry,
    QgsLocatorContext,
    QgsLocatorFilter,
    QgsLocatorResult,
    QgsProject,
    QgsVectorLayer,
)
from qgis.gui import QgisInterface
from qgis.PyQt import QtCore
from qgis.PyQt.QtWidgets import QWidget
from qgis.utils import iface

# project
from locator_grand_lyon.__about__ import __title__
from locator_grand_lyon.toolbelt import PlgLogger, PlgOptionsManager

# ############################################################################
# ########## Classes ###############
# ##################################


class VectorLayerLocatorFilter(QgsLocatorFilter):
    resultSelected = QtCore.pyqtSignal(list, QgsGeometry)
    """Abstract class for QgsLocatorFilter on an QgsVectorLayer

    Must implement at least these method:
    - get_search_layer_name
    - get_search_column
    - get_display_string

    :param iface: An interface instance that will be passed to this class which \
    provides the hook by which you can manipulate the QGIS application at run time.
    :type iface: QgisInterface
    """

    def __init__(self, iface: QgisInterface = iface):
        self.iface = iface
        self.log = PlgLogger().log

        super(QgsLocatorFilter, self).__init__()

    def get_search_layer_name(self) -> str:
        """Get search layer name

        :raises NotImplementedError: method not implemented in derived class
        :return: search layer name
        :rtype: str
        """
        raise NotImplementedError(
            "get_search_layer_name must be implemented in VectorLayerLocatorFilter derived classes"
        )

    def get_search_column(self) -> str:
        """Define column used for search

        :raises NotImplementedError: method not implemented in derived class
        :return: search layer name
        :rtype: str
        """
        raise NotImplementedError(
            "get_search_column must be implemented in VectorLayerLocatorFilter derived classes"
        )

    def get_display_string(self, feature: QgsFeature) -> str:
        """Get result display string from feature

        :raises NotImplementedError: method not implemented in derived class
        :param feature: feature
        :type feature: QgsFeature
        :return: display string
        :rtype: str
        """
        raise NotImplementedError(
            "get_display_string must be implemented in VectorLayerLocatorFilter derived classes"
        )

    def get_group_by_column(self) -> str:
        """Column used to group feature

        :return: group by column
        :rtype: str
        """
        return ""

    def hasConfigWidget(self) -> bool:
        """Should return True if the filter has a configuration widget.

        :return: configuration widget available
        :rtype: bool
        """
        return True

    def get_search_layer(self) -> Union[QgsVectorLayer, None]:
        """Get search layer

        :return: search layer, None is not available
        :rtype: Union[QgsVectorLayer, None]
        """
        # Get the current project
        project = QgsProject.instance()

        # Specify the name of the layer you're looking for
        layer_name = self.get_search_layer_name()

        # Use layerByName() to get a reference to the layer by name
        layers = project.mapLayersByName(layer_name)
        if len(layers) > 0:
            return layers[0]
        else:
            return None

    def can_be_used(self) -> bool:
        """Check if the locator filter can be used

        :return: True if the locator filter can be used, False otherwise
        :rtype: bool
        """
        return self.get_search_layer() is not None

    def openConfigWidget(self, parent: QWidget = None):
        """Opens the configuration widget for the filter (if it has one), with the \
        specified parent widget. self.hasConfigWidget() must return True.

        :param parent: prent widget, defaults to None
        :type parent: QWidget, optional
        """
        self.iface.showOptionsDialog(
            parent=parent, currentPage=f"mOptionsPage{__title__}"
        )

    def check_search(self, search: str) -> bool:
        """Check search from current configuration

        :param search: search value
        :type search: str
        :return: True if search is valid, False otherwise
        :rtype: bool
        """
        plg_settings = PlgOptionsManager.get_plg_settings()
        # ignore if search terms is inferior than minimum number of chars
        if len(search) < plg_settings.min_search_length:
            self.log(
                message=self.tr("API search not triggered. Reason: ")
                + self.tr(
                    "minimum chars {} not reached: {}".format(
                        plg_settings.min_search_length, len(search)
                    )
                ),
                log_level=4,
            )
            return False

        # ignore if search terms is equal to the prefix
        if search.rstrip() == self.prefix:
            self.log(
                message=self.tr("API search not triggered. Reason: ")
                + self.tr("search term is matching the prefix."),
                log_level=4,
            )
            return False

        return True

    def get_filter_expression(self, search: str) -> QgsExpression:
        """Get layer filter expression from search value

        :param search: search value
        :type search: str
        :return: expression for feature search
        :rtype: QgsExpression
        """
        search_column = self.get_search_column()
        expression = QgsExpression(f"{search_column} ILIKE '%{search}%'")
        return expression

    def _send_result_with_group_by(
        self, group_by_column: str, features: QgsFeatureIterator, limit: int = 20
    ) -> None:
        """Send resultFetched signal with features grouped by a column

        :param group_by_column: group by column
        :type group_by_column: str
        :param features:  found features
        :type features: QgsFeatureIterator
        :param limit: limit number of feature send
        :type limit: int, optional
        """
        group_dict = {}
        for feature in features:
            group_by_value = str(feature[group_by_column])
            if group_by_value not in group_dict:
                group_dict[group_by_value] = {}
                group_dict[group_by_value]["features"] = [feature]
                group_dict[group_by_value]["displayString"] = self.get_display_string(
                    feature
                )
            else:
                group_dict[group_by_value]["features"].append(feature)

        nb_val = 0
        for _, res in group_dict.items():
            result = QgsLocatorResult()
            result.displayString = res["displayString"]

            # use the json full item as userData, so all info is in it:
            doc = {"features": res["features"]}
            result.userData = doc

            self.resultFetched.emit(result)

            nb_val = nb_val + 1
            if nb_val > limit:
                return

    def send_result_from_features(
        self, features: QgsFeatureIterator, limit: int = 20
    ) -> None:
        """Send resultFetched signal with found feature and displayString

        :param features: found features
        :type features: QgsFeatureIterator
        :param limit
        :type int: limit number of feature send
        """
        # regroup feature by column if defined
        group_by_column = self.get_group_by_column()
        if group_by_column:
            self._send_result_with_group_by(group_by_column, features, limit)
        else:
            nb_val = 0
            for feature in features:
                result = QgsLocatorResult()
                result.displayString = self.get_display_string(feature)

                # use the json full item as userData, so all info is in it:
                doc = {"features": [feature]}
                result.userData = doc
                self.resultFetched.emit(result)

                nb_val = nb_val + 1
                if nb_val > limit:
                    return

    def fetchResults(
        self, search: str, context: QgsLocatorContext, feedback: QgsFeedback
    ):
        """Retrieves the filter results for a specified search string. The context \
        argument encapsulates the context relating to the search (such as a map extent \
        to prioritize). \

        Implementations of fetchResults() should emit the resultFetched() signal \
        whenever they encounter a matching result. \
        Subclasses should periodically check the feedback object to determine whether \
        the query has been canceled. If so, the subclass should return from this method \
        as soon as possible. This will be called from a background thread unless \
        flags() returns the QgsLocatorFilter.FlagFast flag.

        :param search: text entered by the end-user into the locator line edit
        :type search: str
        :param context: [description]
        :type context: QgsLocatorContext
        :param feedback: [description]
        :type feedback: QgsFeedback
        """
        if not self.check_search(search):
            return

        # process response
        try:
            layer = self.get_search_layer()

            # Check if the layer is valid
            if layer is not None:
                expression = self.get_filter_expression(search)
                request = QgsFeatureRequest(expression)

                # Use getFeatures() to get the features that match the request
                features = layer.getFeatures(request)

                # Iterate through the features and do something with them
                self.send_result_from_features(features)

        except Exception as exc:
            print(f"failure {str(exc)=}")
            self.log(message="Response processing failed.", log_level=1)
            return

    def trigger_result_from_features(self, features: QgsFeatureIterator) -> None:
        """Trigger action result from found featues
        Features are selected from QgsVectorLayer
        """
        layer = self.get_search_layer()
        if layer is not None:
            layer.removeSelection()
            ids = [feat.id() for feat in features]
            layer.selectByIds(ids)

            # finally zoom actually
            self.iface.mapCanvas().zoomToSelected(layer)

            # Emit signal with selected feature
            ids = [feat.id() for feat in features]
            full_geom = QgsGeometry.collectGeometry(
                [feat.geometry() for feat in features]
            )
            self.resultSelected.emit(ids, full_geom)

    def triggerResult(self, result: QgsLocatorResult):
        """Triggers a filter result from this filter. This is called when one of the \
        results obtained by a call to fetchResults() is triggered by a user. \
        The filter subclass must implement logic here to perform the desired operation \
        for the search result. E.g. a file search filter would open file associated \
        with the triggered result.

        :param result: result selected by user
        :type result: QgsLocatorResult
        """
        # Newer Version of PyQT does not expose the .userData (Leading to core dump)
        # Try via get Function, otherwise access attribute
        try:
            doc = result.getUserData()
        except Exception as err:
            self.log(
                message=self.tr(
                    "Something went wrong during result deserialization: {}. "
                    "Trying another method...".format(err)
                ),
                log_level=2,
            )
        self.trigger_result_from_features(doc["features"])
