"""
Linear Referencing plugin
(c) Copyright 2008 Martin Dobias

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.       
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from dlgcalcmeasure_ui import Ui_DlgCalcMeasure

from dlgdberror import DlgDbError
import postgis

from mylogger import MyLogger

from qgis.core import *
from qgis.gui import QgsMessageViewer

class DlgCalcMeasure(QDialog, Ui_DlgCalcMeasure):

	def __init__(self, db, parent=None):
		QDialog.__init__(self, parent)
		self.db = db
		
		self.setupUi(self)
		
		self.connect(self.buttonBox, SIGNAL("accepted()"), self.onOK)

		self.connect(self.cboInputLayer, SIGNAL("currentIndexChanged(int)"), self.populateOutputFields)					

		self.populateInputTables()
		self.populateRoutes()
		
	def populateInputTables(self):
		""" populate combo box with possible events tables from database """
		tables = self.db.list_aspatial_tables()
			
		self.cboInputLayer.clear()
		for table in sorted(tables):
			self.cboInputLayer.addItem(table, QVariant())	
		for table in sorted(self.db.list_spatial_tables('POINT')):
			self.cboInputLayer.addItem(table, QVariant('POINT'))	

	def populateRoutes(self):
		self.cboRoute.clear()
		for route in sorted(self.db.list_routes_2()):
			self.cboRoute.addItem(route)
	
	def populateOutputFields(self):
		table = self.cboInputLayer.currentText()
		geomType = self.cboInputLayer.itemData(self.cboInputLayer.currentIndex()).toString()

		self.populateLayerFields(table, self.cboOutputRouteId)
		self.populateLayerFields(table, self.cboOutputMeasure)		
		self.populateLayerFields(table, self.cboX)
		self.populateLayerFields(table, self.cboY)

		enable = geomType != "POINT"
		self.cboX.setEnabled(enable)
		self.cboY.setEnabled(enable)


	def populateLayerFields(self, layer, combo):
		combo.clear()
		for field in self.db.list_table_columns(layer):
			combo.addItem(field)

	def onOK(self):
		""" do the processing """
		
		route_table = self.cboRoute.currentText()
		input_table = self.cboInputLayer.currentText()
		input_table_type = self.cboInputLayer.itemData(self.cboInputLayer.currentIndex()).toString()
		input_x = self.cboX.currentText()
		input_y = self.cboY.currentText()
		out_measure = self.cboOutputMeasure.currentText()
		out_route = self.cboOutputRouteId.currentText()
		
		# TODO: custom tolerance
		tolerance = 1000
		
		QApplication.instance().setOverrideCursor(QCursor(Qt.WaitCursor))
		try:
			if input_table_type == "POINT":
				log = self.calcMeasureFromPointLayer(input_table, route_table, tolerance, out_measure, out_route)
			else:
				log = self.calcMeasureFromTable(input_table, input_x, input_y, route_table, tolerance, out_measure, out_route)

		except postgis.DatabaseError, e:
			QApplication.instance().restoreOverrideCursor()
			DlgDbError.showError(e)
			return
		
		QApplication.instance().restoreOverrideCursor()
		
		if log is None:
			return
		
		# show the log
		vw = QgsMessageViewer(self)
		vw.setTitle("Measure calculation log")
		vw.setMessageAsHtml(log.dump())
		vw.exec_()

		
		self.accept()


	def calcMeasureFromTable(self, input_table, input_x, input_y, route_table, tolerance, out_measure=None, out_route=None):
		"""
		- load rows from table one by one
		- align point to reference route
		- update db record
		"""
		
		input_pkey = self.db.get_table_primary_key(input_table)
		if input_pkey is None:
			QMessageBox.critical(self, "error", "The input table doesn't have primary key, can't be used!")
			return
		
		sqrtolerance = tolerance ** 2
		
		# construct route_geoms: route_name -> geometry
		route = self.db.get_route_info(route_table)
		route_geoms = self.getRouteGeoms(route)
		if route_geoms is None:
			QMessageBox.critical(self, "error", "Couldn't retrieve route info!")
			return
		
		log = MyLogger()
		
		c = self.db.con.cursor()
		c.execute(u"SELECT %s,%s,%s FROM %s" % (self.db._quote(input_pkey), self.db._quote(input_x), self.db._quote(input_y), self.db._quote(input_table)))
		
		for row in c.fetchall():
			(pkey,x,y) = row
			
			res = self.calcMeasureForPoint(QgsPoint(x,y), route, route_geoms, sqrtolerance,log)
			if res is None:
				log.write("%d: couldn't align" % pkey)
				continue
			(m, minRouteName) = res
			log.write("%d: route %s, M %f" % (pkey, minRouteName, m))
			
			# update DB record
			if out_measure and out_route:
				self.db.update_row(input_table, input_pkey, pkey, { out_measure : m, out_route : minRouteName })
			
		c.close()
		
		return log

	def calcMeasureFromPointLayer(self, input_point_layer, route_table, tolerance, out_measure=None, out_route=None):
		"""
		- align point to reference route
		- update db record
		"""
		
		input_pkey = self.db.get_table_primary_key(input_point_layer)
		if input_pkey is None:
			QMessageBox.critical(self, "error", "The input table doesn't have primary key, can't be used!")
			return
		
		sqrtolerance = tolerance ** 2
		
		# construct route_geoms: route_name -> geometry
		route = self.db.get_route_info(route_table)
		route_geoms = self.getRouteGeoms(route)
		if route_geoms is None:
			QMessageBox.critical(self, "error", "Couldn't retrieve route info!")
			return
		
		log = MyLogger()

		# get point layer geometry column
		geomCol = self.db.get_layer_geometry_column(input_point_layer)
		
		# load point layer
		uri = QgsDataSourceURI()
		uri.setConnection(self.db.host, str(self.db.port), self.db.dbname, self.db.user, self.db.passwd)
		uri.setDataSource('public', input_point_layer, geomCol)
		pointLayer = QgsVectorLayer(uri.uri(), "point_layer", "postgres")
		if not pointLayer.isValid():
			QMessageBox.critical(self, "error", "Point layer invalid!")
			return None
		
		pointLayer.select([])
		f = QgsFeature()
		while pointLayer.nextFeature(f):
			point = f.geometry().asPoint()
			pkey = f.id()
			
			res = self.calcMeasureForPoint(point, route, route_geoms, sqrtolerance,log)
			if res is None:
				log.write("%d: couldn't align" % pkey)
				continue
			(m, minRouteName) = res
			log.write("%d: route %s, M %f" % (pkey, minRouteName, m))
			
			# update DB record
			if out_measure and out_route:
				self.db.update_row(input_point_layer, input_pkey, pkey, { out_measure : m, out_route : minRouteName })
		
		return log		
		
	def calcMeasureForPoint(self, pnt, route, route_geoms, sqrtolerance, log):
		
		# align point to reference route
		minsqrdist = None
		for route_name,g in route_geoms.iteritems():
			(sqrdist, closestPoint, beforeVertexNum) = g.closestSegmentWithContext(pnt)
			if sqrdist < 0: continue
			if sqrdist > sqrtolerance:
				#print "too far",route_name, sqrdist
				continue
			
			if minsqrdist == None or sqrdist < minsqrdist:
				minsqrdist = sqrdist
				minClosestPoint = closestPoint
				minRouteName = route_name
				minBeforeVertex = beforeVertexNum
		
		if minsqrdist == None:
			return None
		
		m = self.db.get_route_m_interpolation(route, minRouteName, minClosestPoint, minBeforeVertex)
		return (m, minRouteName)

	
	def getRouteGeoms(self, route):
		""" construct a dictionary containing geometry of all route features """
		
		uri = QgsDataSourceURI()
		uri.setConnection(self.db.host, str(self.db.port), self.db.dbname, self.db.user, self.db.passwd)
		uri.setDataSource('public', route.table, route.geom_column)
		routeLayer = QgsVectorLayer(uri.uri(), "route", "postgres")
		if not routeLayer.isValid():
			QMessageBox.critical(self, "error", "Route layer invalid!")
			return None
		
		route_geoms = {}
		idx_id_column = routeLayer.dataProvider().fieldNameIndex(route.id_column)
		routeLayer.select([idx_id_column])
		f = QgsFeature()
		while routeLayer.nextFeature(f):
			route_name = unicode(f.attributeMap()[idx_id_column].toString())
			route_geoms[route_name] = QgsGeometry(f.geometry())
			
		return route_geoms
