"""
Linear Referencing plugin
(c) Copyright 2008 Martin Dobias

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.       
"""


from PyQt4.QtCore import *
from PyQt4.QtGui import *

from dlgconfig_ui import Ui_DlgConfig

import postgis


class DlgConfig(QDialog, Ui_DlgConfig):

	def __init__(self, db, parent=None):
		QDialog.__init__(self, parent)
		
		self.setupUi(self)
		
		self.connect(self.listRoutes, SIGNAL("itemDoubleClicked(QListWidgetItem *)"), self.loadRoute)
		self.connect(self.btnCheck, SIGNAL("clicked()"), self.checkDatabase)
		self.connect(self.btnDeleteRoute, SIGNAL("clicked()"), self.deleteRoute)
		self.connect(self.buttonBox, SIGNAL("accepted()"), self.onOK)
		
		self.populateConnections()
		
		self.db = db
		self.databaseOk = (db is not None)
		
	def loadRoute(self, item):
		if item is None:
			item = self.listRoutes.currentItem()
			if item is None:
				return

		table_name = str(item.text())
		route = self.db.get_route_info(table_name)
		if route is None:
			return

		# load route layer
		import qgis.core
		uri = qgis.core.QgsDataSourceURI()
		uri.setConnection(self.db.host, str(self.db.port), self.db.dbname, self.db.user, self.db.passwd)
		uri.setDataSource('public', route.table, route.geom_column)
		route_lines = qgis.core.QgsVectorLayer(uri.uri(), route.table, "postgres")
		if not route_lines.isValid():
			QMessageBox.critical(self, "error", "route layer is not valid!")
			return
		qgis.core.QgsMapLayerRegistry.instance().addMapLayer(route_lines)


	def populateConnections(self):

		settings = QSettings()
		
		settings.beginGroup("/PostgreSQL/connections")
		keys = settings.childGroups()
		
		self.cboConnection.clear()
		for key in keys:
			self.cboConnection.addItem(key)
			
		settings.endGroup()
		
		# select correct item
		con = settings.value("/LRS/db_connection").toString()
		for i in xrange(self.cboConnection.count()):
			if self.cboConnection.itemText(i) == con:
				self.cboConnection.setCurrentIndex(i)
				break
	
	def deleteRoute(self):
		
		item = self.listRoutes.currentItem()
		if item is None:
			QMessageBox.critical(self, "sorry", "No route for deletion selected!")
			return
		
		route_name = str(item.text())
		res = self.db.delete_route(route_name)
		
		if res:
			QMessageBox.information(self, "done", "Route '%s' has been deleted." % route_name)
		else:
			QMessageBox.information(self, "done", "Couldn't delete table for route '%s'. Already deleted?" % route_name)
		
		self.populateRoutes()


	def onOK(self):
		""" save config """
		
		# check that database is okay,
		if not self.databaseOk:
			self.checkDatabase()
			if not self.databaseOk:
				return
		
		settings = QSettings()
		settings.setValue("/LRS/db_connection", QVariant(self.cboConnection.currentText()))
		
		self.accept()
	
	def populateRoutes(self):
		routes = self.db.list_routes_2()
		
		self.listRoutes.clear()
		if len(routes) > 0:
			for r in routes:
				self.listRoutes.addItem(r)
		else:
			self.listRoutes.addItem("(no routes)")
		
		self.lblStatus.setText("Connection OK, metadata OK")
		self.databaseOk = True

	
	def checkDatabase(self):
		""" connect, check for routes """
		
		settings = QSettings()
		#selected = str(settings.value("/LRS/db_connection").toString())
		
		selected = self.cboConnection.currentText()
		
		# get connection details from QSettings
		key = "/PostgreSQL/connections/" + selected
		get_value = lambda x: settings.value( key + "/" + x )
		get_value_str = lambda x: str(get_value(x).toString())
		host, database, username, password = map(get_value_str, ["host", "database", "username", "password"])
		port = get_value("port").toInt()[0]
		
		# connect to DB
		try:
			self.db = postgis.PostGIS(host=host, port=port, dbname=database, user=username, passwd=password)
			self.lblStatus.setText("Connection OK")
		except postgis.DatabaseError, e:
			QMessageBox.critical(self, "error", "Couldn't connect to database:\n"+e.message)
			self.lblStatus.setText("Connection error")
			self.databaseOk = False
			return
		
		# check for lin_ref_routes table
		try:
			self.populateRoutes()
		except postgis.DatabaseError, e:
			res = QMessageBox.warning(self, "problem", "Linear referencing needs a support table in database (lin_ref_routes) to work correctly.\n\nDo you wish to add this table?", QMessageBox.Yes | QMessageBox.No)
			
			if res != QMessageBox.Yes:
				self.lblStatus.setText("Connection OK, missing metadata table")
				self.databaseOk = False
				return
				
			self.db.create_routes_table()
			self.populateRoutes()

		# the rest of the check is done in populateRoutes()
		
