"""
Linear Referencing plugin
(c) Copyright 2008 Martin Dobias

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.       
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *

import resources_rc

from dlgconfig import DlgConfig
from dlgaddrouteevents import DlgAddRouteEvents
from dlgcalibrateroute import DlgCalibrateRoute
from dlgcalcmeasure import DlgCalcMeasure

from routeinfotool import RouteInfoTool

import postgis


class LinearReferencingPlugin:

	def __init__(self, iface):
		# Save reference to the QGIS interface
		self.iface = iface
	
	
	def initGui(self):
		
		# create actions
		self.actionConfigure = QAction(QIcon(":/LRS/icons/configure.png"), "&Configure", self.iface.mainWindow())
		self.actionCalibrateRoute = QAction(QIcon(":/LRS/icons/calibrate_route.png"), "Ca&librate route", self.iface.mainWindow())
		self.actionAddRouteEvents = QAction(QIcon(":/LRS/icons/add_events.png"), "&Add route events", self.iface.mainWindow())
		self.actionCalcMeasure = QAction(QIcon(":/LRS/icons/calc_measure.png"), "Calculate &measure", self.iface.mainWindow())
		self.actionRouteInfoTool = QAction(QIcon(":/LRS/icons/info_route.png"), "&Route info", self.iface.mainWindow())
		self.actionAbout = QAction(QIcon(":/LRS/icons/about.png"), "About", self.iface.mainWindow())
		
		self.actionRouteInfoTool.setCheckable(True)
	
		# add a toolbar
		self.toolBar = self.iface.addToolBar("Linear Referencing")
		self.toolBar.setObjectName("Linear Referencing")
		self.toolBar.addAction(self.actionConfigure)
		self.toolBar.addAction(self.actionCalibrateRoute)
		self.toolBar.addAction(self.actionAddRouteEvents)
		self.toolBar.addAction(self.actionCalcMeasure)
		self.toolBar.addAction(self.actionRouteInfoTool)

		# create connections
		QObject.connect(self.actionConfigure, SIGNAL("triggered()"), self.configure)
		QObject.connect(self.actionCalibrateRoute, SIGNAL("triggered()"), self.calibrateRoute)
		QObject.connect(self.actionAddRouteEvents, SIGNAL("triggered()"), self.addRouteEvents)
		QObject.connect(self.actionCalcMeasure, SIGNAL("triggered()"), self.calcMeasure)
		QObject.connect(self.actionRouteInfoTool, SIGNAL("triggered()"), self.routeInfoTool)
		QObject.connect(self.actionAbout, SIGNAL("triggered()"), self.about)
		
		# add menu items
		if hasattr(self.iface, "addPluginToDatabaseMenu"):
			self.iface.addPluginToDatabaseMenu("&Linear referencing", self.actionConfigure)
			self.iface.addPluginToDatabaseMenu("&Linear referencing", self.actionCalibrateRoute)
			self.iface.addPluginToDatabaseMenu("&Linear referencing", self.actionAddRouteEvents)
			self.iface.addPluginToDatabaseMenu("&Linear referencing", self.actionCalcMeasure)
			self.iface.addPluginToDatabaseMenu("&Linear referencing", self.actionRouteInfoTool)
			self.iface.addPluginToDatabaseMenu("&Linear referencing", self.actionAbout)
		else:
			self.iface.addPluginToMenu("&Linear referencing", self.actionConfigure)
			self.iface.addPluginToMenu("&Linear referencing", self.actionCalibrateRoute)
			self.iface.addPluginToMenu("&Linear referencing", self.actionAddRouteEvents)
			self.iface.addPluginToMenu("&Linear referencing", self.actionCalcMeasure)
			self.iface.addPluginToMenu("&Linear referencing", self.actionRouteInfoTool)
			self.iface.addPluginToMenu("&Linear referencing", self.actionAbout)
		
		self.db = None
		
		self.routeTool = RouteInfoTool(self.iface.mapCanvas())
		self.routeTool.setAction(self.actionRouteInfoTool)
		

		# TODO: check for psycopg2
	
	

	
	def unload(self):
		
		# remove plugin menu items
		if hasattr(self.iface, "removePluginDatabaseMenu"):
			self.iface.removePluginDatabaseMenu("&Linear referencing", self.actionConfigure)
			self.iface.removePluginDatabaseMenu("&Linear referencing", self.actionCalibrateRoute)
			self.iface.removePluginDatabaseMenu("&Linear referencing", self.actionAddRouteEvents)
			self.iface.removePluginDatabaseMenu("&Linear referencing", self.actionCalcMeasure)
			self.iface.removePluginDatabaseMenu("&Linear referencing", self.actionRouteInfoTool)
			self.iface.removePluginDatabaseMenu("&Linear referencing", self.actionAbout)
		else:
			self.iface.removePluginMenu("&Linear referencing", self.actionConfigure)
			self.iface.removePluginMenu("&Linear referencing", self.actionCalibrateRoute)
			self.iface.removePluginMenu("&Linear referencing", self.actionAddRouteEvents)
			self.iface.removePluginMenu("&Linear referencing", self.actionCalcMeasure)
			self.iface.removePluginMenu("&Linear referencing", self.actionRouteInfoTool)
			self.iface.removePluginMenu("&Linear referencing", self.actionAbout)
		
		del self.routeTool
		#del self.eventTool
		
		# remove toolbar
		del self.toolBar


	def about(self):
		from DlgAbout import DlgAbout
		DlgAbout(self.iface.mainWindow()).exec_()
		
	
	def ensureDbConnected(self):
		""" if we're not connected to database yet, do it right now! """
		if self.db:
			return True
		
		settings = QSettings()
		selected = str(settings.value("/LRS/db_connection").toString())
		
		# get connection details from QSettings
		key = "/PostgreSQL/connections/" + selected
		get_value = lambda x: settings.value( key + "/" + x )
		get_value_str = lambda x: str(get_value(x).toString())
		host, database, username, password = map(get_value_str, ["host", "database", "username", "password"])
		port = get_value("port").toInt()[0]
		
		# connect to DB
		try:
			self.db = postgis.PostGIS(host=host, port=port, dbname=database, user=username, passwd=password)
		except postgis.DatabaseError, e:
			QMessageBox.critical(self.iface.mainWindow(), "error", "Couldn't connect to database:\n"+e.message)
			return False
		
		try:
			self.db.list_routes_2()
		except postgis.DatabaseError, e:
			self.db = None
			QMessageBox.critical(self.iface.mainWindow(), "error", "Metadata table is missing. Please configure your database properly.")
			return False
		
		return True
	
	
	def configure(self): 
		dlg = DlgConfig(self.db, self.iface.mainWindow())
		if not dlg.exec_():
			return
		
		if dlg.databaseOk:
			self.db = dlg.db
		else:
			self.db = None
		
		
	
	def calibrateRoute(self):
		""" open dialog to create a new route and calibrate it """
		if not self.ensureDbConnected():
			return
		
		dlg = DlgCalibrateRoute(self.db, self.iface.mainWindow())
		dlg.exec_()
	
	
	def addRouteEvents(self):
		""" open dialog to add route events layer """
		if not self.ensureDbConnected():
			return
		
		routes = self.db.list_routes_2()
		if len(routes) == 0:
			QMessageBox.warning(self.iface.mainWindow(), "sorry", "There are no routes defined yet.")
			return
		
		dlg = DlgAddRouteEvents(self.db, self.iface, self.iface.mainWindow())
		dlg.exec_()
	
	
	def calcMeasure(self):
		""" open dialog for calculation of M value for existing vector layer """
		if not self.ensureDbConnected():
			return
		
		dlg = DlgCalcMeasure(self.db, self.iface.mainWindow())
		dlg.exec_()
	
	
	def routeInfoTool(self):
		""" activate route info tool """
		if not self.ensureDbConnected():
			return
		
		self.routeTool.setDatabase(self.db)
		self.iface.mapCanvas().setMapTool(self.routeTool)
	
