"""
Linear Referencing plugin
(c) Copyright 2008 Martin Dobias

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.       
"""

import math


class Point(object):
	
	def __init__(self, x,y):
		self.x = x
		self.y = y
				
	def dist(self, pnt):
		""" distance from this to another point """
		return math.sqrt((self.x-pnt.x)*(self.x-pnt.x) + (self.y-pnt.y)*(self.y-pnt.y))
		
	def distFromPolyline(self, pl):
		""" distance from polyline (array of points) to point """
		
		if len(pl) < 2:
			raise ValueError
		
		minDist = 99999
		p1 = pl[0]
		for p2 in pl[1:]:
			(dist, pNear) = self.distFromVector(Line(p1.x,p1.y,p2.x,p2.y))
			if dist < minDist:
				minDist = dist
			p1 = p2
		
		return minDist
		
	def distFromLine(self, line):
		""" distance from point to line """
		try:
			perpL = line.perpLineFromPoint(self)
		except ValueError:
			return 0 # no distance
		return self.dist(perpL.p2)
		
	def distFromVector(self, line):
		""" calculates point's distance from vector (has start- and end-point) """
		
		# check whether perpendicular line crosses line between endpoints
		# if so, use distance from line
		intP = line.perpIntersection(self)
				
		if line.t == None:
			if isInInterval(intP.y, line.p1.y,line.p2.y):
				return (self.distFromLine(line), intP)
		elif line.t == 0:
			if isInInterval(intP.x, line.p1.x, line.p2.x):
				return (self.distFromLine(line), intP)
		else:
			y = line.p1.y + (intP.x - line.p1.x)*line.t
			if isInInterval(y, line.p1.y,line.p2.y):
				return (self.distFromLine(line), intP)
		
		# use distance from one of the endpoints
		m1 = self.dist(line.p1)
		m2 = self.dist(line.p2)
		if m1 < m2:
			return (m1, line.p1)
		else:
			return (m2, line.p2)


class Line(object):
		
	def __init__(self,x1,y1,x2,y2):
		
		if x1 == x2 and y1 == y2:
				raise ValueError
		
		self.p1 = Point(x1,y1)
		self.p2 = Point(x2,y2)
		
		# tangent
		self.t = None if x1 == x2 else float(y2-y1)/(x2-x1)
				
		# angle
		if self.t == None:
			# angle is 90 or 270
			self.angle = math.pi/2 if y2 >= y1 else math.pi*3/2
		elif self.t >= 0:
			self.angle = math.atan(self.t) if y2 >= y1 else math.pi + math.atan(self.t)
		else: # self.t < 0
			# atan is positive / negative
			self.angle = math.pi + math.atan(self.t) if y2 >= y1 else 2*math.pi + math.atan(self.t)
					
	
	def tPerp(self):
		""" perpendicular tangent """
		if self.t == None:
				return 0
		elif self.t == 0:
				return None
		else:
				return math.tan(math.atan(self.t) + math.pi/2)
			
	def perpIntersection(self, pnt):
		""" returns point that is intersection of line and perpendicular line from pnt """
		t = self.tPerp()
		x0 = pnt.x
		y0 = pnt.y
		
		# special case - vertical lines (undefined tangent)
		if t == None:
				return Point(x0,self.p1.y)
		elif self.t == None:
				return Point(self.p1.x,y0)
		
		x1 = (y0 - self.p1.y + self.p1.x*self.t - x0*t) / (self.t - t)
		y1 = y0 + t*(x1-x0)
		return Point(x1,y1)

	def perpLineFromPoint(self, pnt):
		""" returns perpendicular line that goes through the point """
		pnt2 = self.perpIntersection(pnt)
		return Line(pnt.x, pnt.y, pnt2.x, pnt2.y)
	
		

def isInInterval(x, v1, v2):
	""" returns whether x is between v1 and v2 """
	return (x >= v1 and x <= v2) or (x >= v2 and x <= v1)
	
	
def nearestPoint(pnt, polyline):
	""" returns tuple of nearest point on polyline and index of segment where it belongs """
	
	if len(polyline) < 2:
			raise ValueError
	
	i = 0
	minI = None
	minDist = 999999999

	p1 = polyline[0]
	for p2 in polyline[1:]:
			
			(dist, pt) = pnt.distFromVector(Line(p1.x,p1.y,p2.x,p2.y))
			if dist < minDist:
					minDist = dist
					if pt.x == p1.x and pt.y == p1.y:
							minPlus = -1
					elif pt.x == p2.x and pt.y == p2.y:
							minPlus = 1
					else:
							minPlus = 0
					minI = i
					minPt = pt
			
			p1 = p2
			i += 1
	
	p1 = polyline[minI]
	p2 = polyline[minI+1]
	
	return minPt, minI
	
	
def offsetPoint(pt, angle, dist):
	angle_rad = angle #/ math.pi * 180
	return Point(pt.x + dist * math.cos(angle), pt.y + dist * math.sin(angle))
	

def linesIntersection(p1,t1, p2,t2):
	""" calc intersection of two (infinite) lines defined by one point and tangent """
	
	# parallel lines?
	if (t1 is None and t2 is None) or t1 == t2:
		return None
	
	if t1 is None or t2 is None:
		# in case one line is with angle 90 resp. 270 degrees (tangent undefined)
		# swap them so that line 2 is with undefined tangent
		if t1 is None: t1,p1, t2,p2 = t2,p2, t1,p1
		
		x = p2.x
	else:
		# usual case
		x = ( (p1.y - p2.y) + t2*p2.x - t1*p1.x ) / (t2 - t1)
	
	y = p1.y + t1 * (x - p1.x)
	return Point(x,y)
	


def offsetLine(polyline, dist):
	
	newline = []
	p1 = polyline[0]
	t_old = None
	for p2 in polyline[1:]:
		lin = Line(p1.x,p1.y,p2.x,p2.y)
		print lin.angle*180/math.pi
		t_new = lin.t
		
		if t_old is None:
			pt_new = offsetPoint(p1, lin.angle+math.pi/2, dist)
		else:
			# calc intersection with last line (with offset)
			pt_new = offsetPoint(p1, lin.angle+math.pi/2, dist)
			pt_new = linesIntersection(pt_old, t_old, pt_new, t_new)
		
		newline.append(pt_new)
			
		pt_old, t_old = pt_new, t_new
		p1 = p2
			
	# last line segment:
	pt_new = offsetPoint(p2, lin.angle+math.pi/2, dist)
	newline.append(pt_new)
	return newline


################################
# testing

from PyQt4.QtCore import *
from PyQt4.QtGui import *

class MyWidget(QWidget):
	def __init__(self, pline, parent=None):
		QWidget.__init__(self, parent)
		self.pl = pline
		self.p = Point(0,0)
		
		self.setMouseTracking(True)
    
	
	def paintEvent(self, ev):
		p = QPainter(self)
		
		self.doLine(p, self.pl)
				
		# zisti info
		(minPt, minI) = nearestPoint(self.p, self.pl)
		
		p.drawText(QPoint(20,20), "min I: %d" % minI)
		
		p.drawEllipse(self.p.x-5, self.p.y-5, 10, 10)
		
		p.setPen(Qt.green)
		p.drawLine(minPt.x, minPt.y, self.p.x, self.p.y)
		
		p.setPen(Qt.red)
		
		self.doLine(p, offsetLine(self.pl, 10))
		self.doLine(p, offsetLine(self.pl, -10))
		
	def doLine(self, p, line):
		# nakresli ciaru
		p1 = line[0]
		for p2 in line[1:]:
			p.drawLine(p1.x, p1.y, p2.x, p2.y)
			p1 = p2

				
	def mouseMoveEvent(self, ev):
		self.p = Point(ev.x(), ev.y())
		self.update()


def do_test():
	
	import sys

	a = QApplication(sys.argv)
	
	l = [ Point(20,250), Point(160, 80), Point(200,150), Point(310,60), Point(280,40), Point(280,20) ]
	w = MyWidget(l)
	w.show()
	
	sys.exit(a.exec_())

if __name__ == '__main__':
	do_test()
