"""
Linear Referencing plugin
(c) Copyright 2008 Martin Dobias

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.       
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *

from mylogger import MyLogger

from dlgcalibrateroute_ui import Ui_DlgCalibrateRoute

from dlgdberror import DlgDbError
import postgis

from qgis.core import *
from qgis.gui import QgsMessageViewer
import calibration

class DlgCalibrateRoute(QDialog, Ui_DlgCalibrateRoute):

	def __init__(self, db, parent=None):
		QDialog.__init__(self, parent)
		self.db = db
		
		self.setupUi(self)
		
		self.connect(self.buttonBox, SIGNAL("accepted()"), self.onOK)
		
		self.connect(self.cboRouteInput, SIGNAL("currentIndexChanged(int)"), self.populateRouteFields)
		self.connect(self.cboCalibration, SIGNAL("currentIndexChanged(int)"), self.populateCalibrationFields)
		
		self.populateLinestringLayers()
		self.populatePointLayers()
		
	
	def populateLinestringLayers(self):
		self.cboRouteInput.clear()
		for lyr in sorted(self.db.list_spatial_tables('LINESTRING')):
			self.cboRouteInput.addItem(lyr)
	
	def populatePointLayers(self):
		self.cboCalibration.clear()
		for lyr in sorted(self.db.list_spatial_tables('POINT')):
			self.cboCalibration.addItem(lyr)
	
	def populateRouteFields(self):
		table = self.cboRouteInput.currentText()
		self.populateLayerFields(table, self.cboRouteId)
		
		self.editRouteOutput.setText(table+"_route")
		
	def populateCalibrationFields(self):
		table = self.cboCalibration.currentText()
		self.populateLayerFields(table, self.cboCalibrationRoute)
		self.populateLayerFields(table, self.cboCalibrationMeasure)

	def populateLayerFields(self, layer, combo):
				
		combo.clear()
		for field in self.db.list_table_columns(layer):
			combo.addItem(field)


	def onOK(self):
		
		line_layer = self.cboRouteInput.currentText()
		line_name = self.cboRouteId.currentText()
		points_layer = self.cboCalibration.currentText()
		points_name = self.cboCalibrationRoute.currentText()
		points_measure = self.cboCalibrationMeasure.currentText()
		route_name = self.editRouteOutput.text()
		
		if self.db.table_exists(route_name):
			QMessageBox.critical(self, "error", "The output table exists already!")
			return
		
		try:
			log = self.calibrateRoute(line_layer, line_name, points_layer, points_name, points_measure, route_name)
			if log is None:
				return
			
			# show the log
			vw = QgsMessageViewer(self)
			vw.setTitle("Calibration log")
			vw.setMessageAsHtml(log.dump())
			vw.exec_()
			
		except postgis.DatabaseError, e:
			DlgDbError.showError(e)
			return

		self.accept()
		
	
	def calibrateRoute(self, line_layer, line_name, points_layer, points_route_name, points_measure, output_route_name):
		
		if output_route_name in self.db.list_routes_2():
			QMessageBox.critical(self, "error", "Route with name '%s' exists already!" % output_route_name)
			return
		
		# find out geometry columns of points, lines
		points_geom = self.db.get_layer_geometry_column(points_layer)
		line_geom =  self.db.get_layer_geometry_column(line_layer)
		
		uri = QgsDataSourceURI()
		uri.setConnection(self.db.host, str(self.db.port), self.db.dbname, self.db.user, self.db.passwd)
		
		# load points layer
		uri.setDataSource('public', points_layer, points_geom)
		lyr_points = QgsVectorLayer(uri.uri(), "X", "postgres")
		if not lyr_points.isValid():
			QMessageBox.critical(self, "error", "calibration points layer is not valid!")
			return
		
		# load lines layer
		uri.setDataSource('public', line_layer, line_geom)
		lyr_lines = QgsVectorLayer(uri.uri(), "Y", "postgres")
		if not lyr_lines.isValid():
			QMessageBox.critical(self, "error", "lines layer is not valid!")
			return
		
		log = MyLogger()
		
		idx_measure = lyr_points.dataProvider().fieldNameIndex(points_measure)
		idx_route_name = lyr_points.dataProvider().fieldNameIndex(points_route_name)
		idx_name = lyr_lines.dataProvider().fieldNameIndex(line_name)
		
		cal_points = []
		roads = {}
		empty_road_names = 0
		
		# start progress dialog
		progdlg = QProgressDialog(self)
		progdlg.setWindowModality(Qt.WindowModal)
		progdlg.setLabelText("(1/5) Loading calibration points...")
		progdlg.setMinimumDuration(0)
		progdlg.setMaximum(lyr_points.dataProvider().featureCount())
		progdlg.setAutoClose(False)
		
		log.write_header("(1/5) Loading calibration points...")
		
		# fetch all calibration points
		f = QgsFeature()
		lyr_points.select([idx_measure, idx_route_name])
		while lyr_points.nextFeature(f):
			pt = f.geometry().asPoint()
			attrs = f.attributeMap()
			# get road name (and convert to lowercase)
			route_name = unicode(attrs[idx_route_name].toString().toLower())
			
			if len(route_name) == 0:
				log.write("ignoring point with empty route name, id %d" % f.id())
				continue
			
			m = attrs[idx_measure].toDouble()[0]
			# add to points dictionary
			cal_points.append( calibration.CalPoint(route_name, pt.x(), pt.y(), m) )
			
			progdlg.setValue(progdlg.value()+1)
		
		
		progdlg.reset()
		progdlg.setLabelText("(2/5) Loading lines...")
		progdlg.setMaximum(lyr_lines.dataProvider().featureCount())
		
		log.write_header("(2/5) Loading lines...")
		
		# fetch all roads
		lyr_lines.select([idx_name])
		while lyr_lines.nextFeature(f):
			line = f.geometry().asPolyline()
			# get road name (and convert to lowercase)
			name = unicode(f.attributeMap()[idx_name].toString().toLower())
			
			progdlg.setValue(progdlg.value()+1)
			
			if len(line) == 0:
				log.write("ignoring invalid line, id %d" % f.id())
				continue
			
			# ignore roads without name
			if len(name) == 0:
				log.write("ignoring line without a name, id %d" % f.id())
				empty_road_names += 1
				continue
			
			# convert from QgsPoint list to XYM list
			points = map( lambda pt: calibration.XYM(pt.x(), pt.y()), line)
			
			if points[0].equalXY(points[-1]):
				log.write("ignoring a road with equal begin and end coordinate, id %d" % f.id())
				continue
			
			# add to roads dictionary
			if not roads.has_key(name):
				roads[name] = calibration.Road(name, points)
			else:
				roads[name].add_part(points)
			
		log.write("ignored %d roads with empty road name" % empty_road_names)
		
		progdlg.reset()
		progdlg.setLabelText("(3/5) Merging lines...")
		progdlg.setMaximum(len(roads))
		
		log.write_header("(3/5) Merging lines...")
		
		for road in roads.values():
			#print road.name #, road.points
			road.merge_parts(log)
			
			progdlg.setValue(progdlg.value()+1)
			
		# do the calibration!
		calibration.calibration(roads, cal_points, progdlg, log)
		
		progdlg.hide()
		
		# save results!
		self.db.create_route(output_route_name)
		
		
		for road in roads.values():
			#print road.name, road.points
			linestring = ""
			for pnt in road.points:
				if len(linestring) > 0: linestring += ", "
				linestring += "%f %f %f" % (pnt.x, pnt.y, pnt.m)
			#print linestring
			self.db.exec_sql("INSERT INTO %s VALUES (DEFAULT, '%s', GeomFromEWKT('LINESTRINGM(%s)'))" % (self.db._quote(output_route_name), road.name, linestring))
		self.db.con.commit()

		return log
