# -*- coding: utf-8 -*-

"""
***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************
"""
__author__ = 'Leandro França'
__date__ = '2021-03-01'
__copyright__ = '(C) 2021, Leandro França'

import math, os, base64, PIL.Image
from math import atan, pi, sqrt, floor

# Imagem para HTML
def img2html(path_file):
    arq = open(path_file, 'rb')
    leitura = arq.read()
    arq.close()
    encoded = base64.b64encode(leitura)
    texto = str(encoded)[2:-1]
    return texto

# Redimensionar Imagem
def ImgResize(path_file, lado, resized):
    caminho, arquivo = os.path.split(path_file)
    img = PIL.Image.open(path_file)
    altura = img.size[1]
    largura = img.size[0]
    if largura < altura:
        new_height = lado
        new_width =int(lado/float(altura)*largura)
    else:
        new_width = lado
        new_height =int(lado/float(largura)*altura)

    img = img.resize((new_width, new_height))
    path_file_reduced = os.path.join(caminho, resized)
    img.save(path_file_reduced)
    del img
    return path_file_reduced

# Imagem para HTML redimensionada
def img2html_resized(path_file, lado=500, resized = 'reduzido.jpg'):
    if os.path.isfile(path_file):
        caminho, arquivo = os.path.split(path_file)
        path_file_reduced = ImgResize(path_file, lado, resized)
        texto = img2html(path_file_reduced)
        os.remove(path_file_reduced)
        return texto
    else:
        return ''


# Repeated variables
dic_color = {'face': 'iVBORw0KGgoAAAANSUhEUgAAABwAAAAcCAYAAAByDd+UAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMMwAADDMBUlqVhwAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAJxSURBVEiJvZY9TBRREIC/t/vujuMOROBilEQTDOLRWFoYhRgrIjY2FCAWmkgwVkKljR30mhALf4KxMSR2GmtbYzSG+JMYNYgokUDk4LjdHYuD4/Z272f30NnsJjvzZr6ZN2/fWwXQf+XhsDgyIZAGTHZXbAVzoph8Nj0yo85fmxlZW7fu7zLEV6JaDevMhnVd5H/gYMOSCe04kq6XZyDsa0/SnIih1I7+8/wK2Zy9oxB6tIAZtkJtwPC5Y5ztS9OUbPDYL9+c5cvCauFdgalFBEIABRgdPM7A6Z4qA3eCC6BFhDAVNiei9PcercISnKLgKg8kVIVdh9owTcOjt22nACmNXagwDHBPk7dnT56/5d7sKyzLAcByHE/s0FNqFC/HLXnx8iOZjVxZH1UPUHycLNvx1Rd82OphrcBoxCAeze988Zj22JMNmubGSOE9k7XIWe7g6tTQdKWkXDLQ1834pd7aBgNXbz3lzYdFly7QlErA1TW/uOKZ4mA9DMDLblosLa8huBdXIOCfTJav35cBSDRGaWtJuOw/fq2yubV3LvxcxXGEEh7qxODtmnsoRc/+k93cGDvjsl8Yf8ynb7+3Q5eygHo+i3LzK5VHhAb6OeVjVQ5WR4V+OfxDoK9TFaACNCE370CJbJsA3dl5IFSFqVSLR9fRkcI2Y+WdFOhEwnvM1CLRWMSji8djJH1+NYpFA7ZIiH9RvxZSZT0obCNiOnNsn8xB7nJJVLhN5J1xpH1zSnDy2QW6yjHLX4dT61MK4M7dB0MLS9mJnGWnEfEedD6Sam2k62CrS/f6/SKZ9ZITXylLm+bc/r3RybHRi4/+And1Z9yayEQKAAAAAElFTkSuQmCC', 'g1': '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', 'github': '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', 'instagram': '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', 'lattes': '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', 'linkedin': 'iVBORw0KGgoAAAANSUhEUgAAABwAAAAcCAYAAAByDd+UAAAACXBIWXMAAA3nAAAN5wGZFKomAAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAAAtBJREFUSIm1lF1IU2EYx//vOcdtbuq06dR0HctVWkbhRUYfhClEVJBgEV0EReBF0VUYfnQVRQVFN9GFUURJIn1AFwlB3pWKRrpKiTBnfmwaNec0d7ZzzttF7uPsDGPt7H/3/N/nPb/ned73PYRSyp5/OdLU5/SckWSZRwrEMsx4pS2r7W5dxTVyrH2wtdPhupwKUKwOlllbiKH5tdMvSSnpLFYsQ8Y4QZR4kIjJZ6fjwp4S8KvS0Ts+h9tvx/E7KGkClCS6los2ijL16D27AwWZegDA4XIrqkst2H9/ADKlmkCZ6KChyhaGhVS73oLdfLYmMBVwdZY+blJsEZoBu0d/qRIEUUbP97nUAJ84XLjXN4HQcS0FJTQ8/4wJr18zICGNXZQSpbkh1wQ+24Ahlw+ziwHNYKAAF+vVVxSgNNcIAKgsNuPN158YmPLCmMaiocoGHfd3KDKleDgwBc9SEEc25WOf3QJrhg5+UYbDNY/HH6YxNS+omAogAfDgaAUy9BH7WZEb9e2D2F5sxq1DZYrNjAzUbclH1RrlLT6xrRCtNXac7BjCi+FZ5Z7YChgSG5O4PgBcqrWrYCFl6Fg8Or4V63LSVwYigfdt0rMrr+tYnNup/GuqgXE6WUnvJ7041fkRp59+gsPlU60f2JiriFWXJhFNzwuoaeuHVxABAK9GZjF6cS9MukjnpRYjOIaBKMsA4nWYgDoGp8MwAJhZDOCd06PISWMZmKNGn9QZfvmxqPLcPvVTMHARTFJnKMcpLl69JOqbSY30f5TUSLUBJqt/FJz0O0x0v/YjTbRDvygrYmE5FiQ5NhUBUe0FZaVHAQSjrjMhjV1jlKAkZJTlmVBoNoQTht0+zCwEQAiwi89B2vKbohTocXpUhVhNOmwuzAzHC34R/ZPeEP0bsV3tbp7wCldUpaZAeUZ9M3PTXn1dx7AtAJwpZDk5hmm6U1594w9TWurjffFO/wAAAABJRU5ErkJggg==', 'RG': 'iVBORw0KGgoAAAANSUhEUgAAAB4AAAAcCAYAAAB2+A+pAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMfwAADH8BdgxfmQAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAOqSURBVEiJvZdLaFxlFMd/57vzajIzTpIxJhNLam0akyyKuqgiFVpdlBTXUiqEaipIBInixpUirjRClS5qsfjAhQvRjQvBtgvFZKE1i+apLU1DXm0ydWYyr/v6XMxNJelcm8SJB77Nved8v/Od87/n4woA4z++BPI60AkY7Iw5wBRahug5cl6YuPgyWp/dIZiP6X5h/MIklZPWaE8NaBABxM9rPAB01IInwBPhKP2JVpoCIS7lb3M+s0BO62rujyhA1QJ8tC7BD+2PERDFcDHLq427+SLVQ7C6uwrUAipa059I8VPhNqcWJzGB73K3OBiJIyJe+ddbTcARgX2hOr7JLmF6fZ2ySkxZJd+Y6mDt+rgLFeGs745CCIiipF3QmneTezhS3wjAwNI0o+X8vcH1IrzX3EHDhi7ElYFGc8ux+bWY4dvVZZbdSoIajcYlgIDA9/k0U2aRs6kuYqr6WLhLWEWt+TqzyLJj8UJDiufva2HBKjFayvKnWWRvqI4zqR5+f+ggT0fiXgzMWiZd4SiGhpFSjslyDoWQc2yf2o1fqKr3RlFM7HuSiCj2/vEzK175A8BAoo2hlk4W7DIHro6Q1i59sWZOt3ZyZmWGa2aJU40PYmnN0dlR8lXE5fsp2biUPZiwNgwEG+HcX/NcNQu0BSMcqksA8FXuJm8uTnM4mmQw2c5EOc/xuStVoWsH2LIVtMuMVWZ/uJ42I+QlCueyS3yeXSJApWVafCfX9sB1otgTjGBrzai5uu6d6S3+BQrbmFoGmv5EKw+Hd3Epn2aklNvqFsAmThwQoTfaRNq1ecAI8VzsfnpjSYYLGU7MjeH6XwT/DRwWg8GmdgygOxLDRjO4MMmX2SWyPsLZjN2z1HnX5tjMZQ5d/42L+RVComgJhMn5TrcagaEilgyad25eo+A6vJZs5/FQ/c6D12y4nOPj5etEVYBPUt3E/i+wi/B++gaXixke3RXndEsnBtvrsy/YQAgi3s3zj3LTWvPKwgRpx6IvkeKtht3bghsM9L29MZP9wTAn4y30xpsJKUXBsUk7JjnXxUIz75jMmgWOxZo5HG2iURTTZp6s62w6hbsuiZgIHzV3UK8MLG8bhRAU4cP0DX4prXqB8FQ4yhvJdg5E4sxbJY7PXWHWsTYNdthQcrnzfeo7bkDV2StaEwZsNLZsWjKuAqY2PtUi3lLeEt+Br0UoiWwFCjChQA9tJaImJnyg6H72UzQvAmNUfjN2yhxgDOEkXc989jdK2myw4HcGQgAAAABJRU5ErkJggg==', 'twitter': 'iVBORw0KGgoAAAANSUhEUgAAABwAAAAcCAYAAAByDd+UAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAANIgAADSIBbrwGKgAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAONSURBVEiJpZbNixxFGIefqumeGfd7R9xoks2usAaChkg8iJ4SwYMIgSCiZ0E9iBD/Ar1H9CA5mZsfIKjJWQ+eVklMCHETgwkYDetmWTMzOzM7PTM93f3m0NvT1dM9H64Fw1RX9VtP/3719tulPvpJinqaz0TkNWBeRAAw/8OuZPSzrjPGkGrgy7eBX3nf0lOcE5G3+iEIyG5QDE72B8HC+xLj8yBvq9yctgQ5PTbMGM+6HgKM5k5bCPOCGRw/dXLBcazcHQ9tzJorWUnYoH3rtzKe1wgHZjTFnKLs+Gw1/QGwMMYyF9MKZvNQbg23MoI9u2BxaqXApK2I2t81j69u7FBrC8f2FUCEX9ZbvTj14SVPokUPzyteXdacv9FluzM8MY7vs3jzSJGs5vpCTisqLZ+zq1XaXoAfBIiANhXsn4S5guLdozYHp9SuugxbtHBqpZAJA8jnFIEId6td3nhmCkvTSyYthl1eEAZM5xXvHM3z8pKNrdN7sTyTY8KwMavZWrE8Z3PhVoN21+/FWqZdf277cCgHgFZwctHi+cdzXN70uLbZ5X4zDJywhrJ67ePVMjuu33NGRMKkiWy91wi4VfE5Usr1giZsxYlFmxOLNs2usNHwkTFgIuB0gwRM6AEBhOMLOf6qBeyf1MwW0pZN2oqnSuPJc3qJkgEMnyhU+sqT+bEWHNXu173M6qN7dBHW/vVouOMYNrr9Ue6k1IkI2kyathfwxc0WHf//Q6+sO4nMjl4v3Z/yd7d9zl1tcn2rS7BH7u1yh3/qbkpdIktN+S8tFTm2YO9Z3cXfa/F6mMWDtEJEuHjbodoO9gT7+V6TOw/acaKYYsiwVBDqnYBPL9X5daOD+x/2c73W5evr1cRaiSxN7mHy81Pv+Hy51uDza3V23NFqNxpdPlnd2i1j/eriD7LVDwRhylaslGxeOFDk8KOj9/K3zRbnr5RxjDJmnoHifSQGPmIpTi4VefFgcWRhjtoDx+PCzRqX15vJxDMAZh8E9cGPFTEVTucVzz2R5+nH8hyatSlaMVwEyo7HnbLL1Q2Htc0WfmCeX/rPM5HSGKzO/FCW7JvDX9FS2Br8QHBcHy+QAWrSVvbDDEvNY2AS6LjBgIdJqxlmZfSvRaSaejUyioGQMZeR9sNgIlQ0It8PByZrYbaVSfUDYCjUd5ZI7T2RGQ+R1wUpkYKl92mYlf2QSBmKb1zbO/MQw3IUMFMV01gAAAAASUVORK5CYII=', 'udemy': 'iVBORw0KGgoAAAANSUhEUgAAABwAAAAbCAYAAABvCO8sAAAACXBIWXMAAAqZAAAKmQHZaLrkAAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAABMpJREFUSIm1Vl1MU2cYfs45rXT2lFKqcU5mE/kbrGj42c2iDoIJF2vWJhOImY6LuUSzCw1cebOLBsIar3AJiuHGhQmGzC0YVJatC4ZM58VC+A+UKT8rpB0teNqdnkN73l1QaktbYCx7kjc5+d73PM/3fPnyfi+D3YGNBhMNQ3TdD4CioURjWzA7CZWUlOhtNtsXlZWVDXq9/m1NxhvqtcEsRpMTBmvykSRJ8szMzFRfX9/t3t7ebwEEdiOctJGDBw/yHR0d34yPj8uCIBAR0fygTP2fCHT3/VXqPbNGzisBCi5FiIjI4/EoT548Eevr65sBaNOZSbXIXrhw4b2rV6/+VFhYyGu1WryaU/D8KxG+qQjWg5RQrH2TxZGTKpRd0YDLYODxeKi/v99z+fLlDyRJmtnqdqsg29jY+OmlS5c68/PzuZCfMPx1CEvPw/jbk/6UGBbQmVgUnctAnm0fQqEQnj59Kp8/f97idrt/jheNF2TOnj17sqWl5ZeCggIOAGa+k/HcISaQ80dYmM6oISwo8AyHEfK9dqw7wuGj73kAQCQSgdPplC0WS4Usy2PYuFjgNov1er2hs7Nz2Gw279tc801F8OdQGACgMbLIOaXC6eta5JxSw3RGjUMVKqy6IpBfAUoYUPMM3jmXsXFULIvDhw9zRqOx7vHjxx0ApPiNc7du3eoLBoMUj9kHEvWcXKMfPw+Q3xWhdJj5QaIHta/o/odrSbn5+Xnl+PHjLfHmcODAgbempqbCW4u9Y2F68VBKKxSPdVGh6fupa+/duycBOBS7KNeuXbseCASUXTHvAZOTk4pGo/kMG80DqoGBAff/JUZEJAgC2Wy2XwGoWACsTqczprruPT09KC4uRmlpKYqKijA8PJxUY7fbceLECZSWlsJsNqeiAc/zKCsrKwHAqgCoNRqNOlWh3+/H5OTkxi3VaBAKhZJq3G43RkZGAADHjh1LKQgAJpNpPwA1CyBLrU6plwQi2rkoDXQ6HQBksQBWg8HgnoniwTDp34KFhQUC4GcBrEuStL4TGRH9J4culysIIMwCUEZHR1/sRLYbwXQOl5aW6NGjR78BUFgASldX1+3l5eUktq0Ee3W4uLhILperOyb47NmzuxMTE+GthZmZmTsKKsrrVySVQ1EUcefOHRHAw01BAPDa7fa2xcXFBEaDwRD7VhQFsiwnEXq93m28AePj48rNmzebAfy1NZflcDi8q6ur8S2JsrOzN2cWslqtCR1kZWWFCgsLY/ny8vKE/NzcnFJRUfEHAH2qzTAAzN3d3aIoirGfiouLY4Q8z1N1dTU1NzeT3W5PyAGghoaG2H/Ly8tUW1vrB/AutpmdWJVKVXnjxo2Az+dTiIja2trIYDAkEKeK3Nxccrs3WvLLly+Vuro6H4DTiDbs7cACyL948aJrYmIiIssytba20tGjR1MKcRxHZrOZhoaGaGVlhQYHByMFBQW/A8hPJZbOKgNgP8/zDU1NTS0WiyVTURSmvb2dmZ6ehiAI4DgORqMRNTU1sFqt5Ha7yeFweAcGBr4E0AVAjG5qV4LxbrUAqvPy8j6uqqqqyc3NzcrJyeFkWabZ2dnI2NiYz+l09guC8ACAE0AQ28ylOwnGC8dP3tnRdR/+5eT9D8n7oeJbT9bKAAAAAElFTkSuQmCC', 'youtube': 'iVBORw0KGgoAAAANSUhEUgAAACIAAAAbCAYAAAAZMl2nAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMZwAADGcBG5JHNgAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAItSURBVEiJxZe/a1NRGIafNLdQoS2xg2aQDG5OnfwLImpLxZql6tJBKAgFO3TNFKqOcSpaqBDBYkkqDkXEQqkIBUmHLJlK9LYWqiYdQvOjYPM65AZqvIZ7k9v0hW8733se7jkv57tgI4EhuCNICvYEVYHarKrlkbQ8Dbs97SDGBN8FRUGtA4Dmqlmeu4LRVgA+wVNBycPN/1eHgscCnx3IE2vBaUOchJlrhhjt0pdorpLgJtSPww/kgJCjS+S9doHLPcAYcN5p10ogQNVbkAAwYgD3gX6nXfnxcVLBIL3Ly0RyOYdZbKl+4B5WVB2f66upKUlSIZ/Xy5kZvQ6F9Lvzu7KDoNIOSEM/9/eVmJ7Wu2CwE5AybpuaQRoyt7e1ODmpjaGhtmA8A2kou7Wl55GI0gMDZwvS0Je1NcXDYWX6+hz59nR+6e11NRzmbiLBi+FhfjlY70H6/lW5XGYlFqN3aYlnpom/2yC1Wo0PCwv8mJ9nIpPhnItez0A2Uimy8TiRzU0uHh+77jeACriC/0vZdJqP0SjX1td5eHTUrk3ZAArAJbede6bJ+2iUK6urPDo4aBegoYIBfAYmsBtSbFSsVFicneVCMskD03TW1FoCPiG4rfoI5yjvbwcHddT523KyioJbCPyCbx4au62vwkq4YERnM6EdCq43H9Scuj+zxuxujK+LMCVBTK0CovogvaPT+68xBTec5slQPU3Lqk9wroanpqpYHm8sT9un5w+YJnqy6UwJgAAAAABJRU5ErkJggg=='}
{'face': 'iVBORw0KGgoAAAANSUhEUgAAABwAAAAcCAYAAAByDd+UAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMMwAADDMBUlqVhwAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAIUSURBVEiJvZa9quJAGIbfMfEYTUSUQAqLYOnehJobsBTlHLbRQtKKlY3lObegxRa7pRdg7y0saiuoKAErMZif2Squ2eTEOCfsC1Pk+3uYb2YyQwCg1+u9UUqHAKoAOCQrB8ASwPt0Ov1JdF3/bprmj4QhoXp5eXnjTdMcUEr/Bw+WZQ15Smn1q4UIIZBlGaIoghBys2+3W1yv19u367rfeAAc6ww5jkOz2USj0YAoigH/aDTCbrfzpfCUUrAC2+02NE2LjPm3NjNQkiTU6/WHsFAgi1RVRSqVCthd171BwmozA/P5fMA2n88xm81g2zYAwHGccCAL9H43elosFrhcLpE5zMCwHMdxImtRSp8DptNpZDIZAIAgCAG/IAi+42Ga5q29nkin06FxgZqmodvtxooFgPF4jPV67bM9NcNnW384HJI7h49kWRZOp9PXgOfzGfv9HgCQy+VQKBR8fsMwYFkWAOB4PMJ13UAN0mq1Yq/hfVytVoOu6z7/YDDAZrP5Wzzk6CR6LOLEJLqGcWrxYX2OW5wJmOQOjQX0ApOERgIrlQoTUJblgK1cLodeWZ4IIeDDngZx5P1T75XNZiFJUmQeTyl1wPAW/awrD7rlpDiOW3qL/cz4DBY1CCG/U4qifDw7O1YpivJBAGAymbwahjG0bbtKKeXjJJdKJaiq6rOtVqvAjU8IsTmOWxaLxfd+v//rD1H2cZ8dKhk8AAAAAElFTkSuQmCC', 'github': 'iVBORw0KGgoAAAANSUhEUgAAAB0AAAAdCAYAAABWk2cPAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMOwAADDsBdtCd4gAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAW2SURBVEiJrVZdaBNZGD33TpL+RosxxWZSu7ZpDYQ1CPVBUSws2Cq6sPWvFW1BQcyKaOnDwvqgQvFh6ypoFn3xUd1OygrarK6UKoq7FAqFVWulKba1rtVgW5PWZjqZ++2DTjY/av3ZA8Mw3537ne+c+829w/ARIKKFQ0NDG/v7+zeEQqGysbGxopmZGVteXt5Lh8PxrLy8fNDtdgdlWe5gjL2cKx/70GAkEqkIBoMtiqLU9vX1SVNTUxBCQJIko5g3SRiD1WqF1+vV6+rqfqupqTmcnZ098EmkRGS5cuXKTydOnPh+cHDQzDlPHgPnPEGYSMQYhBBgjMHtdmuHDx/+paqq6gfG2OycpJFIxHbkyJH2QCBQ9SEX5gLnHHv27Pmzubm5Nj8//3nymCn5YWxsrLCxsfGv7u7uUkmSMtR8CoQQOHfu3KqRkZG70Wh0pdVqDWeQEpF569atSnd3d6lhn67rifX7GBhFCiHAOYfJZMK1a9fKOOeXiegbxpgKAImMmqad7Ojo2A68WZ/169fj4MGDyM3NxdDQEDRNA2MMRAQhRILEWEcAsFqt2LZtG/bt24eJiQmMjo6CMYZHjx4tNplMedevX/8DeLumd+7cqdixY8d9IjIzxqBpGi5cuIC1a9cCAEZGRuD3+6GqKjweDwoLC2E2m6GqKl68eIF79+7Bbrdj//79sNvtYIxBURQ0NzeDMQbGGLKysrT29nbP8uXLB0wA0Nra2kJEZqN6zjkWLFgAo2uXLFmC1tbWhCLjnm6t0cEAYLfbEzEAiMVi5jNnzrQA2M47Ozttvb293xmTjZfGx8cTVQJvujH5OR3J73HOMT4+nhJnjOHmzZube3p6FvJgMPitECK5oZCVlYX8/Pz3d8wcEEIgNzcXFoslpShVVaXOzs4NksVi+XFyctKTTFpfX4+Ghob3qpoLRASXy4W+vj6EQqEUtZqmxfmTJ09cyRMkSUJ9ff1nkRkwLN65c2fiMzLug4ODX3MhRFHyJjBv3jyUlJR80cYAvFFVWlqKnJyclHgsFnOYdF1fkGyjyWSCJEmfbS3wnyqLxZJoQANCiDwOIOUoisVieP369f+iNBKJQNO09FzTnHP+LDkyNTWF4eHhLyYUQiAUCkHTtJQxk8n0D8/Ozh5MnxQMBqHr+hcTX716NcVaxhjmz5//N3e5XL+nW6koCoaHhz/bYiJCb28vbty4kRIXQmDp0qUdfPXq1VeJKG5s5gAwPT2NvXv34vHjxymJPhYPHz6Ez+eDqqrpxejV1dW/MwAoKytTVFXd6vV6sWbNGly+fBmjo6Ow2WzYtWsXamtrUVxcnOjE9D8JXdcTa9je3o6LFy9ienoayUIAoKCgQLl///52AIDP5yuVZVldtmwZBYNBGhgYoI0bN5IsyyTLMpWUlNCqVauoq6uL4vE4paOtrY0qKytp8eLF5HQ6SZZlcjqdKZfD4Zg9evSoC0j6XfF4PKdevXp1yGKx4NKlS4jH4/D5fAiHw+Cco7CwELdu3YLVak1pDiJCOBxGVVUVotFoip3JShctWnSyp6enGUg6xFtaWrpu3769Rtf1r54+fYrdu3dj5cqVKCoqgsfjQWNjI9xud8ZJwxiD2WxGMBhEOBzGuyBJ0t1jx441BAKBzE/i+PHjdofDEXI6nXTq1CmKRqM0MzNDExMTFIlESNd10nU9w97Z2VmqqanJsPSt1aHTp0/bUxx4B7Ht7Nmz7URUVV5ejhUrViAnJwcFBQU4cOBAxrYGAJqmYdOmTXjw4EGKA5zzu01NTZsPHTr0PJ0nA4qiWLxe70mHwzEryzI5HA6qq6sjTdMyVBpKq6urqbi42FA3W1lZ+bOiKJY5ydLR1NTkqqioaCsqKopv2bLlg6Tr1q0jWZY1j8fzq9GlXwS/328LBAIN8Xi8Tdf1fiKafMs3KYTo13W97fz58w1+v9/2Mfn+BQw/D7WnyIOMAAAAAElFTkSuQmCC', 'instagram': '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', 'lattes': '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', 'linkedin': 'iVBORw0KGgoAAAANSUhEUgAAABoAAAAaCAYAAACpSkzOAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAM6QAADOkBmiiHWwAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAJRSURBVEiJ7ZU/aCJREMa/t7soa5LOoGihktZOQSxSWykBIYVaWZ6NVa6wSZ/WI5BGziJypA1CQBLsgkKKgBjFQkEtkouJ+AdxdeeKnHvurXdcXK84uK/amfl4v9m3s++xbrdrury8PGm324dEZMYGxRj7arPZvoRCoSPh/Pz8pFqtftgkYEnmTqeT6PV6xDscjs9EZPpLIADAy8uLS5jP52bGGACA4ziEw2H4/X5IkoSrqytcX1/rBhHRrrCcCAaDCAQCShyJRPD6+oq7uzvdMG458Hg8GoPX69UN0YDG47HGsCqnG5TP50FESjyZTFAoFDYCYvF4nBbDAAB7e3vw+XyQJAnFYhGPj4+6IUQE4eekJEm4vb0F8DaFC1mtVphMb3+BLMtotVogItjtdjgcDhiNRvT7fTw8PKzcbhXI5XIhlUqpOkkkEpjP5zg+PoYg/LCfnp5if38fbrdbteB0OsXFxQVubm5+DTIYDKoiYww8z4OIwPO8qhaLxbCzs6Pp3GAwIBqN4vn5Gff390qe0zj/UKsgywqFQqpY843eo1KphHK5DLPZjIODAxiNRqXmdDqxvb2N4XCoD1Sv13F2dqbEs9kM0WhU5bFYLApo7a1bTOZClUpF49na2lKe1wb1ej1VvGqklwdobdB79R/0j4FkWdYYiEh1dawrAcATgF0AaDabyOVyyuE5HA4xmUwAAJlMRjl2ZFlGo9FQLTQajZDNZiGKouKp1WqL8hNLJpPpwWCQ0N3ybySKYpqLxWJHjLFP399s03riOC4dCAQ+fgMeouMzfwx22gAAAABJRU5ErkJggg==', 'RG': 'iVBORw0KGgoAAAANSUhEUgAAAB4AAAAcCAYAAAB2+A+pAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMfwAADH8BdgxfmQAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAM2SURBVEiJtZdLSytJFMd/1dXB+OiQRJEQBEnbPgISJGIQcePKjyBZzoP5AH6OuR9gFjPMznwC1650EbKSy00WPhJ8EMlDtG1M0p2axZCA1871cZM/NDRFnfqdqjrnVJUAODg4+E0IsQ8sA5LRyANKQogve3t7/4hcLvcH8NeIYL5SSv2uAftDHhSl1A/7CCH2dWBxWNBoNIppmgSDQarVKufn53ie59d1RQO0YUBjsRg7OztomkatVmNpaYnNzU2EEH7dNX0YUKUUpmlyd3dHPp9HKcXV1RXT09MIIXyXfihgKSWGYVCpVPoQ27axbXugjS/4HcHh29btdlFKsbq6SiwWA6BQKHB/f/82WEpJKpUiEAi8aA8EAiilaLVaNBoNrq+vabfbr+BCCG5vb3l8fCSTyaDr/ov6KrA8z6NcLtNqtUgkEszPz+M4Ds1mE9u2MQyDjY0Ndnd3mZmZ6ds4jkMoFAKg0Wjw8PCAEIJOp+ML9nWnZ5hIJJBSUiqV+gMIIbAsi3Q6zdbWFoeHh7iuS7lcZn19HcdxsG0by7Ko1+sD93lgcCml6Ha7SClftZ+dnbG4uIhhGMzOznJzc0OlUkHTNBYWFtB1nXq9zunp6aA8/lxUe57H09MThmEwPj7ed+ji4oLLy0uEEHieNyiHPw+WUjI5OYlS6lXE9krmj6DwyaplmiaGYVCtVqnX658Z4u0ZCyGIx+O0223GxsaYm5sjHo9Tq9U4OTn5FPRdYCkly8vLCCEIh8MopSgUCpTLZVzXHR3YdV2Ojo7wPI/t7W1isRjBYPCnoPDOPVZK4XlePz2SySThcHj04J4ajQbFYhEpJZlM5lWOjwwMUCwWaTabRCIR0un08MG9gt/778l1XfL5PJ1OB9M0WVlZGR54amoKy7IIBoPouo5lWYRCof7SNptN8vk83W6XVCrF2toaExMTHwKLXC734vCVUpJOp9F1/cW5rGkapVLpRcGIRqMkk0kikQiO43B8fMzz8/O7wd73Mx90EfArg0opNE178/LwnboaUPID+H2DnPkgFOCbppT68lGrn5VS6k8tm83+DfwKfOX/Z8ao5AFflVK/ZLPZf/8DudZq3wvXLmgAAAAASUVORK5CYII=', 'twitter': 'iVBORw0KGgoAAAANSUhEUgAAABoAAAAaCAYAAACpSkzOAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMMQAADDEBLaRWDgAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAK5SURBVEiJpZU7T+NAFEZP4kAePBwZ8TA0EUIRKKIgQhSImnZ/wDYr7Q/barddbRu2QVAQhChAgBQqQClwFEIcx8IYfLdKNk5sJ4GRRh6PPN+Z+90749jZ2VnGsqyfwD6QEREARISocdRcT7dF5AD4lmi327+AL0GCn4QgIhkR+SIiPxIish8k3i/yQVCn7ydEJD0sgjDBMUCZxCg2hQmqqsri4iKKotBoNKhWq741vesSvULpdJrJyUkajcZQy7a2tlheXqa3FQoFzs/PmZubQ9M0Dg8Pu2vivQJLS0sUi0Wy2WykFfl8fgACkEql2N3dJZfLcXFxgaqq/0G9O04mk8RiMba3t8nlcqHRrK6uDkB628TEBDs7O5imORgRgGVZ3Y/z+Tx7e3ssLCz4QMlkkng8HglqNpscHBzw9vbmz1EHVK1WWVtb6wpNTU1RLBZ5f3/HMAzq9TqO40RCAGq1Gq7r+jboK4aZmRmurq7Y3Nz0LVQUBV3X0XV9KATAcZwBy7vlLSJomjbU/1Ha8/PzAMhXdQ8PD5+GABiGEQ1qt9tcXl5+CvL09MTLy8ugdf0T9/f32LbNxsYGs7OzY4Our68DD/oASEQoFApMT0+PDTFNk7u7u0BQPAhULpdxXXds0PHxMZ7nDUBCQa1Wi1KpRKVSwTTNkSAnJyfU6/XQWz4QJCI4jsPj4yOKogyFnJ6ecnt7G3nTd3OUSCSYn59HURRUVUXXdTKZTCTAtm2Ojo585dwP6rx3Qa+vrzSbTdbX11lZWSEWi4UCLMvi5uaGSqWC53mh/67esa/qTNOkXC4DoGka2WyWVCoFgOu6tFotDMPAtu3QXISNA8tbRKjVaoEnPCwPYZZ1nnHP8+ww2Ed7P0RE7LjneX/7ff6ocES0pbiIfBWRP57n2aMAwywKGdsi8ltRlO//AFPkniYXwGRMAAAAAElFTkSuQmCC', 'udemy': 'iVBORw0KGgoAAAANSUhEUgAAABwAAAAbCAYAAABvCO8sAAAACXBIWXMAAAqZAAAKmQHZaLrkAAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAABOBJREFUSIm1ln9I1Gccx1/3PU9vevo9f45WK8gzyR/EYesfafMwWMVKYf2AIg0WYQyK6i/7YxAmhf41IwJZYGubVn/V5iKZDvsxXNBALbW0WebpuZO7ySne96zvZ39Mr7vz/LG2veEDD8/n87zfz/vh+WVgeVBmwzAbKbP9HkBmQ5+NRWFYSig/P18tLS393OFwlKuq+n5cXJyps7PTkJ6eTkZGhmiaFujv7++7ceNG/fXr178FJpcjPG8i6enplvr6+q97enoCk5OTIiLy8OFDqayslLKyMjl8+LDU1NTI+Pi4iIi43W797t2703v37j0DJCxkJlqncuDAgQ+OHz/+U3Z2tiU+Ph6Xy8WlS5cYHBxkeno6rDgtLQ273c7+/fsxmUy43W5pbm7+o6Ki4iNN0/oj3UYKKidOnCg7cuTIVzabzejz+WhsbKS7uxuPx7PgciiKwooVK9i2bRsOhwNN0+jo6Ajs27fvk5GRkdZQUSVUfNeuXYUVFRVf2Ww2I8CDBw9ob28PE8vIyGDnzp1s2rQJVVUB0HUdp9PJzZs3AYiLi2Pz5s2xly9f/iE2NjY31FjMXENVVeupU6dasrKyjMEZGN4sgKqq5ObmcvDgQRISEgB4/vw5DQ0NDA0NoWkauv5m9RRFobCwMLa2tvbnY8eOZQITAHPkxrq6uqbi4uJck8kUHDQ8PExXVxc2m42jR4+yZcsWYmNjg3mr1UpRURGqquJyuZiZmWH79u3BvMlkYuXKle+0trbK2NhYO38fH0hLS3vvyZMnryQCz549k3v37kV2R4WmadLa2ho1d+3aNQ14N+i+srKydmpqSl8W81ugr69PN5vNn83tmZiWlpaR/0tMRGRyclJKS0t/AWIUQElMTEydv9nh6tWr5OTkYLfbWb9+PZ2dnfNqqqqq2LBhA3a7nby8vGg0JCQkUFBQkA8oMYDJbDabohV6vV56e3sBMJvN+P3+eTVOp5Ouri4A1q5dG1UQYM2aNfGASQGsoTtzMYjIsuqiITExEcCqAH9OTU29NVEoQs9tJF6+fCmAVwFmNE2bWYpMRP6Vw/7+/inglQLo3d3dg/+F4EIOXS6X3Lp161dAVwD9ypUr9S6Xax5bJMHbOhweHpaBgYHGoGBHR8d3PT09ryILk5KSlhQMvT+jOfT7/TQ0NEwDP84JArhPnz79pdPpDGNMTk4OIw4EAvMI3W73It7g8ePH+sWLF88A45E5a01NjXtiYiL0SpKUlJS5P4uUlJSE3SAej0eys7OD+YKCgrD80NCQvnHjxt8BNdpkDEBeU1PTtN/vDw7KyckJElosFikuLpbq6mqpqqoKywFSXl4eHDc2Nia7d+/2AmHvYSSUmJiYovPnz096vV5dRKSurk6Sk5PDiKNFZmamjI6OiojIixcv9D179niADwl/5KOLAlmHDh0a6O3tfT0zMyPnzp2T1atXRxUyGo2Sl5cn9+/fF4/HI3fu3Hm9bt2634CsaGILWTUA8RaLpfzkyZPVO3bsSNJ13XDhwgXD06dP8fl8GI1GUlNT2bp1KyUlJTI6Oipnz5513759+wvgG2CauQd3GYKhbhOAYpvN9qnD4fjYZrNZV61aZQwEAjIwMPD60aNHnra2tmafz/c90AZMsci/dCnBUOHQn/fcefHyD3/efwGDjFmrMyUVtQAAAABJRU5ErkJggg==', 'youtube': 'iVBORw0KGgoAAAANSUhEUgAAACEAAAAaCAYAAAA5WTUBAAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAMNwAADDcBracSlQAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAKkSURBVEiJxZcxSBthFIC/918uGhxMsyptcGjoIgGHW9wSihQKGZrNthHsIC7S1TqWgkRwcZV2lKo0FouixaEuEQ4khIIdrClkEoIZgiae93fQgFgll5jqt93x3v++e3f/3TvhGgYHBx/UarUUEAMeA48A/3WxDagBf4A9EdmsVqsfd3d3j64GyeWDZDJpFAqFKWAC6G6haCPKIjKTzWbfA+4/Ev39/V2dnZ3LwNP/UPwqaycnJy9yuVwFQF2cVIFA4PMdCQAMBQKBL8lk0gAwACzLmtJav7kjgTp95XK5ViwWf0g0Gg36/f7fQLBRltYaEWkU5hkRKTmO06dM03ztRaAuEQ6H0Vq3RUJrHTIM46XR29v7jvNt2BClFEtLS0QiEfb39ymVSrfujIg4Cog0mxiPx1lYWGBycpKenh5c122cdANa64gCHra6QCKRYHl5mfHxcYLBII7jtLJMWAEdrUrUSaVSrK6uMjo6SigU4uzsrJn0DtU4xhs+n4+xsTFWVlYYHh6mq6vL821qm0Qd0zSZmJggk8kQi8XuR6LO1tYW+XzeUzd87S6ey+WYnZ0ln88jIijV+DrbJnF4eMj09DTb29u4rtvU+8MHVLnFDjk+PiadTrO+vs7p6WkrS1R9QAGPb8zLaK2Zm5tjcXGRSqXiqe03sO8D9pqVyGQyzM/PUywWMQzjNgIAv3xa600Ree4l2nEcRkZGODg4QCmFYRi3KQ6AiGzIwMBAt2EYB9znp9y27bLWesZjUtsELkjbtl2ur6osy/qqtX7W7io3ISIb2Wx2CHDrT5TrOE4SWLsjgW+O4yS4mLiv9ldZljWptX6Lx2mrSY6A9M7OzgeuG/kvE41Gg6ZpvhKROPCE85mj5Z8fEfnpuu5313U/2bZdvhr0F9Fo9phaoDu9AAAAAElFTkSuQmCC'}

class Imgs:
    social_BW = '''<a target="_blank" rel="noopener noreferrer" href="https://www.geoone.com.br"><img title="GeoOne" src="data:image/png;base64,'''+dic_color['g1']+'''"></a> <a target="_blank" rel="noopener noreferrer" href="https://www.youtube.com/leandrofranca"><img title="Youtube" src="data:image/png;base64,'''+dic_color['youtube']+'''"></a> <a target="_blank" rel="noopener noreferrer" href="https://www.facebook.com/geoleandrofranca/"><img title="Facebook" src="data:image/png;base64,'''+dic_color['face']+'''"></a> <a target="_blank" rel="noopener noreferrer" href="https://www.linkedin.com/in/leandro-fran%C3%A7a-93093714b/"><img title="Linkedin" src="data:image/png;base64,'''+dic_color['linkedin']+'''"></a> <a target="_blank" rel="noopener noreferrer" href="https://www.researchgate.net/profile/Leandro_Franca2"><img title="ResearchGate" src="data:image/png;base64,'''+dic_color['RG']+'''"></a> <a target="_blank" rel="noopener noreferrer" href="https://www.instagram.com/geoleandrofranca/"><img title="Instagram" src="data:image/png;base64,''' + dic_color['instagram'] + '''"></a> <a target="_blank" rel="noopener noreferrer" href="http://lattes.cnpq.br/8559852745183879"><img title="Lattes" src="data:image/png;base64,''' + dic_color['lattes'] + '''"></a>'''

    social_table_color = '''<table style="text-align: right;" border="0"
     cellpadding="2" cellspacing="2">
      <tbody>
        <tr>
          <td><a target="_blank" rel="noopener noreferrer" href="https://www.geoone.com.br">
          <img title="GeOne" style="border: 0px solid ; width: 28px; height: 28px;" alt="udemy"
           src="data:image/png;base64,'''+dic_color['g1']+'''">
           </a>
          </td>
          <td><a target="_blank" rel="noopener noreferrer" href="https://www.youtube.com/leandrofranca">
          <img title="Youtube" style="border: 0px solid ; width: 28px; height: 28px;" alt="youtube"
           src="data:image/png;base64,'''+dic_color['youtube']+'''">
           </a>
          </td>
          <td><a target="_blank" rel="noopener noreferrer" href="https://www.facebook.com/geoleandrofranca/">
          <img title="Facebook" style="border: 0px solid ; width: 28px; height: 28px;" alt="facebook"
           src="data:image/png;base64,'''+dic_color['face']+'''">
           </a>
          </td>
          <td><a target="_blank" rel="noopener noreferrer" href="https://www.linkedin.com/in/leandro-fran%C3%A7a-93093714b/">
          <img title="Linkedin" style="border: 0px solid ; width: 28px; height: 28px;" alt="linkedin"
           src="data:image/png;base64,'''+dic_color['linkedin']+'''">
           </a>
          </td>
          <td><a target="_blank" rel="noopener noreferrer" href="https://www.researchgate.net/profile/Leandro_Franca2">
          <img title="ResearchGate" style="border: 0px solid ; width: 28px; height: 28px;" alt="RG"
           src="data:image/png;base64,'''+dic_color['RG']+'''">
           </a>
          </td>
          <td><a target="_blank" rel="noopener noreferrer" href="https://www.instagram.com/geoleandrofranca/">
          <img title="Linkedin" style="border: 0px solid ; width: 28px; height: 28px;" alt="linkedin"
           src="data:image/png;base64,'''+dic_color['instagram']+'''">
           </a>
          </td>
          <td><a target="_blank" rel="noopener noreferrer" href="http://lattes.cnpq.br/8559852745183879">
          <img title="Lattes" style="border: 0px solid ; width: 28px; height: 28px;" alt="lattes"
           src="data:image/png;base64,'''+dic_color['lattes']+'''">
           </a>
          </td>
        </tr>
      </tbody>
    </table>'''

lftools_logo = """iVBORw0KGgoAAAANSUhEUgAAAG0AAABdCAYAAACvguj7AAAABHNCSVQICAgIfAhkiAAAAAlwSFlzAAAJMgAACTIB1O6aLwAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAACAASURBVHic1X15kFzVdffvLu+97p7pmZFGLYE0GoJGC0JIAiEggBAyMXjFUQwKiXE220DKJNgJuJyQxCE2qXJVYpKK4zJOPsqGgtggsE3YZL4AQuwIhUWWBrRrFkkzPat6edtdvj/63afXPd09I8CG71R1Tb+et9x3f+937rnnnnMewf8/Qt7jPnoGx89knw9cZtIRH4TUa9eU3zZt2vSu27958+Z6AM30tw9UPkyg1bYl3jbg5PP5+LdCoVC1v+d5095LKpWqAiCbzcbbuVxOA1PArAXsQwHgBw1aXaBqQSoUCsSA4vs+AYAlS5YAAFzXrfq9mTiOowEgnU5rANi7d2/V76lUShsg64D4oQHwgwCtIVC1IPm+T5Lg+L5PgiAgYRiS2bNnEwAIw5AIIeJzJr8b4Zzr5HfLsjQAjI2NacuytG3b2nEcnQTTcRxdC+KHBcBfJ2ik9nuSUbVATUxM0CRAhw8fbhkcHLwYAD///PP/r5SSmI9SigBAa2trfIHe3t7uTCbjzZs3bzSVSslisQgAoJRqxlj8KRaLmjGmOec6nU6rJJAdHR3qJAH8tYDHfw3XqAtWPp8nBw4ciIHq6uoitm2TUqlEXdclSil6+PDhUycnJy8XQnxUa30xgLRt29/yfd+xbZsCIIQQopQiSilSLpfhOA4BgPHx8TtGRkbO7evrAyHkOIA8pfQ4gCFK6TAhZIgQkuecj3POh1Kp1PD8+fOH2traRLFY1Eop5bquymQyuqWlRTmOowcGBrTjOLpQKOgNGzboCDxzbzpxr79S8H6VoDUFa3JykiYZBYAGQcD27NlzthDicinlb2mtz6o5D9Lp9D6lVEsELNFaE6UUtSyLACfGOADKHKu1bgfQLqUEAJi/AOD7PgDoyclJDA0NBYSQMQBDAEYopXlCyFFCyAgh5CildJhzPjx79uyjk5OT5UKhoD4I8H4VoM0ILNu26eDgIC0UCq2Dg4OXSimvUEpdprWeaw62LIusWbMGl1xyCR577DH09vaquXPnDpTL5dODIFigtc4ASAOwtdZZrTUhhBQppYIxNkspxTZu3Kh939f5fB4jIyN6bGwMk5OTjdrsaK1PBXAqAJ0ENymRqi0DOAogD2AYwDFKaV4pdSTazgM4Fv3130uHNmrsr+J8xIBVKBRisIaHh9mhQ4dOO378+EeklOuVUpcByJiDOjo6yNq1a7Fu3TqsX7+eZjIZqpSin/zkJ0mpVFK5XA4ArOkaMj4+Ds/z8NBDDyGXy4EQoqOPklKqiYkJPTo6qkZHRzE2NhYDOjo6isRvcF0XaM6YmbBpFBUgh1EB8i4AL83guLryfjGtLlhmzGpra2ODg4OrDx069HEp5RVa65XJY04//XSyfv16nHfeeXTNmjVUa82UUgwAwjDE8PAwJicnkU6n2ck2zLBFa0201gQABYCOjg50dHRgyZIlilIqKaWKEKISh2oA8DwP+XyejI2NVYGbz+cxPj6O0dFRMjIygvHxcR2GYaNmdEaf5dH2f5/sfSTl/QCtLmD5fJ52d3eTX/ziF/+qlPoEgDlmJ8dxsHbtWlxyySXkoosuorlcjkkpudaa1FNJZj5lWfUJtm7dOmzcuBGEEDz88MPYtm0bCKk0q0lHAgCUUlQpRQGAEKIppZIxJg2AqVQKCxcu1AsXLqy6X0JILcPIxMQExsfH9ejoKCIgkc/nYRi8e/duHD9+HACONG3UNPJeQSPJv5s2bSIHDhygk5OTtKuriwwPD1Ot9e8CcHK5HC6++GJccsklOP/884llWVxKyZVSVAjR9CJ79uwBYwxf/OIX8dprr2Hfvn2VixKCv/7rv8Yf/uEfxvt+6lOfwn333YdbbrkFQIVplFJwziGlRKNxCqiwUUrJpZScUmoYKM0DYHYz+0ZtiMGL2EsWLVpUV2XedNNNePHFF4GKinzX8m5Ba6gOJycnaS6XowAYAKa15ueeey6+//3vAwCRUlpSSh6G4bTjKSEEnHP09/cjlUqhs7MzBgwA/uIv/qIKMCPXXnst7r33Xrz44osQQkApBa01HMeB1hpCCAghoHXj4cgwkBDCI/aJCKAqy7AeePV+A4Dh4WGgYtUO1Z7nZISe7AFoog7nzp1L29vbGaWUCyEsSmkaAEun00RKafu+nxFCWOamAIBSiponGUBFFaZSKViWhT179mDBggW466674v9fdNFFuO666xo2cuXKlWCMwbDYgEQIqTr3lJsjBJSe6BbDviAIUjVtJ8m+SIyZVccmt/P5PFAxRpJ0P2lj8GRBmwLYgQMHaD6fp7lcjg4ODnIAVhiGdhiGdqFQaCOEWJzztBCiqoc453AcBwCqnnhKadyhhBCMj49jZGQEWmt4ngegMibefvvtdcE2YlkWbNuGUiq+RnJ8M+Cl0+lakOJrcF6tiCIN4ZjxN9EnVeAljzFgBkFgxrOjkSdoytRopvJumAYkAEuoQ04ptZRStpTSHh8fXzk5OflTADbnPG4U5xzpdBqccwRBEHcqcIJdyU585513oLU2qgUA8LnPfQ7z589v2kADevKBqKcSCSFTWKeUQhAEVW01orUmQghLCGFrrWkSPPMQ1WNdPp8n0bWHDhw4QOsAN2M5GdDixtUDbHR01Mpms5Zt204+n7+uWCw+orVeAVSAYowhnU7Dtm1oreH7ftyBhBA4jlNXXe3ZswdhGIKxirWfSqVwww03TNtYcy5CSBUjG1mTlmXBcRwkOj5uo23bSKfTcRuAypgXBEHMugRQdVkXqUZCKT02OTlJ6wA3YwBnClpDwMz4lc1mrWKxmBsYGLhHCPEPAFJApdMymUzcIY0AS3ZIUvbu3QspZQzClVdeiY6OjmkbbNghhKg6dzPrkTFWFzilVNzO5P+j83EhhA2A1KhMcw4CxEYIKKXDuVyOvhfgZgJaU8CUUpZSyhoZGVk2PDz831rrjyQaHHeE2fY8bwpgSXVYK7t27QJjLN7n93//92dyXzHIQogqBhvrsZFQShsyDqgAW/uAKaVoGIaxukwwzvg+ycjIiDk+Tynl7wW46UCrAiyfz5MkYIQQrpSyjh07tq5UKj2ste6pPYExr5VSdRnWDLBCoYAjR47ErFm6dCnOPPPM6e4JQDVotRbhd77zHdx99914/fXXY+MmKY2AU0rB87y6oJuxznhyalln1gotyxqLphExcPl8npwMcDOepxmzvquriwBgQgiulLKGh4cv8X3/HlQct0ilUrj99tuxfft2/Nd//VfcCbWdY1lWU8CAihEipUQ6nQYAfOxjH5tpc6tAAyoMMUbPwYMH8dhjj8W/L126FCtXrsSqVauwYsUKdHZ2glIK27bNKkAMWFLMWB0EAbTWMXCcc1BKZeTA1gBIPp/XAIht2+NKKSt6MHRXV5fO5/MUlfmbmQc2nbs1Ay1GfcOGDdS4paSULAgCA9hv+r7/Q0SAdXR04M4778TZZ5+NT3/601i5ciWuu+66KaYzIaThGJaU3t5eaK1jAK644oppjzFizm9A45zHRsjcuXOxc+dOMMYgpURvby96e3vxwAMPAAA6OzuxatUqnHXWWVi6dCl6enqqHjADqPnNcZwqLVIPuIhpuqOjYzwMQye6J23bts7lcsjn89iwYQO2bt1qzOmG4E3HNDOOEd/3yfDwMAPAM5mMNTQ0tMJ13XsAtAJAW1sb7rnnHixdujQ+eOnSpUilUlNA01rDdV0wxuKntZ68/fbbACqsmT9/fhwXMhOxbRvACdCMilRKYc6cOZBS4vrrrwdjDNu3b8fOnTvjfYeHh7Fly5aYjdlsFmeddRZWrlyJ1atX4+yzz54CYiqVqhqvhRCWZVmaEKK11hgZGdGEkLCtrU1IKR1KqZZS6tHRUQ1A+76vC4UC2bRpE4nW5hqyrRFoVetgZhxTSnEhhDUxMdHped7/AdAOAOl0Gj/4wQ+qAAMqllrSXDedZ27M+AKNuyppcAAVppnfzjvvvKYg1UqtegQQgzZ37lwopdDT04M/+ZM/gVIK27dvx/33348tW7ZgdHS0aj5XLBbx0ksv4aWXKqspqVQKy5cvj1Xq6tWrMWvWrEaMCwkhGB4e1oSQMUJImhAihRCaUqoBaM65zuVyOp/PI5/P602bNpmAorpsqwdarBZrxzHf9zkAa2xs7N+01otMR9xxxx0455xzppzo6NGjVdYb5xy2bUMIgTAM45sz3oowDGPV6Xke+vv7Y8vzggsumCle8bWAahPfuLXmzJkDpRSeeuop7Nu3D1u3bsXo6Gi836xZswAA7e3tWLNmDWzbxqOPPgpjAXqeh7fffhvvvPMOHnzwQQAnVOqZZ56JpUuXmgeYSCl5uVwOPc+jnPNJVNYOlRBCRaApSqlyHEfVjG8GsCnANVSPRi12d3eTwcFBRinlmUyG5/P567XWl5v9rr/+enzkIx+pe46f/vSn0FrHHZgEL+l1l1JWASiEwK5du6qO3bVrF2zbRldXFxYsWIBoMbShmOOM10VrHRsic+dWFsf/53/+Z8qcz7ZtXHrppbjqqquwbt26+Dw333wzPvaxj6FcLiMMQ7S1taFYLMYPxejoKJ555hk888wzCMMQmUwGZ511FlatWkU7Ozs5AME5LzLGWgFISqlUSknHcSQAOTExoZRSuru7Ww8PDzdVk7WgVbHM8zwyMTFB0+k0E0Lw0dHRbiHELWbn888/H3/+539et9Py+Ty2bdtWBVKtrzA556kF8J133gFwAuj77rsP9913X3ysmei2t7fH34FKzIfv+xgYGIDruigUClOsPgO4AXHevHm49NJLsX79elx00UXIZDKole7ublxzzTX4yU9+Eru2tm3bhv7+fmzfvh3bt2/Hjh07UCqVYFkWisUiXnjhBbzwwgtAZcVDWpZVVkq1ouIwVkIIKaWUjuNIxphyHEdNTEwoz/Niw6Ve39ZVj0mWua7LhBBMKcVLpdK3EBkexmnbyIi4//77Y2uNMdZwAdNIEkCtNfbv3w+g8cKnASdywk4RYxTUm1O1tbXBtm0wxvDTn/4Uy5cvb+p8NvKnf/qnePDBB+O556OPPoqbbroJl156KYDKg7dr1y5s374dW7duxeOPP568Prcsy6eUthJChJQypJSGlFKhlJKUUmFZluScq+7ublXDNiABYF31mGRZEARMKcUnJiYuUUp93Oxzww03oLu7u+7NSSmxefPmKo/5TDrFCCEEb7/9dgzkxo0b8Qd/8AcYHBzEkSNHMDAwgHw+j0KhAN/34XkeCoUCgIqll0qlMDExgZdffhmEENi2HfsgTYfncjkMDQ3NeLIOAKeccgouvvhiPPvsswCAzZs348Ybb4wfNsYYVq1ahVWrVuGLX/wifvCDH+Cf//mfEQQBwjCkjuOktdatAEIAASEkVEqF0bYolUqSUqp835fN2JYELbn6HLMMAAuCgIdh+Gdmx3nz5uFLX/pSw5vbsWMHhoaG4m0zOZ6p+L6Pvr6+Kk/IihUrsGLFihmf49lnn8Xv/d7vxavWRox1msvlcOTIEYyPj8eGx0zkd37nd2LQ8vk8XnnlFVx00UV19/3CF76An/zkJzh69CjS6TQsy1oK4DkpZcA596WUoQEuYl1o27Z0HIcZtqGOQVLrkqgay1zXpZxzNj4+fo5Sar3Z6Utf+lI8D6onTzzxBADEi45mvJmp1DqJu7q6Tup4oL7JDyBmfC6Xg9YaO3fuPKnzXnbZZWhvb4+3t2zZ0rQNyYc7DMPucrm8mFLaorVuJYSkGWMO59zWWtsALM45c12XTkxMUM/zyIYNG6Ys4dT1I/m+T0qlErUsi5XLZR4EQXzlzs5ObNq0qemNGQMEQDyBPhlJTqqB9xe0JNOAilV6MmLbNj760Y/G288991zT/a+66ip0dnbG257nnQ+gRSmVJoSkpZRpKWXKgFYul7llWaxUKtFGSSUGNIITTmHa1dVFstksLZVKHEBGShm38lOf+hRSqVTDRh4+fBiDg4MAKkyjlM7IZZWUWssxEQk1Y5kJ0wBUxZzMVJIT/aNHj+LgwYMN902lUvj0pz8db4dhuEhrnQWQMcG2jDHHgEYp5aVSiWezWdrV1UWMlwSJVYMqpuXzeeL7PikUCrRUKjHbtunY2NgGQkjW7HPllVc2vaHXX3+9avvdMM14QhhjyGazyGaz0x9UI+ZBqbfoSQiJQTt8+PBJn/vcc8+t2v7f//3fpvt/5jOfib8rpSzXdZeiAlqKUupIKVOMMQeARQjhtm3TUqnECoVCvAqQPN8U9bhkyRJ4nkfT6TTlnDMhROxanz9/PlauXNm0gb/85S/j75lMJo7FmKmEYYhDhw7FQM9kwbOemDG33qJnErSkJ2SmsnDhwqrJ/XQqdsWKFVUq3vO8JYSQVPRJU0odVKKmrSAIOOecpdNp6nle3eGrynqMsitJR0cHKZfLjFLKpZSrzQ5r166d9ob27NkTfz/11FMxOTl5Ukzbv38/hBCx8VJvojsTaaQejcyZU4mdHRwcxL333lt3n1Qq1dCImjVrFvr7+0EIwQsvvIBt27Zh/vz5WLx4cd39165di4GBAdOmuYQQR2udFkI4lFJHa+0QQixKKfc8jyulWEdHB7EsK8YFkfXIcWI8w+7du4lt2yQIAppOp4nruq0A4oXNNWvWTNdX6O/vj7+bfLGTAc2Abjo9mXN2MtIMNEIIOjs7wRjD6Ogovva1r530+Y2vFABeeeUVXHPNNbjuuuvwzW9+s+7+a9aswc9//nMAgJQyK4TIMsZKlFKHUmoDsKSUNiGEM8aY4zjE8zyqlCKe55HIiVw9ppnxrFQq0XQ6TX3fZ5OTk8tRccEAAM4444ymNyKEqDs/OxlDZM+ePVVraO+WabXrabVCKUVnZ2fTgNVmknQWmHM0m48uX748uUnCMJwXsc3WWjtaa5sQwgkhXAjBfN9n6XSaBkFAfN8nyXGtigJLliyB67qkXC5T27apEKIqTu3UU09teiMTExNVIXHmJk5mTNuzZ0/sqwR+dUwDgO9///tx5LERpZRJZWooxmlsVrVff/11bN68GS0tLQ2POeWUU6q2pZRtqIxjtlKKU0otQgiXUjIpJWtpaaFBENBsNkvnzp0ro+wdABFot912Gx555BGSTqcRhiGRUhKtNSWExLNIxhhs227o6wMq5m+9zpmpepRS4uDBg1Xs4pw3vWYjMU7iZkE8uVwuXsA8WZFSxqANDw/HjoRGbbUsC5zzZHvSqJj4llLKJoRwrTUnhHDOORVCUKUUCcOQpNNpFAoFctttt+G2226rgLZr1y4CVLIohRAkk8kQKSVVSsX+HSklLrzwwpO6sSeffBLlchlBEKBcLk+7/759+zA+Po6Wlpa4Ix955BE88sgjJ3VdoKKyyuUyCoVC02srpWbUtmbiui48z8M//dM/4Xvf+96MjhFCOAC4UoqjQh4eMY4FQcBSqRRljJEgCIjJbo1w0jEFPM8jtm0TIQRhjBEhBCWEzFyvNZGZMq3WCHk/ZLqMnPdDDONOximutY6ZBYBFgFEpJTNMA0CEEMT3/bgkB9DAyx+pRwKgZH5jjOELX/hC04aMj4/HK7kAcNppp2Hv3r0Nkx1q5eDBg7AsC7lcLh5XzjjjDFxyySXTHlsrWmt8+9vfrjtPlFLGY68JdXg35zcPhOu64JzjE5/4RNM4lh/+8IfxMZRSQQihWmumtWaMMQMYCcOQRokfdZ+CKa2VUhJKKVFKEUJIIXmjN9xwQ1PD4OjRo1WgzZ8/H4cOHTIe7mk74uDBg8hms1i8eDHeeOMNABUP/8033zztsfXkX//1XwFMZa5Zxd63bx+6u7tPehUCqPSHAcD3fViWhc997nNYv3593f3L5TL+8z//M95mjAmtNQVACSFUKUUppXF4eVRmoy5wVaBFVQZioZRWZZQfOXJkSvBOUtra2qq2DVtm8iQrpbB//34sX7686sGYzpJrJjUD/xT52c9+hieffBJr1qzBmjVrcO6552LFihUzmqIkpwqu64IQ0tR6rGOkCVNOI8rcIIhCyy3Lqjp/LS5NezOTyew3+hoABgYGmoLW0tKC1tbWuKNLpYp2nQlofX19cF0XK1euRPKa5hzvRkwcSq2YDlm2bBkef/xxvPnmm3jzzTfxwx/+EOl0GqtWrcK5556LNWvWYPHixXWDapOdaizVZoxNOh2ithVRSchABFycbTPd2Ni0N9Pp9NDExMSE1roDAN544w1cdtllTU+4cOFC9Pb2AoCpDDAj0IwRsmrVKuzevTv+/b1Ydibyq1aSoJkxLTk+vfLKK3jllVcAVLSHYeLatWtjH2ISNPOQNWOaUfdAhWWWZZUB6Ci5Q0e/a8wgM7SqN23b1slJXBiGoJTukVKeD1RWpKeT008/PQbN1OuYiboxoJ111llVnvf3yrRm6rGnpwe2beOUU07BE088gZdeegnbtm3Dtm3b0NfXBwA4fvw4tm7diq1btwIAZs+ejdWrV2PFihU477zzMG/ePARBAKC59ya5EsA5LwHQukIrA5QGoAkhOgzDqgfdtm2dDE6aQgHGWBxESQjRjLEdBrS33noLk5OTVSu3tXLmmWfi8ccfB1BJoIjCoxvub2TPnj2wbRtnnHFG3EHAexvTbNuuC7phCeccPT09cRDRlVdeGS89HT58GM899xyef/55PP/88/FqwNjYGJ555pk4+ri7uztWj41AKxQKVUxzHGfC1DTRWiuttWKMKSmlYozpqMqCNnW8as9XtzcTB6pMJvMMIsqGYdh0eR1A1dKNWQSdTrTW2Lt3L5YtWwbHcWIPPFDppHc712rEtKRqM/7Ut956q2qf0047DZ///Odx5513YufOnXjqqafwD//wD7j88surQi36+vpiL0gjz8qWLVtiNgKVYUdrLbXWCoAkhEittaSUqijyWEWEqasq4x5NpVLacRzNOTdV2ZQQQmUymSOEkL1mv+m8E2effXbceOPamU4GBwdRKpViwJMr1UKIKZbXTKWeIVLrIDaO3DfffLPheQghOPPMM3H99dfjnnvuwc0334w5c+Ygm80inU4jnU4jlUo1HAb++79P1HrhnJc45yVCiCSECK210FpLpZRUSinGmBRCKM65MtXxTBU8cw4KACtWrIiLV3LOdblc1mEYKs65iYL9v+aA1157rWkwjOM48cruTD3oyfEMmBoTYtahTlYsy5qycv1uQKuV559/HpZlobW1FR//+MfR29sbp3XVyi9/+Uts37493k6n0yNaa0UIEQAEpVQAEIwxYWIgOecyDENVLpdNSUMNnMCJAhWHsallaFlWFdO01mL27NmPM8biOVtUE6ShXH55HDUOSum04BnQVq+urLcuWLCgSq2+F9CSIefAVNB6enrgOM4U9dhIDh06VLXv5ZdfjnQ63dAve+edd8bfKaWipaXlKCpxjiEAE/cYaK0FAKG1FkmmWZal9+7di2w2q2+77TYANYbI3r170d7ertLptHJdV3HOZZRj5XLOH5FSfh4AnnnmGezcubNh6MEVV1yB22+/HUCl42pTaGvlnXfeAWMsfuodx4mDSQHEgUIA4koHIyMjmJychBAC5XI5NjhaWlpiD4zJ3kyq6eTSkUl/Wrx4MXbt2oWhoSHMmzevYTsB4KGHHoq/M8aa5szt2rULTz31VLydyWSGUKlIF2qtQ0JIACCI2BYCCKMSTzIIApVOp1WhUFC1541Bi1JtdGT2K8dxZBRrLrTWoq2t7ZHx8fHfFkJktdb4xje+gc2bN9edg82ePRsbNmzAj3/849gYaAbavn37sGTJkqrJ6cKFC2PQHn/8cezcuTOOME4O6s1kdHQUQRCYYmhTSlQYIJctW4Zdu3bhrbfeqtIStSKlrBqfLrnkkoaJIFJK/P3f/30yV1u0trYeQ0UlBlrrQGsdEEICQogPIFBKCVQSNSTnXLquq0ypXlNTGaioRw1Ab968OTZGUqmUcl1XSyml1loQQgLG2HHHcX5kDuzt7W0YWwFUan2YrBetdcOqAkNDQ5icnMTKlSvR39+P++67D9dff33VuNnX14cXXngBhw4dQhAEEELEoeDj4+MN22DE5HwnHcXAiVzqnp4eaK2nHde2bNlStTJ/7bXXNtz3vvvuqwpyam1tHSCEuIQQP/p4AHyllK+UCqSUISoMFFJK6bquTqVSyhghiQwanaSJzmazOp1O67GxMdXS0iIppaJUKgnGWEgI8dvb25/VWn+0XC4vB4B/+Zd/wfnnn183Hv7CCy+sKjVRm1xopLe3F2EY4rnnnsOTTz5Z9T+Tt5YskeR5XlwiAqiw5d///d/jOZLneTG7vvvd7+Ltt9+Opx5RXnTV+aWUWLRoEVzXxfe+9z0IIfCJT3wiHl+NCCHwb//2b/H2kiVLcPHFF9cFrLe3F3fccUe8nUqliplMZtgwS2vta619AC7n3Hz3URnjREtLi1BKyUKhoMrlsu7q6qrylEzRbdG4FqtI27aFeQqUUn46nf5PKeW3fd+3fd/HV77yFTz00ENTnMWEEGSz2VgtKqUghKiq7yGEwFtvvYUwDDE2NhYnSESemCoXlqnkwxhDe3s7Fi9ejMHBwThXuZ6afvTRR7F//34wxpBKpaYku5vM0NNOOy1O2vjRj36EH/3oR1i8eDGuvvpqXH311WhtbcWDDz5Y5an58pe/XHcOWiwW8ZWvfCW+DudcZbPZQwB8QohHCHG11i4AV2vtaa1dpZRPCAkBhLZtCyGE9H1fmarmtVJ1p7lcThcKBZXNZkkYhrJcLivOuQAQSil9y7I8QsiA4zg/CMPwz5VS6O/vx1e/+lXceeedU+L7TeanEZP9mSx3ZCzHycnJGJRkFsr8+fPR09OD5cuX44wzzkBPT08cb3Hrrbcin89XqbykmAemXu1iEpVXMmxbvnw53njjDUgpwRjDvn378O1vfxvf/e53cfXVV1c5Fc4888y6lRbCMMRXNecAnwAAEnlJREFUv/rV2AUWPbiHOefFiGU+AC8CzCWEeFprjzHmhWHoSynDVColwjBULS0t0rIsBUAlx7MkaObFAdiwYYMeGBjQmUxGtbW1Cdd1BYCQEBJIKX0AXmtr62uMsSfHx8evAIAXX3wRX/va13DHHXdUqUAhRJVrx6i7pOzbtw+O4yCbzaKnpweLFy+O/y5YsCDu+GQRGSPmSW8EWpLVtfuZY43DePXq1XjnnXfw+c9/PjZKgIrv8+67746PI4Tgb//2b6ewTCmFr3/963j++efj31pbW4cdxxlFBahy9CkBcAGUCSFlpZQnpfQJIYFt27F6PH78eKwao/zrWEXWdb87jqNbWlpUGIYyk8mIcrkcRsaIB8BVSqXS6fRP0+n0giNHjqwAgF/84he49dZb8Y//+I9xZxk1V89qMwnwt956K5YsWYJTTjllivdECBFbivWAMfs3mgcawM2DUg80I2bJKZVK4YEHHsDOnTtx77334oknnqiyVi3LQm9vb1WFAyEE/u7v/i72uQLArFmzJlOpVD8AT2vtEUJcAGWlVJkQUmSMlZVSLmPMBeBJKQMAYSaTEUEQyJaWFiWlbO7GikTncjmdSqV0R0eHSqfTSgghETFNCBEopbyI2kUp5X90dXXtNwc//PDDuPHGG+F5Hg4cOIDJycl4nlSvQznnWL9+PU499dS67q4ka+tZoKbTpgNNCDFlkl1rFNX6IFeuXIlbbrllSu5aEAT41re+hWuvvRZ79+6F53m46aab8LOf/SzeZ968ecVUKnXAGBuU0jIqzCoRQkpKqTKAUjSeeUKIIJqzhUIImU6nVUdHh0qlUnrr1q1TlmuSoMXvVslms7qvr0/7vi+DIJDR/CGklPpKKQ9AWUpZZowVgyD4wW/8xm/EI/Szzz6Lz372s9i4cSOklDHrGGNTyhZNJ2bya6QRaDNRj8ljKaVTmNbV1YVMJhN7433fx5/92Z/FJj7nvKrCwuuvv47PfvazuOqqq/D000/Hv8+bN69ECNmvtfZQUYMlACVCSFFrXQRQYoyVpJRlVJjnUUpjyzEIAun7vuzr6zNv3Jjypo26Lvgk27LZrEylUpJSGlBKA8aYF1k/ZUJIiXN+3Pf9O88+++w9BpQDBw7EoJiSfI7jVBklplbWdNKsglw9L0dSkoZIbWmKWqGUYunSpRgbG8OhQ4fwl3/5l1Xztm984xu4++678Z3vfCfONwvDMF7WAYBFixZNUkr3RqowBkxrXdRaFwAUCSFFQkgJQFlr7TLGPNO3LS0tIpvNSsOyWgMkbmvNtploV7FNKSWUUoJSGgghfKMipZQlpVQRwPFjx47dvXr16hdrPR/JuVptwryUclrvRk2BzCnFXIDpQUtW0jHtqCdGRd5yyy1V7qc/+qM/wu/+7u8CAH7rt36rKqnQtLGnp2fYdd0DkUlfxgmAilrrQsSyglKqKKVMqkafUhoopUQYhlUsqzVA4vtu1FlJtrmuK6WUgjEWZDKZgHPuAihTSouU0gIhpKC1LgwNDT1xzjnnPNzd3a1MJyUnxKYjk0AIIZoyrlZFJkGbKdNq52aN1viWLVsG3/fjyjwAsGnTJnz9618HALz88svYuHEj7r///vj/8+fP1wsXLjxULpf7IyDKAIqMsSRgBQDHCSEFSmmBUloEUOacu5lMJmCMBVJK4brutCwDGoNWxbbZs2fLjo4OobUWYRj6jDFfKeWiMraVlFJFWnn5zvH+/v4dLS0td15wwQUTwInqb57nVVUqTQInpZzChnqdDyB2RxkAzG/1xKynJR+KRj5QrTVOO+20uDQUAFxzzTX45je/if7+ftx444344z/+Yxw6dAhA5YH5zd/8TZ9Sutt13REAbmRoJME6DuA4pfQ4ISRmGSpjmcsY88Mw9LXWoqOjQ8yePbsey6beV732I8qFitBWAwMDJJfLGTVJU6mUL6WkUdweo5RSEwamlMLExIQ+duzYjxljN0RvXwJQAccEdpoaw0lz3PO8uL5HUsz0wIBjXGLTMY0xFrvBADRMJTZq+tRTT0VLSwtc18WXv/xlfPKTn8Rf/dVf4bHHHqti+Omnn67nzp07dPjw4aMAApww68uEkJJRhQBi0AAUtNZFKWU5WjXxGGNBEARCKSV835f5fF53dXWZyfQUA8RIQ6YBFUsyl8vp9vZ2lc/nlai0PCyXy2HkM3MBlIUQJaVUUWtdoJQeV0pNCiFKUkphWVYYVWOLTy6EgOu6UEpVVfIxVmWyKJiRWrYJIaZlWm9vbxUz6wWtJq9HCMHy5ctBKcUrr7yCz3zmM3j44YdjwNrb2/Hbv/3bZULIrsOHDw+gYkyUSOWlRAakSQCTSqlJpdRxSulxrXVBKVUUQsQGCOfc55wbM1/k83nV3t6upnm5XlPQ4vsyatJxHD137lzZ2dkptNYhYyxgjPnRoFuWUhoL6Til9HikBsAY01prwRiTtaWWTJRuLUCGkab4ZXSeKpaY4mdAfdAOHz4cBwiZOsbJikBhGMJ13SkW6bJlyyCEwAsvvBCfd9asWdi4caPf09OzZ/v27bvL5XLBmPCEkOOk8n62ydpPFOx7XGtdkFIWUQGszBjzo/4LOzs7RTabVY7jTKsWjTQLSIzV5NatW9WGDRvQ19dHc7mc7OjoIEIIYlkWlFJECEEIIaYMoK70o3JRKYtHSKXmoQKgOOcUAJWVNww2a1vsVDb5arZtx2Of1joGtfY8+XweN9xwQ8wQKWVco9+cs5EsW7YMQOWhWLx4MS699FL/6NGjA6+++up4FM8REEL8SB36hJCyeXABlJRSxcjgKEbaJx7HtNZlzrkflVsKAYggCGQ+n1e5XE5FBTqbsgw4iXK4yfGtq6tLaK1JVA2HRB2jCSFaKaW01koI4QHQ5nVWZoHTgMcYo5xzqpQitd6KWjEdXc9r4vs+duzYEceVvPrqq/ibv/mbqohes/42k5iVM844A5xznHLKKaK7u3vPU089VQIgUBm7QrMehso8zHjszXysxBgz06Ci1rokpSwbYCOV6FNKQ611vXFsRjIdaDHbNm/ejE2bNqFQKKiBgQGay+UEpRSWZYExpsMw1BFwilKqwjB0AWDLli366aefJhdccIFev349WbdunW5vbycGPADEsiyKihHTFMDa3wkhUErhlltuiZMtkpFbBuRkbcl6YnyhhBB0dXXplpYWffDgQd3W1jZmlky01kG04uwBSC6xGH+i8XSUUHE8FCmlbjQNcC3L8izL8iO2CqVUvXFsWpYBiXzqaYQAwO7du7Fu3Tq4rouxsTFkMhlt5lGUUsM0LYTQWmvf87xQa90qhJjV19eH5557Dj/+8Y+xY8cOlMtlzJo1i2Sz2UqobaVXlVGFjDEynUP41Vdfxe7du5HJZFAul6cEthqGnX/++VU5z8klIBMyWFEQWgHQO3bswMDAAGlvbz9sWdZk5OEoRZ6MIiJLEBULscA5L0TjeAERw7TWJSGEyxirUomEkCrAFi1apE4GsJMBrSlwtm3riYkJtLS0mLcBatu2vblz5745Z86cLS0tLc8opYZQqTKeO3bsGH355ZfxwAMP4Omnn8bExATJZDKYM2eOGa/iDjTFYwyQxvynlOK1115Db28vVq1ahWw2i/b2dixcuBBdXV1YtGgRWltbcfToUVx44YVYtWqVAQiUUvOcqBPPywnp6+vTb775JtLp9LEoesrMvYz/sBBZygUz/4qAKmqtS1HYd8wu27b948ePi9bW1lAIId8LYMDJvYqrVlWqAwcOIJ/Po6urS3d2dmJiYsIwT9m2rVzXlYwxOXv27ANtbW1HKaWPe543Z3h4+MJisbjG9/2zDh486Nx111246667MH/+fKxbtw7r1q3DmjVrzNys6kZ0lFUSXYcQQrB69eoDPT09SmtNisWi8H1f+74vh4aGWqWU3fWsxKSQmldmLVu2DFprXSgU0rlcbjzy1vu6stLsUUrdhA/WRWW5yuOcu1JKPwxDL51OB2EY+p7nCa112NnZKQCIfD6v3wtgwLt/f1oMXD6f19FbnYRSSkdxk0prrWzblkopKaUMlVKCEBJkMhl/4cKFjxFCngrDsC2fz59dKBTOdl131ZEjR7IPPPCAfuCBB9De3o4LL7wQl112GS644ALzjhpiOjiaV2kpJZ566qm+l19+uVTp/xPWysjIyDyt9UJT3LqR5PN5HDx4UO/duxf79+83BdW067rtAMYMaIQQLwLLV0q5EYCe1tpTSvmRH9G3bTvQWgeMMQFAhGEoABiG6TqAnZScLGiGbUDCOAGg8vk8ovq72rIsRQhRhBAVgSY9zxNCiDAKbAkopR7n3FuwYMGzSqmXCSHpfD5/xsTExNpSqbR6cnKyfcuWLXrLli1IpVI499xzyWWXXaYvvfTSeDU8AUSRUjqBaN6pT7x0LgOc8Fe6rhuDc+DAARw4cAB79uyJC3zWdmIQBDmt9VjkSPCitntKKZ9S6kkpA0KIJ4QIKKW+4zghYyww1qGOIqsWLFgg+/r6dC6Xq2d0TLnudDLzzO7Gx055y24ul6OO47BCocAYYyYZ3NJaW77vW5xzB4CllHI457ZSytFam8o1tlLKGR8fXzw6OrqmXC6vklLGwYW2bWPt2rXYsGEDdu/ejZ///Oekp6fnJ+3t7WNIZFMCwMjIyML+/v5Pzp8/H5RSHDlyxMzpGnYSIcS1LOuIZVkDqVSqr6ur6znOeZEQEkRg+WbRkhAShGEYOI4TUkoDnAiBE9lsVkYmvWpvb1fZbFa/H4AlO/7dStW7UWrfCVooFGgqlWKEEF4qlTjnnCEq3OX7vgXAsm3b0pWqNTYAizFmowKwDcAGwMfHx7tHR0fPKZfLq8MwXFhzbSxatOg/Ojo6hnECMAPaaf39/Z+r23BCJOd8mHN+NJVKDaZSqcG2tra+1tbWYSTmZagENQWIFoEJIUEQBCGA0HGcOLxbCCFbWlqE1lp4niez2awaGBjQBrDEmy1Oegyrlff6ItfYOAEqgUHJca67u1tPTEzobDarwjCUUbEuyTkPHcexCCE8CAJOCLF0VGZIVN5oaBl2ArA6OjoKs2bNegfAg8ViMTc8PHxuqVQ6JwiCxQBYZGqP4ERNRPM2pTZU8uxcy7KOOY7T7zjOkXQ6fXTWrFmHLMtySZRmhCiO3kT5aq1jQBAxSAhhwBKIVpqVUiJaJBaZTEYWCgU1d+5c1dfXF0+a3w92JeW9Mq3eearUped5pLu7m0xMTMQq07Zt5nke45wzGr0dKgxDFoYhR1SokkR1okjlDUhMCBF/V0oxSiktlUptw8PDa2fPnt3b0dExblhm1GMQBLbWmqVSqZLx1kR/JaVUKaUUjerja60F51yY78lJsGVZAkBoWZY0vwkhZCqVkkEQSKMKOzo6VF9fn06lUvr9VIfNOvv9kIbgmTfKG/Bc16WWZTHXdZnjODRiHSOV+lA0DENOoopsUkpGKWWU0vi71poyxohSijLG4veW6Zq3BSLKaDUfKaWilCrjvVFKScaYyQ+TlctJA6LknEuttbBtW0auJ5VOp6XruiqbzcooTrRKFTbw1L9nsOp18q/inFXvrKkFr1Qq0ajsLrNtm4VhSG3bppxzFgQBY4yxMAyZrrziikUAUsYYU0rRyGFNlVKUcx4zTJ94d1lVErrJsozcbDAJfFHqrOScS0KItixLSimlbdtSCCGDIFCWZakgCKTjOMp1XdnS0qI6OjqUyTRqAlbt9/e1g99vaQpeUm0GQUCy2Sx1XZf6vk8NgOl0moRhaCq8UsuyqBCCSikpAEQLsSQCcApg8cUj4MIwTKbGKgBgjCnOuYqSKJUQQlqWpVzX1UmgTNqRbdv6gwKrqjN/xTIteL7vk66uLuL7PkkCKIQgUc1DGoYhbW1tRRAEVClFbNumSimS/ACA4zhT7sn3/UoGZZR8bj5BEChKqbZtWxWLRViWpaLv2gA1NjamW1palOM4emBgIE6l/SDAqurEX5PUBQ+oFAhNAmjUZxAEJAxDMnv2bGJAlFKS1tZWIqWMPwawZiWgjDPZVAwwn2KxaDJftQHJsiydZFQtUADwQYBl5NcJWqNrVrEPAGoBBCplDQ0TDZAAEIYhAQAhRNXfpHDOdfKvyUZJAuQ4TpzbXAsUUFlPbBIG8GsBy8gHAVqz609hIHACRKDyogcASIKZ/L2ZOI6jgUpBAKACTvL3WpCAhoyqt/1rkw8atKTUBRA4ASJQDWRy52Q9xEaSLOsAALUAAVUgAR8ioJLyYQItKY3aVfV7EsyTlRpwgMaAfCiASsqHFbR6MpO2NttnJp3/oQOonvw/NRIhF61HRJcAAAAASUVORK5CYII="""

geom_icons = {'ponto': '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',
'linha': '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',
'area': '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',
'nogeom': '/9j/4AAQSkZJRgABAQEAeAB4AAD/4QBYRXhpZgAATU0AKgAAAAgABAExAAIAAAARAAAAPlEQAAEAAAABAQAAAFERAAQAAAABAAADElESAAQAAAABAAADEgAAAAB3d3cuaW5rc2NhcGUub3JnAAD/2wBDAAIBAQIBAQICAgICAgICAwUDAwMDAwYEBAMFBwYHBwcGBwcICQsJCAgKCAcHCg0KCgsMDAwMBwkODw0MDgsMDAz/2wBDAQICAgMDAwYDAwYMCAcIDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAz/wAARCAAWABYDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREAAgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYkNOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOEhYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwD91KKKK/Dz2AooooAKKKKACiiigD//2Q=='
}
