#! python3  # noqa: E265

"""
Main plugin module.
"""

# standard
import tempfile
from pathlib import Path

# PyQGIS
from qgis.core import QgsLayoutExporter, QgsSettings
from qgis.gui import QgisInterface, QgsLayoutDesignerInterface
from qgis.PyQt.QtCore import QCoreApplication, QLocale, QTranslator
from qgis.PyQt.QtGui import QIcon, QImage
from qgis.PyQt.QtWidgets import QAction, QApplication
from qgis.utils import iface

# project
from layout_clipboard.__about__ import (
    DIR_PLUGIN_ROOT,
    __icon_path__,
)

# ############################################################################
# ########## Classes ###############
# ##################################


class LayoutClipboardPlugin:
    def __init__(self):
        """Constructor."""
        self.iface: QgisInterface = iface
        self.actions: dict[int, QAction] = {}

        # translation
        # initialize the locale
        self.locale: str = QgsSettings().value("locale/userLocale", QLocale().name())[0:2]
        locale_path: Path = DIR_PLUGIN_ROOT / "resources" / "i18n" / f"layout_clipboard_{self.locale}.qm"
        if locale_path.exists():
            self.translator = QTranslator()
            self.translator.load(str(locale_path.resolve()))
            QCoreApplication.installTranslator(self.translator)

    def initGui(self):
        """Set up plugin UI elements."""
        self.iface.layoutDesignerOpened.connect(self._on_designer_opened)
        self.iface.layoutDesignerWillBeClosed.connect(self._on_designer_closed)

        for designer in self.iface.openLayoutDesigners():
            self._on_designer_opened(designer)

    def unload(self):
        """Cleans up when plugin is disabled/uninstalled."""
        self.iface.layoutDesignerOpened.disconnect(self._on_designer_opened)
        self.iface.layoutDesignerWillBeClosed.disconnect(self._on_designer_closed)

        for designer in self.iface.openLayoutDesigners():
            self._on_designer_closed(designer)

    def _on_designer_opened(self, designer: QgsLayoutDesignerInterface):
        # Copy action
        copy_action = QAction(self.tr("Copy to Clipboard"), designer.window())
        copy_action.setIcon(QIcon(str(__icon_path__)))
        copy_action.setShortcut("Ctrl+Shift+C")
        copy_action.setToolTip(self.tr("Copy layout to clipboard as PNG (Ctrl+Shift+C)"))
        copy_action.triggered.connect(lambda: self._copy_layout(designer))

        toolbar = designer.layoutToolbar()
        actions = toolbar.actions()

        # Insert at position 14 (or at end if fewer actions)
        if len(actions) >= 14:
            toolbar.insertAction(actions[14], copy_action)
        else:
            toolbar.addAction(copy_action)

        self.actions[id(designer)] = copy_action

    def _on_designer_closed(self, designer: QgsLayoutDesignerInterface):
        designer_id = id(designer)
        if designer_id in self.actions:
            action = self.actions.pop(designer_id)
            designer.layoutToolbar().removeAction(action)
            action.deleteLater()

    def _copy_layout(self, designer: QgsLayoutDesignerInterface):
        layout = designer.layout()

        exporter = QgsLayoutExporter(layout)
        settings = QgsLayoutExporter.ImageExportSettings()
        settings.dpi = 150

        with tempfile.NamedTemporaryFile(suffix=".png", delete=False) as tmp:
            tmp_path = Path(tmp.name)

        result = exporter.exportToImage(str(tmp_path), settings)

        if result != QgsLayoutExporter.Success:
            designer.messageBar().pushWarning(self.tr("Error"), self.tr(f"Export failed: {result}"))
            return

        image = QImage(str(tmp_path))
        QApplication.clipboard().setImage(image)
        tmp_path.unlink(missing_ok=True)

        designer.messageBar().pushSuccess(
            self.tr("Clipboard"), self.tr(f"Layout copied ({image.width()}x{image.height()}px)")
        )

    def tr(self, message: str) -> str:
        """Get the translation for a string using Qt translation API.

        :param message: string to be translated.
        :type message: str

        :returns: Translated version of message.
        :rtype: str
        """
        return QCoreApplication.translate(self.__class__.__name__, message)
