# -*- coding: utf-8 -*-

from qgis.core import (
    QgsLocatorFilter,
    QgsLocatorResult,
    QgsProject,
    QgsMessageLog,
    Qgis
)

class GroupLocatorFilter(QgsLocatorFilter):
    """
    Locator filter that searches for layer groups in the current project.
    The search is case insensitive and will match any part of a group's name.
    """
    def __init__(self, iface):
        self.iface = iface  # Save QGIS interface handle to access the layer tree view
        super().__init__()
    
    def name(self):
        return "GroupLocatorFilter"
        
    def clone(self):
        return GroupLocatorFilter(self.iface)
        
    def displayName(self):
        return "Layer Group Search"
        
    def prefix(self):
        # The search prefix that triggers this filter is "grp"
        return "grp"
        
    def fetchResults(self, search, context, feedback):
        """
        Searches the project's layer groups for names that contain the search string.
        """
        # Ignore empty searches (or enforce a minimum character count)
        if len(search.strip()) < 1:
            return

        search_lower = search.lower()
        root = QgsProject.instance().layerTreeRoot()
        groups = self._getAllGroups(root)
        for group in groups:
            # Case-insensitive match against any part of the group's name
            if search_lower in group.name().lower():
                result = QgsLocatorResult()
                result.filter = self
                result.displayString = group.name()
                # Save the group node for later reference
                result.userData = group
                self.resultFetched.emit(result)
    
    def triggerResult(self, result):
        group = result.userData
        if group is None:
            return

        layer_tree_view = self.iface.layerTreeView()
        if layer_tree_view:
            layer_tree_view.setCurrentNode(group)
            layer_tree_view.setFocus()

    
    def _getAllGroups(self, node):
        """
        Recursively collects all group nodes from the given layer tree node.
        """
        groups = []
        for child in node.children():
            # Check if the child is a group node
            if child.nodeType() == child.NodeGroup:
                groups.append(child)
                groups.extend(self._getAllGroups(child))
        return groups
    
    def info(self, msg=""):
        QgsMessageLog.logMessage("GroupLocatorFilter: " + msg, "GroupLocatorFilter", Qgis.Info)


class GroupLocatorPlugin:
    """
    Main plugin class that registers the GroupLocatorFilter with QGIS.
    """
    def __init__(self, iface):
        self.iface = iface
        self.filter = GroupLocatorFilter(iface)
    
    def initGui(self):
        # Register the locator filter with QGIS on plugin initialization.
        self.iface.registerLocatorFilter(self.filter)
    
    def unload(self):
        # Unregister the locator filter when the plugin is unloaded.
        self.iface.deregisterLocatorFilter(self.filter)
